﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

/*
* @brief
*   0 から (2^N)-1 の値をランダムな順番で保持するリスト
*
* @detail
*   0 から (2^N)-1 の値をランダムな順番で保持するリストです。
*   使用メモリ量を削減するため、2つのリストを組み合わせて擬似的に巨大なリストを作っています。
*   片方のリスト値の奇数ビットのみを保持し、もう片方は値の偶数ビットのみを保持します。
*   2つのリストから要素をそれぞれ取り出し、その OR を取ることで、0 から (2^N)-1 の値を得ることができます。
*/
class RandomList
{
public:
    void Init(uint32_t N) NN_NOEXCEPT;

    uint32_t GetValue(int index) NN_NOEXCEPT
    {
        const int oddIndex = index & ((1 << m_OddListBitWidth) - 1);
        const int evenIndex = index >> m_OddListBitWidth;

        return m_OddList[oddIndex] | m_EvenList[evenIndex];
    }

    size_t GetSize() NN_NOEXCEPT
    {
        return 1 << (m_OddListBitWidth + m_EvenListBitWidth);
    }

private:
    static const uint32_t MaxListBitWidth = 15;
    static const uint32_t MaxListSize = 1 << MaxListBitWidth;

    uint32_t m_OddList[MaxListSize]; // 要素の奇数ビットのみ有効なリスト（偶数ビットは常に0）
    uint32_t m_EvenList[MaxListSize]; // 要素の偶数ビットのみ有効なリスト（奇数ビットは常に0）

    uint32_t m_OddListBitWidth;
    uint32_t m_EvenListBitWidth;

    void Shuffle(uint32_t *buffer, uint32_t size, uint32_t count) NN_NOEXCEPT;
};
