﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Net.Sockets;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.Htcs;

namespace LdnTestBridge
{
    internal class TestBridge
    {

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public TestBridge()
        {
            m_HtcsCommunicator = new HtcsCommunicator();
            m_HtcsPortDescriptor = new HtcsPortDescriptor(HtcsPeerName.Any, "LdnTestBridge");
            m_Groups = new TestGroupRepository();
        }

        /// <summary>
        /// ブリッジを起動します。
        /// </summary>
        /// <param name="port">ポート番号です。</param>
        public void Start(int port)
        {
            // Target マネージャに接続します。
            while (m_HtcsCommunicator.Start() == false)
            {
                Console.Write("Failed to connect to NintendoTargetManager.");
                Thread.Sleep(1000);
            }

            // 指定されたポート番号で待ち受けるサーバを構築します。
            m_Listener = new TcpListener(System.Net.IPAddress.Loopback, port);
            m_ListenerTask = this.ListenAsync(m_Listener);

            // TargetManager にサーバを登録します。
            var mapItem = new PortMapItem(
                m_HtcsPortDescriptor, (System.Net.IPEndPoint)m_Listener.LocalEndpoint);
            while (m_HtcsCommunicator.RegisterPort(mapItem) != 0)
            {
                Console.WriteLine("Failed to register port {0}.", port);
                Thread.Sleep(1000);
            }
            Console.WriteLine("Test Bridge started on TCP port {0}.", port);
        }

        /// <summary>
        /// ブリッジを停止します。
        /// </summary>
        public void Stop()
        {
            // サーバーを停止します。
            m_Listener.Stop();
            m_ListenerTask.Wait();

            // 登録していたポート情報を削除します。
            m_HtcsCommunicator.UnregisterPort(m_HtcsPortDescriptor);
            m_HtcsCommunicator.Stop();
        }

        private async Task ListenAsync(TcpListener listener)
        {
            listener.Start(16);
            await Task.Run(() =>
            {
                while (true)
                {
                    TcpClient client;
                    try
                    {
                        client = m_Listener.AcceptTcpClient();
                    }
                    catch (SocketException)
                    {
                        // Stop() が実行されました。
                        break;
                    }
                    var target = new TestTarget(client, m_Groups);
                    Task.Run(() =>
                    {
                        target.Process();
                    });
                }
            });
        }

        private HtcsCommunicator m_HtcsCommunicator;
        private HtcsPortDescriptor m_HtcsPortDescriptor;
        private TcpListener m_Listener;
        private Task m_ListenerTask;
        private IRepository<TestGroup> m_Groups;
    }
}
