﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "Definition.h"
#include "Util.h"
#include "PacketTransfer.h"
#include "LdnAccessPoint.h"

NN_ALIGNAS(4096) char g_AccessPointMonitorThreadStack[8192];
const int AccessPointMonitorThreadPriority = nn::os::DefaultThreadPriority + 1;

void AccessPointMonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_AccessPointCreated)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 接続中のステーションを取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo);
        if (result <= nn::ldn::ResultInvalidState())
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 接続状態が変化したステーションを検索します。ノード番号 0 は AP なので無視します。
        nn::os::LockMutex(&app.mutex);
        ::std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);

    FinalizeSocket();
}

nn::Result CreateAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // アクセスポイントモードとして起動します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::OpenAccessPoint());

    // Advertise で配信するデータを設定します。
    const char message[] = "Welcome to LDN network!";
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::SetAdvertiseData(message, sizeof(message)));

    nn::ldn::IntentId BasicSampleIntentId =
        nn::ldn::MakeIntentId(LocalCommunicationId, BasicSampleSceneId[pApp->ldnSetting.group]);

    // 新規に構築する LDN ネットワークのパラメータを設定します。
    nn::ldn::NetworkConfig  network = {};
    nn::ldn::SecurityConfig security = {};
    nn::ldn::UserConfig     user = {};
    network.intentId = BasicSampleIntentId;

    if (pApp->ldnSetting.band == LdnBand_5GHz)
    {
        if (pApp->ldnSetting.channel5GHz == LdnChannel5G_36)
        {
            network.channel = 36;
        }
        else if (pApp->ldnSetting.channel5GHz == LdnChannel5G_40)
        {
            network.channel = 40;
        }
        else if (pApp->ldnSetting.channel5GHz == LdnChannel5G_44)
        {
            network.channel = 44;
        }
        else if (pApp->ldnSetting.channel5GHz == LdnChannel5G_48)
        {
            network.channel = 48;
        }
    }
    else
    {
        if (pApp->ldnSetting.channel2GHz == LdnChannel2G_1)
        {
            network.channel = 1;
        }
        else if (pApp->ldnSetting.channel2GHz == LdnChannel2G_6)
        {
            network.channel = 6;
        }
        else if (pApp->ldnSetting.channel2GHz == LdnChannel2G_11)
        {
            network.channel = 11;
        }
    }

    network.nodeCountMax = pApp->ldnSetting.nodeCountMax;

    if (pApp->ldnSetting.secrity == LdnSecurity_StaticAES)
    {
        security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    }
    else
    {
        security.securityMode = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Debug);
    }
    security.passphraseSize = sizeof(Passphrase);
    ::std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    ::std::strncpy(user.userName, "AP", nn::ldn::UserNameBytesMax);

    // アクセスポイントとして新規にネットワークを構築します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::CreateNetwork(network, security, user));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, AccessPointMonitorThread, pApp,
        g_AccessPointMonitorThreadStack, sizeof(g_AccessPointMonitorThreadStack),
        AccessPointMonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void DestroyAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークを破棄し、 LDN ライブラリの使用を終了します。
    nn::ldn::DestroyNetwork();
    nn::ldn::CloseAccessPoint();
}

void LdnAccessPoint(void* arg) NN_NOEXCEPT
{
    nn::Result result;

    // LDN ネットワークを構築します。
    auto& app = *static_cast<ApplicationResource*>(arg);
    result = CreateAccessPoint(&app);
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    app.netState = NetworkState_AP_Created;

    InitializeSocket();

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    nn::os::WaitEvent(&app.cancelEvent);

    // LDN ネットワークを破棄し、データの送受信を終了します。
    //DestroySocket(&app);
    DestroyAccessPoint(&app);
    app.netState = NetworkState_AP_Destroyed;
}
