﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define __STDC_FORMAT_MACROS
#include <cinttypes>
#include <cstdio>
#include <cstring>
#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include "Util.h"
#include "Graphics.h"
#include "DisplayInfo.h"

void WriteInfo(ApplicationResource app)
{
    int x = 1;
    int y = 1;
    int infoX[7] = {35, 52, 55, 72, 75, 92, 95};
    char buffer[128];

    if (app.appState == ApplicationState_Ldn)
    {
        if (app.ldnSetting.mode == LdnMode_AccessPoint)
        {
            WriteText(x, y++, "ACCESS POINT", Color::Green);
            WriteText(x, y++, "(Push Y : Start to Send Packet / Push B : End AP / Push X : Disconnect all Station)", Color::Orange);
        }
        else
        {
            WriteText(x, y++, "STATION", Color::Green);
            WriteText(x, y++, "(Push Y : Start to Send Packet / Push B : End Station)", Color::Orange);
        }
    }
    else if (app.appState == ApplicationState_Communicating)
    {
        if (app.ldnSetting.mode == LdnMode_AccessPoint)
        {
            WriteText(x, y++, "ACCESS POINT (Sending Packet)", Color::Red);
            WriteText(x, y++, "(Push Y : Stop to Send Packet)", Color::Orange);
        }
        else
        {
            WriteText(x, y++, "STATION (Sending Packet)", Color::Red);
            WriteText(x, y++, "(Push Y : Stop to Send Packet)", Color::Orange);
        }
    }

    ::std::sprintf(buffer, " Group : %s", strSettingGroup[app.ldnSetting.group]);
    WriteText(x, y++, buffer);

    ::std::sprintf(buffer, " Channel : %d", app.networkInfo.common.channel);
    WriteText(x, y++, buffer);
    if (app.networkInfo.ldn.securityMode == nn::ldn::SecurityMode_Product)
    {
        WriteText(x, y++, " SecurityMode : StaticAES");
    }
    else
    {
        WriteText(x, y++, " SecurityMode : None");
    }

    ::std::sprintf(buffer, " Node Count : %d(Max : %d)", app.networkInfo.ldn.nodeCount, app.networkInfo.ldn.nodeCountMax);
    WriteText(x, y++, buffer);

    ::std::sprintf(buffer, " My IP Address : %s", ConvertToString(app.ipv4Address));
    WriteText(x, y++, buffer);

    if (app.translateSetting.castTo != PacketCastTo_Ping)
    {
        ::std::sprintf(buffer, "BtiRate");
        WriteText(infoX[0], y, buffer);
        ::std::sprintf(buffer, "|");
        WriteText(infoX[1], y, buffer);
        ::std::sprintf(buffer, "PacketLoss");
        WriteText(infoX[2], y, buffer);
        ::std::sprintf(buffer, "|");
        WriteText(infoX[3], y, buffer);
        ::std::sprintf(buffer, "Recv Packet Count");
        WriteText(infoX[4], y, buffer);
        ::std::sprintf(buffer, "|");
        WriteText(infoX[5], y, buffer);
        ::std::sprintf(buffer, "Recv Data Size");
        WriteText(infoX[6], y++, buffer);

        if (app.ldnSetting.mode == LdnMode_Station)
        {
            ::std::sprintf(buffer, " AP IP Address : %s",
                ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address));
            WriteText(x, y, buffer);
            ::std::sprintf(buffer, "%.2lf KByte/sec", app.receiveInfo[0].packetDataRate);
            WriteText(infoX[0], y, buffer);
            ::std::sprintf(buffer, "|");
            WriteText(infoX[1], y, buffer);
            ::std::sprintf(buffer, "%.2lf%%", app.receiveInfo[0].packetLossRate);
            WriteText(infoX[2], y, buffer);
            ::std::sprintf(buffer, "|");
            WriteText(infoX[3], y, buffer);
            ::std::sprintf(buffer, "%d packet", app.receiveInfo[0].packetCount - app.receiveInfo[0].packetLoss);
            WriteText(infoX[4], y, buffer);
            ::std::sprintf(buffer, "|");
            WriteText(infoX[5], y, buffer);
            ::std::sprintf(buffer, "%.1lf KByte", static_cast<double>(app.receiveInfo[0].receivePacketSize) / 1000);
            WriteText(infoX[6], y++, buffer);
        }
        for (int i = 1; i < nn::ldn::NodeCountMax; i++)
        {
            if (app.ipv4Address != app.networkInfo.ldn.nodes[i].ipv4Address && app.networkInfo.ldn.nodes[i].ipv4Address.raw != 0)
            {
                ::std::sprintf(buffer, " STA[%d] IP Address : %s",
                    i, ConvertToString(app.networkInfo.ldn.nodes[i].ipv4Address));
                WriteText(x, y, buffer);
                ::std::sprintf(buffer, "%.2lf KByte/sec", app.receiveInfo[i].packetDataRate);
                WriteText(infoX[0], y, buffer);
                ::std::sprintf(buffer, "|");
                WriteText(infoX[1], y, buffer);
                ::std::sprintf(buffer, "%.2lf%%", app.receiveInfo[i].packetLossRate);
                WriteText(infoX[2], y, buffer);
                ::std::sprintf(buffer, "|");
                WriteText(infoX[3], y, buffer);
                ::std::sprintf(buffer, "%d packet", app.receiveInfo[i].packetCount - app.receiveInfo[i].packetLoss);
                WriteText(infoX[4], y, buffer);
                ::std::sprintf(buffer, "|");
                WriteText(infoX[5], y, buffer);
                ::std::sprintf(buffer, "%.1lf KByte", static_cast<double>(app.receiveInfo[i].receivePacketSize) / 1000);
                WriteText(infoX[6], y++, buffer);
            }
        }
    }
    else
    {
        ::std::sprintf(buffer, "RTT");
        WriteText(infoX[0], y++, buffer);

        if (app.ldnSetting.mode == LdnMode_Station)
        {
            ::std::sprintf(buffer, " AP IP Address : %s",
                ConvertToString(app.networkInfo.ldn.nodes[0].ipv4Address));
            WriteText(x, y, buffer);
            ::std::sprintf(buffer, "%" PRId64 " [ms]", app.receiveInfo[0].rtt);
            WriteText(infoX[0], y++, buffer);
        }
        for (int i = 1; i < nn::ldn::NodeCountMax; i++)
        {
            if (app.ipv4Address != app.networkInfo.ldn.nodes[i].ipv4Address && app.networkInfo.ldn.nodes[i].ipv4Address.raw != 0)
            {
                ::std::sprintf(buffer, " STA[%d] IP Address : %s",
                    i, ConvertToString(app.networkInfo.ldn.nodes[i].ipv4Address));
                WriteText(x, y, buffer);
                ::std::sprintf(buffer, "%" PRId64 " [ms]", app.receiveInfo[i].rtt);
                WriteText(infoX[0], y++, buffer);
            }
        }
    }

    y = 24;
    if (app.sendWait)
    {
        ::std::sprintf(buffer, "Sending");
        WriteText(x, y, buffer, Color::Red);
    }

    y++;
    ::std::sprintf(buffer, " Send %s", strSettingCastTo[app.translateSetting.castTo - 1]);
    WriteText(x, y++, buffer);
    if (app.receiveError && app.appState == ApplicationState_Communicating)
    {
        ::std::sprintf(buffer, "Recive Error!");
        WriteText(x + 60, y - 1, buffer, Color::Red);
    }

    if (app.packetSwitched)
    {
        ::std::sprintf(buffer, "Recive packets are switched!");
        WriteText(x + 80, y - 1, buffer, Color::Red);
    }

    int sumCounter = 0;
    for (int i = 0; i < app.networkInfo.ldn.nodeCount; ++i)
    {
        sumCounter += app.counter[i];
    }
    ::std::sprintf(buffer, " Send Packet : %d packet", sumCounter);
    WriteText(x, y++, buffer);

    ::std::sprintf(buffer, " Time : %d sec", app.time);
    WriteText(x, y++, buffer);

}// NOLINT(readability/fn_size)

void DisplayInfo(ApplicationResource app) NN_NOEXCEPT
{
    BeginText();

    if (app.appState == ApplicationState_Initialized)
    {
        int x = 1;
        int y = 1;
        char strWrite[BufferSize];
        bool selected;

        for (int i = 1; i < InitCount; ++i)
        {
            if (app.ldnSetting.mode == LdnMode_AccessPoint && i == 1)
            {
                ::std::memcpy(strWrite, strInit[i - 1], BufferSize);
            }
            else
            {
                ::std::memcpy(strWrite, strInit[i], BufferSize);
            }
            selected = i == app.initCursor;
            WriteText(x, y++, strWrite, selected ? Color::Orange : Color::White);
        }
    }
    else if (app.appState == ApplicationState_Setting)
    {
        char strSettings[9][BufferSize];
        int settingX = 1;
        int parameterX = 3;
        int valueX = 32;
        int y = 1;

        ::std::memcpy(strSettings[0], strSettingGroup[(app.ldnSetting.group)], BufferSize);
        ::std::memcpy(strSettings[1], strSettingMode[(app.ldnSetting.mode)], BufferSize);
        ::std::memcpy(strSettings[2], strSettingBand[(app.ldnSetting.band)], BufferSize);

        if (app.ldnSetting.band == LdnBand_5GHz)
        {
            ::std::memcpy(strSettings[3], strSettingChannnel5g[(app.ldnSetting.channel5GHz) - 1], BufferSize);
        }
        else
        {
            ::std::memcpy(strSettings[3], strSettingChannnel2g[(app.ldnSetting.channel2GHz) - 1], BufferSize);
        }
        ::std::memcpy(strSettings[4], strSettingSecurity[(app.ldnSetting.secrity)], BufferSize);
        ::std::sprintf(strSettings[5], "%d", app.ldnSetting.nodeCountMax);
        ::std::memcpy(strSettings[6], strSettingCastTo[(app.translateSetting.castTo) - 1], BufferSize);
        ::std::sprintf(strSettings[7], "%d", app.translateSetting.rate);
        ::std::sprintf(strSettings[8], "%d", app.translateSetting.packetSize);


        WriteText(settingX, y++, strSetting[0]);
        for (int i = 0; i < LdnParameterCount; ++i)
        {
            const bool selected = (i + 1) == app.settingCursor;
            WriteText(parameterX, y, strLdnParameter[i], (selected & !app.focus) ? Color::Orange : Color::White);
            WriteText(valueX, y++, strSettings[i], (selected & app.focus) ? Color::Orange : Color::White);
            if (app.ldnSetting.mode == LdnMode_Station && i == 1)
            {
                y += LdnParameterCount - 2;
                break;
            }
        }
        WriteText(settingX, y++, strSetting[1]);
        for (int i = 0; i < PacketParameterCount; ++i)
        {
            const bool selected = (i + 1 + LdnParameterCount) == app.settingCursor;
            WriteText(parameterX, y, strPacketParameter[i], (selected & !app.focus) ? Color::Orange : Color::White);
            WriteText(valueX, y++, strSettings[i + LdnParameterCount], (selected & app.focus) ? Color::Orange : Color::White);
            if (app.translateSetting.castTo == PacketCastTo_Ping)
            {
                break;
            }
        }
    }
    else if (app.appState == ApplicationState_Ldn)
    {
        int x = 1;
        int y = 1;

        if (app.netState == NetworkState_AP_Created || app.netState == NetworkState_STA_Connected)
        {
            WriteInfo(app);
        }
        else if (app.netState == NetworkState_AP_Creating)
        {
            WriteText(x, y, "Creating", Color::Green);
        }
        else if (app.netState == NetworkState_STA_Connecting)
        {
            WriteText(x, y, "Connecting", Color::Green);
        }
        else if (app.netState == NetworkState_AP_Destroying)
        {
            WriteText(x, y, "Destroying", Color::Green);
        }
        else if (app.netState == NetworkState_STA_Disconnecting)
        {
            WriteText(x, y, "Disconnecting", Color::Green);
        }
    }
    else if (app.appState == ApplicationState_Communicating)
    {
        WriteInfo(app);
    }

    EndText();
}
