﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/ldn.h>
#include <nn/ldn/ldn_Settings.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/ldn/testLdn_CommandLineParser.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Log.h>
#include "Config.h"

namespace
{
    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronization* g_pSync;
    nnt::ldn::ISynchronizationClient* g_pClient;
    nnt::ldn::ISynchronizationServer* g_pServer;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pSync->Synchronize(keyword))

    //! スキャンに関するテスト結果の定義です。
    struct TestResult
    {
        //! スキャンの試行回数です。
        int32_t trial;

        //! スキャン成功回数です。
        int32_t success;

        //! スキャン失敗の回数です。
        int32_t failure;

        //! 連続して失敗した回数です。
        int32_t burst;

        //! 連続して失敗した回数の最大値です。
        int32_t burstMax;
    };

    //! 現在のシーンです。
    enum TestScene
    {
        TestScene_Setup,
        TestScene_CreatingNetwork,
        TestScene_NetworkCreated,
        TestScene_Scanning,
        TestScene_Cleanup,
        TestScene_Finalized,
    };

    //! テストの状態です。
    struct TestStatus
    {
        int32_t scene;
        int32_t socket;
        int32_t sendCounter;
        int32_t elapsed;
        TestResult result;
        nn::ldn::Ipv4Address address;
        nn::ldn::SubnetMask mask;
    };

    //! データ通信で送受信するデータです。
    struct Data
    {
        int32_t counter;
        int32_t hash;
    };

    // テストの設定値です。
    nnt::ldn::TestConfig g_TestConfig;

    // 状態変化を監視するイベントです。
    nn::os::SystemEvent g_StateChangeEvent;

    // socket のバッファです。
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // スキャンバッファです。
    nn::ldn::NetworkInfo g_ScanBuffer[nn::ldn::ScanResultCountMax];
    nn::ldn::NetworkInfo g_Network;

    // 送信スレッドです。
    nn::os::ThreadType g_SendThread;
    NN_ALIGNAS(4096) char g_SendThreadStack[16 * 1024];
    const int SendThreadPriority = nn::os::DefaultThreadPriority;

    void SetLdnSettings() NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::SetOperationMode(
            static_cast<nn::ldn::OperationMode>(g_TestConfig.operationMode)));
    }

    void PrintTestResult(const TestResult& result) NN_NOEXCEPT
    {
        NNT_LDN_LOG_INFO("Scan.Success : %d/%d (%.2f%%)\n",
            result.success, result.trial, result.success * 100.0f / result.trial);
        NNT_LDN_LOG_INFO("Scan.Burst   : %d\n", result.burstMax);
    }

    void SendThread(void* arg) NN_NOEXCEPT
    {
        auto& status = *static_cast<TestStatus*>(arg);

        // ブロードキャスト送信用のアドレスを生成します。
        NNT_LDN_LOG_DEBUG("send thread started\n");
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(PortNumber);
        addr.sin_addr.S_addr = nn::socket::InetHtonl(
            nn::ldn::MakeBroadcastAddress(status.address, status.mask).raw);

        // 一定時間毎にパケットの送信を繰り返します。
        auto interval = nn::TimeSpan::FromMicroSeconds(1000000 / g_TestConfig.packetRate);
        for (;;)
        {
            // 開始時刻を取得します。
            nn::os::Tick startedAt = nn::os::GetSystemTick();

            // 送信データを生成します。
            Data data;
            data.counter = nn::socket::InetHtonl(status.sendCounter);
            data.hash = data.counter ^ INT32_C(0x5C5C5C5C);
            ++status.sendCounter;

            // カウンタの値を送信します。
            auto ret = nn::socket::SendTo(
                status.socket, &data, sizeof(data), nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr));
            if (ret < 0)
            {
                break;
            }

            // パケットレート調整のためスリープします。
            auto elapsed = (nn::os::GetSystemTick() - startedAt).ToTimeSpan();
            auto diff = interval - elapsed;
            if (0 < diff)
            {
                nn::os::SleepThread(diff);
            }
        }
        NNT_LDN_LOG_DEBUG("send thread finished\n");
    }

    void StartCommunication(TestStatus* pStatus) NN_NOEXCEPT
    {
        // UDP 通信に使用するソケットを生成します。
        pStatus->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
        ASSERT_NE(nn::socket::InvalidSocket, pStatus->socket);

        // ソケットを Bind します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(PortNumber);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;
        ASSERT_EQ(0, nn::socket::Bind(
            pStatus->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)));

        // ブロードキャスト通信を有効化します。
        int isEnabled = 1;
        nn::socket::SetSockOpt(
            pStatus->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

        // データ送信を開始します。
        NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
            &g_SendThread, SendThread, pStatus,
            g_SendThreadStack, sizeof(g_SendThreadStack), SendThreadPriority));
        nn::os::StartThread(&g_SendThread);
    }

    void StopCommunication(TestStatus* pStatus) NN_NOEXCEPT
    {
        // データの送信を終了します。
        nn::socket::Shutdown(pStatus->socket, nn::socket::ShutdownMethod::Shut_RdWr);
        nn::socket::Close(pStatus->socket);
        pStatus->socket = static_cast<int32_t>(nn::socket::InvalidSocket);

        // スレッドを停止します。
        nn::os::WaitThread(&g_SendThread);
        nn::os::DestroyThread(&g_SendThread);
    }

    void UpdateSetup(TestStatus* pStatus) NN_NOEXCEPT
    {
        // LDN ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("initializing the ldn library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::InitializeSystem());
        SetLdnSettings();

        // socket ライブラリを初期化します。
        NNT_LDN_LOG_DEBUG("initializing the socket library...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        // 状態変化を通知するイベントを取得します。
        nn::ldn::AttachStateChangeEvent(g_StateChangeEvent.GetBase());

        // アクセスポイントとステーションの分岐です。
        if (g_TestConfig.nodeIndex == 0)
        {
            NNT_LDN_LOG_DEBUG("starting accesspoint mode...\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenAccessPoint());
            pStatus->scene = TestScene_CreatingNetwork;
        }
        else
        {
            NNT_LDN_LOG_DEBUG("starting station mode...\n");
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::OpenStation());
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            pStatus->scene = TestScene_Scanning;
        }
    }

    void UpdateCreatingNetwork(TestStatus* pStatus) NN_NOEXCEPT
    {
        // ネットワークのパラメータを適当に設定します。
        nn::ldn::NetworkConfig network = { };
        network.intentId = nn::ldn::MakeIntentId(LocalCommunicationId, g_TestConfig.sceneId);
        network.nodeCountMax = static_cast<int8_t>(g_TestConfig.nodeCount);
        network.channel = g_TestConfig.channel;
        nn::ldn::SecurityConfig security = { };
        security.securityMode = static_cast<nn::Bit16>(g_TestConfig.isEncrypted ?
            nn::ldn::SecurityMode_Product : nn::ldn::SecurityMode_Debug);
        security.passphraseSize = sizeof(Passphrase);
        std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
        nn::ldn::UserConfig user = { };
        std::strncpy(user.userName, ApUserName, nn::ldn::UserNameBytesMax);

        // ネットワークを構築します。
        NNT_LDN_LOG_DEBUG("creating a network...\n");
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::CreateNetwork(network, security, user));
        ASSERT_EQ(nn::ldn::State_AccessPointCreated, nn::ldn::GetState());

        // ネットワークの情報を取得します。
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetNetworkInfo(&g_Network));
        NNT_ASSERT_RESULT_SUCCESS(nn::ldn::GetIpv4Address(&pStatus->address, &pStatus->mask));
        NNT_LDN_LOG_INFO("Channel      : %d\n", g_Network.common.channel);
        NNT_LDN_LOG_INFO("SSID         : %s\n", g_Network.common.ssid.raw);

        // データフレームの送信を開始します。
        StartCommunication(pStatus);

        // 次のシーンへの遷移です。
        pStatus->scene = TestScene_NetworkCreated;
    }

    void UpdateNetworkCreated(TestStatus* pStatus) NN_NOEXCEPT
    {
        // 何もしません。
    }

    void UpdateScanning(TestStatus* pStatus) NN_NOEXCEPT
    {
        // スキャンを開始します。
        NNT_LDN_LOG_DEBUG("scanning...\n");

        // IntentID が一致するネットワークをスキャン対象とします。
        int count;
        nn::ldn::ScanFilter filter = { };
        filter.networkId.intentId = nn::ldn::MakeIntentId(LocalCommunicationId, g_TestConfig.sceneId);
        filter.flag = static_cast<nn::Bit32>(nn::ldn::ScanFilterFlag_IntentId);

        // ネットワークをスキャンします。
        const int retryCountMax = 1;
        bool isFound = false;
        for (int i = 0; i < retryCountMax && !isFound; ++i)
        {
            NNT_ASSERT_RESULT_SUCCESS(nn::ldn::Scan(
                g_ScanBuffer, &count, nn::ldn::ScanResultCountMax,
                filter, g_TestConfig.channel));
            isFound |= (0 < count);
        }

        // スキャン結果を保存します。
        ++pStatus->result.trial;
        if (isFound)
        {
            ++pStatus->result.success;
            pStatus->result.burst = 0;
        }
        else
        {
            ++pStatus->result.failure;
            ++pStatus->result.burst;
            pStatus->result.burstMax = std::max(pStatus->result.burst, pStatus->result.burstMax);
        }

        // 現時点のテスト結果を出力します。
        PrintTestResult(pStatus->result);
        NNT_LDN_LOG_INFO_WITHOUT_PREFIX("\n");
    }

    void UpdateCleanup(TestStatus* pStatus) NN_NOEXCEPT
    {
        // ステーションを終了します。
        NNT_LDN_LOG_DEBUG("cleaning up...\n");
        nn::ldn::FinalizeSystem();

        // データ通信を終了します。
        if (pStatus->socket != nn::socket::InvalidSocket)
        {
            StopCommunication(pStatus);
        }

        // 次のテストに移ります。
        pStatus->scene = TestScene_Finalized;
    }

} // namespace <unnamed>

//
// スキャンのエイジングです。
//
TEST(Aging, Scan)
{
    // テストの設定を表示します。
    NNT_LDN_LOG_INFO("Scene Id     : %d\n", g_TestConfig.sceneId);
    NNT_LDN_LOG_INFO("Node Type    : %s\n", g_TestConfig.nodeIndex == 0 ? "AP" : "STA");
    NNT_LDN_LOG_INFO("Node Index   : %d/%d\n", g_TestConfig.nodeIndex, g_TestConfig.nodeCount);
    NNT_LDN_LOG_INFO("Encryption   : %s\n", g_TestConfig.isEncrypted ? "enabled" : "disabled");
    NNT_LDN_LOG_INFO("Packet Rate  : %d packets/sec\n", g_TestConfig.packetRate);
    NNT_LDN_LOG_INFO("Channel      : %d\n", g_TestConfig.channel);
    NNT_LDN_LOG_INFO_WITHOUT_PREFIX("\n");

    // テストの状態を初期化しておきます。
    TestStatus status;
    std::memset(&status, 0, sizeof(status));
    status.socket = static_cast<int32_t>(nn::socket::InvalidSocket);

    // 他の開発機と同期します。
    NNT_LDN_LOG_DEBUG("synchronizing...\n");
    NNT_LDN_SYNC("Aging.Start");

    // テスト開始時点の時刻を取得しておきます。
    nn::os::Tick startedAt = nn::os::GetSystemTick();

    // ネットワークの構築と破棄を繰り返します。
    while (status.scene != TestScene_Finalized)
    {
        // テストが始まってからの経過時間を取得します。
        auto now = nn::os::GetSystemTick();
        status.elapsed = static_cast<int>((now - startedAt).ToTimeSpan().GetSeconds());
        if (g_TestConfig.duration <= status.elapsed)
        {
            status.scene = TestScene_Cleanup;
        }

        // シーンに応じた処理です。
        switch (status.scene)
        {
        case TestScene_Setup:
            UpdateSetup(&status);
            break;
        case TestScene_CreatingNetwork:
            UpdateCreatingNetwork(&status);
            break;
        case TestScene_NetworkCreated:
            UpdateNetworkCreated(&status);
            break;
        case TestScene_Scanning:
            UpdateScanning(&status);
            break;
        case TestScene_Cleanup:
            UpdateCleanup(&status);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        // 60FPS 程度になるようにスリープします。
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // スキャン成功率が閾値を上回っていれば成功とみなします。
    if (0 < status.result.trial)
    {
        float successRate = status.result.success * 100.0f / status.result.trial;
        ASSERT_GE(successRate, g_TestConfig.threshold);
    }
}

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // コマンドライン引数に関する設定です。
    nnt::ldn::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::ldn::CommandLineOptionFlag_NodeCount |
        nnt::ldn::CommandLineOptionFlag_NodeIndex |
        nnt::ldn::CommandLineOptionFlag_Channel |
        nnt::ldn::CommandLineOptionFlag_Encryption |
        nnt::ldn::CommandLineOptionFlag_Duration |
        nnt::ldn::CommandLineOptionFlag_PacketRate |
        nnt::ldn::CommandLineOptionFlag_Threshold |
        nnt::ldn::CommandLineOptionFlag_SceneId |
        nnt::ldn::CommandLineOptionFlag_OperationMode);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax + 1;
    setting.defaultSceneId = SceneId;

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::ldn::Parse(&g_TestConfig, setting, argc, argv);

    // 他の開発機と同期する準備です。
    int result;
    if (g_TestConfig.nodeIndex == 0)
    {
        g_pServer = new nnt::ldn::HtcsSynchronizationServer(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pServer->CreateServer("nn::ldn::AgingScan", g_TestConfig.nodeCount - 1);
        g_pSync = g_pServer;
    }
    else
    {
        g_pClient = new nnt::ldn::HtcsSynchronizationClient(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        result = g_pClient->Connect("nn::ldn::AgingScan");
        g_pSync = g_pClient;
    }

    // テストを実行します。
    int exitCode;
    if (result == nnt::ldn::SynchronizationResult_Success)
    {
        exitCode = RUN_ALL_TESTS();
    }
    else
    {
        NNT_LDN_LOG_ERROR("failed to sync: %d\n", result);
        exitCode = result;
    }
    delete g_pSync;
    nnt::Exit(exitCode);
}
