﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/ns/ns_SystemUpdateApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/socket.h>
#include <nnt.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_NifmUtility.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt/result/testResult_Assert.h>
#include "Params.h"
#include "Precondition.h"

namespace
{
    // テストの設定値です。
    struct TestConfig : public nnt::lcs::TestConfig
    {
        int testCount;
    } g_TestConfig;

    // LCS/Socket の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // 同期に使用するオブジェクトとバッファです。
    nnt::ldn::ISynchronization* g_pSync;
    char g_SynchronizationBuffer[16 * 1024];

    // 同期用のマクロです。
    #define NNT_LDN_SYNC(keyword)\
        ASSERT_EQ(::nnt::ldn::SynchronizationResult_Success, g_pSync->Synchronize(keyword))

    // テストを実施する条件を指定するためのマクロです。
    #define NNT_LDN_REQUIRES_NODE_COUNT(count) \
        if (g_TestConfig.nodeCount < (count))\
        {\
            return;\
        }

    /**
     * @brief       テストを繰り返すごとに待ち時間を変化させるためのヘルパーです。
     */
    class FlexSleeper
    {
    public:
        /**
         * @brief       コンストラクタです。
         * @param[in]   delayMin            待ち時間の最小値です。
         * @param[in]   delayMax            待ち時間の最大値です。
         * @param[in]   repeatCount         テストを繰り返す回数です。
         * @details
         *   FlexSleep() を呼び出した側と FixedSleep() を呼び出した側のスリープ時間差を指定範囲内で呼び出す度に変化させます。delayMin が 0 より大きい場合は FlexSleep() を呼び出す側と FixedSleep() を呼び出す側の両方で最低 delayMin だけのスリープが実行されます。delayMin が 0 より小さい場合は、FixedSleep() を呼び出す側に -delayMin のスリープが、FlexSleep() を呼び出す側は 0 ～ delayMax のスリープが実行されます。
         */
        FlexSleeper(nn::TimeSpan delayMin, nn::TimeSpan delayMax, int repeatCount)
        {
            NN_ASSERT(delayMin < delayMax);
            NN_ASSERT(delayMax >= nn::TimeSpan());
            NN_ASSERT(repeatCount > 0);

            m_TimeoutMax = delayMax - delayMin;
            m_ConstantFlexTimeout = delayMin < nn::TimeSpan() ? nn::TimeSpan() : delayMin;
            m_ConstantFixedTimeout = delayMin < nn::TimeSpan() ? nn::TimeSpan() - delayMin : delayMin;
            m_CountMax = repeatCount - 1;
            m_Count = 0;
        }

        /**
         * @brief        呼び出し回数に対応する時間だけスリープします。
         */
        //
        void FlexSleep()
        {
            auto flexTimeout = nn::TimeSpan::FromNanoSeconds(m_TimeoutMax.GetNanoSeconds() * m_Count / m_CountMax);
            nn::os::SleepThread(m_ConstantFlexTimeout + flexTimeout);

            ++m_Count;
        }

        /**
         * @brief       固定時間スリープします。
         */
        void FixedSleep()
        {
            nn::os::SleepThread(m_ConstantFixedTimeout);
        }

    private:
        nn::TimeSpan m_TimeoutMax;
        nn::TimeSpan m_ConstantFlexTimeout;
        nn::TimeSpan m_ConstantFixedTimeout;
        int m_CountMax;
        int m_Count;
    };

} // namespace <unnamed>

//
// 複数のクライアントが同時に接続・切断をしたときに GetNodes() が正しく機能することを検証します。
//
TEST(BeforeStart, SimultaneousJoinAndLeave)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    if (g_TestConfig.nodeIndex == 0)
    {
        // ホストとしてセッションを構築します。
        const auto id = nnt::lcs::Application::Timing01.value;
        const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
        nnt::lcs::HostStarter starter(settings);
        ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        nn::os::ClearSystemEvent(&stateChangeEvent);
        NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.SessionOpened");
        NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.ScanCompleted");

        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            nn::os::Tick beginTick;

            int actualCount = 0;
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];

            //  全クライアントを同時に接続させて、GetNodes() の動作を確認します。
            beginTick = nn::os::GetSystemTick();
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                auto timeout = JoinTimeout - (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, timeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.Joined");

            // 全クライアントを同時に離脱させて、GetNodes() の動作を確認します。
            beginTick = nn::os::GetSystemTick();
            while (actualCount > 1)
            {
                // クライアントの離脱を待機します。
                auto timeout = JoinTimeout - (nn::os::GetSystemTick() - beginTick).ToTimeSpan();
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, timeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // FIXME: すぐに離脱が反映されないことがある不具合のワークアラウンド
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));

                // 現在接続中のクライアントを確認します。
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
                ASSERT_GE(g_TestConfig.nodeCount - 1, actualCount);
                ASSERT_LE(1, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::SleepThread(StateChangeTimeout);
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.Disconnected");
        }
    }
    else
    {
        // セッションの構築を待機します。
        NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.SessionOpened");

        // 接続先のセッションをスキャンします。
        nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
        int count;
        int index;
        NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
            sessions, &count, &index, nn::lcs::ScanResultCountMax,
            nnt::lcs::DefaultHostUserName, ScanTimeout));
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
        NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.ScanCompleted");

        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            int actualCount;

            // 全クライアント同時にセッションに参加します。
            {
                // セッションに参加します。
                const auto& session = sessions[index];
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(session, JoinTimeout));
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());

                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_NE(0, nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
                bool isMyIndexFound = false;
                for (int i = 1; i < actualCount; ++i)
                {
                    if (nodes[i].index == nn::lcs::GetMyIndex())
                    {
                        isMyIndexFound = true;
                        break;
                    }
                }
                ASSERT_TRUE(isMyIndexFound);
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.Joined");

            // 全クライアント同時にセッションから離脱します。
            {
                // 現在接続中のクライアントを確認します。
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount;
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                bool isMyIndexFound = false;
                for (int i = 1; i < actualCount; ++i)
                {
                    if (nodes[i].index == nn::lcs::GetMyIndex())
                    {
                        isMyIndexFound = true;
                        break;
                    }
                }
                ASSERT_TRUE(isMyIndexFound);

                // セッションから離脱します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("BeforeStart.SimultaneousJoinAndLeave.Disconnected");
        }
    }
} // NOLINT(impl/function_size)

//
// 複数のクライアントが StartContentShare() と同時に LeaveSession() によって
// 離脱をしたときに意図しない状態にならないことを検証します。
//
TEST(Starting, SimultaneousLeave)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Starting.SimultaneousLeave.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-200), nn::TimeSpan::FromMilliSeconds(200), RepeatCount);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    if (g_TestConfig.nodeIndex == 0)
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            nnt::lcs::HostStarter starter(settings);
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousLeave.SessionOpened");

            int actualCount = 0;
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];

            // クライアントの接続を待機します
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousLeave.Joined");

            // 待ちます。
            sleep.FlexSleep();

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Starting.SimultaneousLeave.Disconnected");

            // 全てのクライアントが抜け終わるまで待機します。
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_Completed)
            {
                ASSERT_EQ(state, nn::lcs::State_Transferring);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, LeaveReactionTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
            ASSERT_EQ(1, actualCount);
            ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
        }
    }
    else
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            // セッションの構築を待機します。
            NNT_LDN_SYNC("Starting.SimultaneousLeave.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousLeave.Joined");

            // 待ちます。
            sleep.FixedSleep();

            // セッションから離脱します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());

            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousLeave.Disconnected");
        }
    }
}

//
// 複数のクライアントが StartContentShare と同時に lcs::Finalize() によって
// 離脱をしたときに意図しない状態にならないことを検証します。
//
TEST(Starting, SimultaneousFinalize)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Starting.SimultaneousFinalize.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-200), nn::TimeSpan::FromMilliSeconds(200), RepeatCount);

    if (g_TestConfig.nodeIndex == 0)
    {
        nn::lcs::Config config = nnt::lcs::CreateDefaultHostConfig();
        nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

        // イベントを取得します。
        nn::os::SystemEventType stateChangeEvent;
        nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
        ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            nnt::lcs::HostStarter starter(settings);
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousFinalize.SessionOpened");

            int actualCount = 0;
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];

            // クライアントの接続を待機します
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousFinalize.Joined");

            // 待ちます。
            sleep.FlexSleep();

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Starting.SimultaneousFinalize.Disconnected");

            // 全てのクライアントが抜け終わるまで待機します。
            nn::lcs::State state;
            while ((state = nn::lcs::GetState()) != nn::lcs::State_Completed)
            {
                ASSERT_EQ(state, nn::lcs::State_Transferring);
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, LeaveReactionTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
            ASSERT_EQ(1, actualCount);
            ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);

            NNT_LDN_SYNC("Starting.SimultaneousFinalize.Validated");
        }
    }
    else
    {
        nn::lcs::Config config = nnt::lcs::CreateDefaultClientConfig();
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            {
                nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

                // イベントを取得します。
                nn::os::SystemEventType stateChangeEvent;
                nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // セッションの構築を待機します。
                NNT_LDN_SYNC("Starting.SimultaneousFinalize.SessionOpened");

                // 自分の順番が回ってくるまでスキャンをして待ちます。
                nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
                int count;
                int index;
                do
                {
                    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

                // セッションに参加します。
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("Starting.SimultaneousFinalize.Joined");

                // 待ちます
                sleep.FixedSleep();

                // lcs::Finalize() によって離脱します。
            }

            NNT_LDN_SYNC("Starting.SimultaneousFinalize.Disconnected");
            NNT_LDN_SYNC("Starting.SimultaneousFinalize.Validated");
        }
    }
}

//
// 複数のクライアントが StartContentShare と同時に参加をしようとしたときに
// 意図しない状態にならないことを検証します。
//
TEST(Starting, SimultaneousJoin)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("Starting.SimultaneousJoin.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-200), nn::TimeSpan::FromMilliSeconds(200), RepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    if (g_TestConfig.nodeIndex == 0)
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            nnt::lcs::HostStarter starter(settings);
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousJoin.SessionOpened");
            NNT_LDN_SYNC("Starting.SimultaneousJoin.ScanCompleted");

            // 待ちます。
            sleep.FlexSleep();

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // 配信開始時点で参加しているクライアントのノード情報を検証します。
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));
            ASSERT_GE(g_TestConfig.nodeCount, actualCount);
            ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
            for (int nodeIndex = 1; nodeIndex < actualCount; ++nodeIndex)
            {
                ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[nodeIndex].userName);
            }

            NNT_LDN_SYNC("Starting.SimultaneousJoin.Joined");
            NNT_LDN_SYNC("Starting.SimultaneousJoin.Leaved");
        }
    }
    else
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            // セッションの構築を待機します。
            NNT_LDN_SYNC("Starting.SimultaneousJoin.SessionOpened");

            // 接続先のセッションをスキャンします。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
                sessions, &count, &index, nn::lcs::ScanResultCountMax,
                nnt::lcs::DefaultHostUserName, ScanTimeout));
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            NNT_LDN_SYNC("Starting.SimultaneousJoin.ScanCompleted");

            // 待ちます。
            sleep.FixedSleep();

            // セッションに参加します。
            auto joinResult = nnt::lcs::TimedJoin(sessions[index], JoinTimeout);
            ASSERT_TRUE(joinResult.IsSuccess() || nn::lcs::ResultCommunicationError::Includes(joinResult));
            if (joinResult.IsSuccess())
            {
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            }
            else
            {
                ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
            }

            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Starting.SimultaneousJoin.Joined");

            if (joinResult.IsSuccess())
            {
                // セッションから離脱します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }

            NNT_LDN_SYNC("Starting.SimultaneousJoin.Leaved");
        }
    }
}

//
// コンテンツ配信を開始してから全台が LeaveSession() によって離脱した時に
// 意図しない状態にならないことを検証します。
//
TEST(AfterStart, SimultaneousLeaveAll)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    if (g_TestConfig.nodeIndex == 0)
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            {
                // ホストとしてセッションを構築します。
                const auto id = nnt::lcs::Application::Timing01.value;
                const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
                nnt::lcs::HostStarter starter(settings);
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.SessionOpened");

                // クライアントの接続を待機します
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount = 0;
                while (actualCount < g_TestConfig.nodeCount)
                {
                    // クライアントの接続を待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                    nn::os::ClearSystemEvent(&stateChangeEvent);

                    // 現在接続中のクライアントを確認します。
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                    ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                    ASSERT_LE(2, actualCount);
                    ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                    for (int i = 1; i < actualCount; ++i)
                    {
                        ASSERT_NE(0, nodes[i].index);
                        ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                    }
                }
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Joined");

                // コンテンツ配信を開始します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Started");

                nn::os::SleepThread(StateChangeTimeout);

                // セッションを破棄します。
            }
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Leaved");
        }
    }
    else
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            // セッションの構築を待機します。
            NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Joined");

            NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Started");

            nn::os::SleepThread(StateChangeTimeout);

            // セッションから離脱します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("AfterStart.SimultaneousLeaveAll.Leaved");
        }
    }
}

//
// コンテンツ配信を開始してから全台が Finalize() によって離脱した時に
// 意図しない状態にならないことを検証します。
//
TEST(AfterStart, SimultaneousFinalizeAll)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    if (g_TestConfig.nodeIndex == 0)
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            {
                nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

                // イベントを取得します。
                nn::os::SystemEventType stateChangeEvent;
                nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // ホストとしてセッションを構築します。
                const auto id = nnt::lcs::Application::Timing01.value;
                const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.SessionOpened");

                // クライアントの接続を待機します
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount = 0;
                while (actualCount < g_TestConfig.nodeCount)
                {
                    // クライアントの接続を待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                    nn::os::ClearSystemEvent(&stateChangeEvent);

                    // 現在接続中のクライアントを確認します。
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                    ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                    ASSERT_LE(2, actualCount);
                    ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                    for (int i = 1; i < actualCount; ++i)
                    {
                        ASSERT_NE(0, nodes[i].index);
                        ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                    }
                }
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Joined");

                // コンテンツ配信を開始します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Started");

                nn::os::SleepThread(StateChangeTimeout);

                // セッションを破棄します。
            }
            NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Leaved");
        }
    }
    else
    {
        for (int testCount = 0; testCount < RepeatCount; ++testCount)
        {
            {
                nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

                // イベントを取得します。
                nn::os::SystemEventType stateChangeEvent;
                nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // セッションの構築を待機します。
                NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.SessionOpened");

                // 自分の順番が回ってくるまでスキャンをして待ちます。
                nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
                int count;
                int index;
                do
                {
                    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

                // セッションに参加します。
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Joined");

                NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Started");

                nn::os::SleepThread(StateChangeTimeout);

                // セッションから離脱します。
            }
            NNT_LDN_SYNC("AfterStart.SimultaneousFinalizeAll.Leaved");
        }
    }
}

//
// コンテンツ配信を開始してから親交代をしているタイミングで、ホストが LeaveSession()
// によってセッションを破棄したとき、意図しない状態にならないことを検証します。
//
TEST(AfterStart, SimultaneousHostMigrationLeaveHost)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-400), nn::TimeSpan::FromMilliSeconds(400), RepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    for (int testCount = 0; testCount < RepeatCount; ++testCount)
    {
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            if (g_TestConfig.nodeIndex == 0)
            {
                // ホストとしてセッションを構築します。
                const auto id = nnt::lcs::Application::Timing02.value;
                const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.SessionOpened");

                // クライアントの接続を待機します
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount = 0;
                while (actualCount < g_TestConfig.nodeCount)
                {
                    // クライアントの接続を待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                    nn::os::ClearSystemEvent(&stateChangeEvent);

                    // 現在接続中のクライアントを確認します。
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                    ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                    ASSERT_LE(2, actualCount);
                    ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                    for (int i = 1; i < actualCount; ++i)
                    {
                        ASSERT_NE(0, nodes[i].index);
                        ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                    }
                }
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Joined");

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.ReadyToStart");

                // コンテンツ配信を開始します。
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Started");
            }
            else
            {
                // セッションの構築を待機します。
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.SessionOpened");

                // 自分の順番が回ってくるまでスキャンをして待ちます。
                nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
                int count;
                int index;
                do
                {
                    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

                // セッションに参加します。
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Joined");

                nn::os::SleepThread(StateChangeTimeout);
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.ReadyToStart");

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Started");

                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }

            if (g_TestConfig.nodeIndex == 0)
            {
                sleep.FixedSleep();

                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Leaved");
            }
            else
            {
                sleep.FlexSleep();

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Leaved");

                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ShareCompleteTimeout));
                ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
                ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
        }

        NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveHost.Finalized");
    }
}

//
// コンテンツ配信を開始してから親交代をしているタイミングで、クライアントが
// LeaveSession() によって離脱したとき、意図しない状態にならないことを検証します。
//
TEST(AfterStart, SimultaneousHostMigrationLeaveClients)
{
    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-400), nn::TimeSpan::FromMilliSeconds(400), RepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    for (int testCount = 0; testCount < RepeatCount; ++testCount)
    {
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            if (g_TestConfig.nodeIndex == 0)
            {
                // ホストとしてセッションを構築します。
                const auto id = nnt::lcs::Application::Timing02.value;
                const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.SessionOpened");

                // クライアントの接続を待機します
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount = 0;
                while (actualCount < g_TestConfig.nodeCount)
                {
                    // クライアントの接続を待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                    nn::os::ClearSystemEvent(&stateChangeEvent);

                    // 現在接続中のクライアントを確認します。
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                    ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                    ASSERT_LE(2, actualCount);
                    ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                    for (int i = 1; i < actualCount; ++i)
                    {
                        ASSERT_NE(0, nodes[i].index);
                        ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                    }
                }
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Joined");

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.ReadyToStart");

                // コンテンツ配信を開始します。
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Started");
            }
            else
            {
                // セッションの構築を待機します。
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.SessionOpened");

                // 自分の順番が回ってくるまでスキャンをして待ちます。
                nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
                int count;
                int index;
                do
                {
                    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

                // セッションに参加します。
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Joined");

                nn::os::SleepThread(StateChangeTimeout);
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.ReadyToStart");

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Started");

                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }

            if (g_TestConfig.nodeIndex == 0)
            {
                sleep.FixedSleep();

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Leaved");

                nn::os::Tick timeout = nn::os::GetSystemTick() + nn::os::Tick(ShareCompleteTimeout);
                while (!nn::os::TimedWaitSystemEvent(&stateChangeEvent, ShareCompleteTimeout))
                {
                    ASSERT_LT(nn::os::GetSystemTick(), timeout);
                }

                auto state = nn::lcs::GetState();
                if (state == nn::lcs::State_ContentsShareFailed)
                {
                    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_CommunicationError, nn::lcs::GetContentsShareFailureReason());
                }
                else
                {
                    ASSERT_EQ(nn::lcs::State_Completed, state);
                }
            }
            else
            {
                sleep.FlexSleep();

                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Leaved");
            }
        }

        NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveClients.Finalized");
    }
}  // NOLINT(impl/function_size)

//
// コンテンツ配信を開始してから親交代をしているタイミングで、受信者が
// LeaveSession() によって離脱したとき、意図しない状態にならないことを検証します。
//
TEST(AfterStart, SimultaneousHostMigrationLeaveSenders)
{
    const auto cond = (GetPrecondition("AfterStart", "SimultaneousHostMigrationLeaveSenders", nnt::lcs::Application::Timing02));

    // テストケース開始前の同期です。
    NNT_LDN_REQUIRES_NODE_COUNT(2);
    NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Start");

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-400), nn::TimeSpan::FromMilliSeconds(400), RepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    for (int testCount = 0; testCount < RepeatCount; ++testCount)
    {
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            if (g_TestConfig.nodeIndex == 0)
            {
                // ホストとしてセッションを構築します。
                const auto id = nnt::lcs::Application::Timing02.value;
                const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.SessionOpened");

                // クライアントの接続を待機します
                nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
                int actualCount = 0;
                while (actualCount < g_TestConfig.nodeCount)
                {
                    // クライアントの接続を待機します。
                    ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                    nn::os::ClearSystemEvent(&stateChangeEvent);

                    // 現在接続中のクライアントを確認します。
                    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                    ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                    ASSERT_LE(2, actualCount);
                    ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                    ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                    for (int i = 1; i < actualCount; ++i)
                    {
                        ASSERT_NE(0, nodes[i].index);
                        ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                    }
                }
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Joined");

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.ReadyToStart");

                // コンテンツ配信を開始します。
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
                ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Started");
            }
            else
            {
                // セッションの構築を待機します。
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.SessionOpened");

                // 自分の順番が回ってくるまでスキャンをして待ちます。
                nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
                int count;
                int index;
                do
                {
                    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
                } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

                // セッションに参加します。
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Joined");

                nn::os::SleepThread(StateChangeTimeout);
                ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.ReadyToStart");

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Started");

                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }

            if (g_TestConfig.nodeIndex == cond.masterIndex)
            {
                sleep.FixedSleep();

                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Leaved");
            }
            else
            {
                sleep.FlexSleep();

                NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Leaved");

                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ShareCompleteTimeout));
                ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
                ASSERT_PRED1([](nn::lcs::ContentsShareFailureReason reason) {
                    return 0
                        || reason == nn::lcs::ContentsShareFailureReason_NodeLeaved
                        || reason == nn::lcs::ContentsShareFailureReason_CommunicationError
                        ;
                }, nn::lcs::GetContentsShareFailureReason());
                nn::os::ClearSystemEvent(&stateChangeEvent);
            }
        }

        NNT_LDN_SYNC("AfterStart.SimultaneousHostMigrationLeaveSenders.Finalized");
    }
}  // NOLINT(impl/function_size)

// LUP 中の SuspendSession と ResumeSession を全クライアントが同時に行ったときに
// 意図しない状態にならないことを検証します。
//
TEST(Lup, SimultaneousSuspendAndResume)
{
    NNT_LDN_REQUIRES_NODE_COUNT(2);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    if (g_TestConfig.nodeIndex == 0)
    {
        // テストケース開始前の同期です。
        NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Start");

        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            nn::ns::SystemUpdateControl control;
            NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
            ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.SessionOpened");

            // クライアントの接続を待機します
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount = 0;
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Joined");

            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.ReadyToStart");

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Started");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.SystemReceived");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Rebooted");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Completed, &stateChangeEvent, ShareCompleteTimeout);

            // セッションを破棄します。
        }
        NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Leaved");
    }
    else
    {
        if (!g_TestConfig.resume)
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // テストケース開始前の同期です。
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Start");

            // セッションの構築を待機します。
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Joined");
            nn::os::SleepThread(StateChangeTimeout);
            ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.ReadyToStart");

            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Started");

            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // システムの受信をします。
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_EulaRequired, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

            // セッションコンテキストを保存します。
            nn::lcs::SessionContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));

            // 全クライアントが同期を取ってサスペンドします。
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.SystemReceived");
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SuspendSession());
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);

            // 再起動します。
            nnt::lcs::RequestReboot();
            nnt::lcs::RequestResume();
        }
        else
        {
            {
                nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

                // イベントを取得します。
                nn::os::SystemEventType stateChangeEvent;
                nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // セッションコンテキストを復元します。
                nn::lcs::SessionContext sessionContext;
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::LoadSessionContext(&sessionContext));

                // 全クライアントが同期をとってセッションを再開します。
                NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Rebooted");
                auto beginTick = nn::os::GetSystemTick();
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedResumeSession(sessionContext, JoinTimeout));
                auto endTick = nn::os::GetSystemTick();
                NN_LOG("Elapsed %d us\n", (endTick - beginTick).ToTimeSpan().GetMicroSeconds());

                // パッチを受信します。
                nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
                ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

                nnt::lcs::TimedWaitTransition(nn::lcs::State_Completed, &stateChangeEvent, ShareCompleteTimeout);
            }

            // セッションから離脱します。
            NNT_LDN_SYNC("Lup.SimultaneousSuspendAndResume.Leaved");
        }
    }
}  // NOLINT(impl/function_size)

// クライアントが SuspendSession() すると同時にホストが LeaveSession() をしても
// 意図しない状態にならないことを検証します。
//
TEST(Lup, SimultaneousLeaveSuspend)
{
    NNT_LDN_REQUIRES_NODE_COUNT(2);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-400), nn::TimeSpan::FromMilliSeconds(400), LupRepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    if (g_TestConfig.nodeIndex == 0)
    {
        // テストケース開始前の同期です。
        NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Start");

        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            nn::ns::SystemUpdateControl control;
            NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
            ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SessionOpened");

            // クライアントの接続を待機します
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount = 0;
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Joined");

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.ReadyToStart");

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Started");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SystemReceived");

            // 待ちます。
            sleep.FlexSleep();

            // セッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);
        }
        NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Leaved");
    }
    else
    {
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // テストケース開始前の同期です。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Start");

            // セッションの構築を待機します。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Joined");
            nn::os::SleepThread(StateChangeTimeout);
            ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.ReadyToStart");

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Started");

            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // システムの受信をします。
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_EulaRequired, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

            // セッションコンテキストを保存します。
            nn::lcs::SessionContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SystemReceived");

            // 待ちます
            sleep.FixedSleep();

            // 同期を取ってサスペンドします。
            auto suspendResult = nn::lcs::SuspendSession();

            // サスペンドのあとにセッション破棄された場合は成功します。
            if (suspendResult.IsSuccess())
            {
                nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);
            }
            // サスペンドより先にセッション破棄された場合は InvalidState により失敗します。
            else if (nn::lcs::ResultInvalidState::Includes(suspendResult))
            {
                NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, suspendResult);
                ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
            }
        }

        NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Leaved");
    }
}  // NOLINT(impl/function_size)

// クライアントが SuspendSession() すると同時にホストが FinalizeSession() をしても
// 意図しない状態にならないことを検証します。
//
TEST(Lup, SimultaneousFinalizeSuspend)
{
    NNT_LDN_REQUIRES_NODE_COUNT(2);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(-400), nn::TimeSpan::FromMilliSeconds(400), LupRepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    if (g_TestConfig.nodeIndex == 0)
    {
        // テストケース開始前の同期です。
        NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Start");

        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            nn::ns::SystemUpdateControl control;
            NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
            ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SessionOpened");

            // クライアントの接続を待機します
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount = 0;
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Joined");

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.ReadyToStart");

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Started");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SystemReceived");

            // 待ちます。
            sleep.FlexSleep();

            // セッションを破棄します。
        }
        NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Leaved");
    }
    else
    {
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // テストケース開始前の同期です。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Start");

            // セッションの構築を待機します。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Joined");
            nn::os::SleepThread(StateChangeTimeout);
            ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.ReadyToStart");

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Started");

            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // システムの受信をします。
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_EulaRequired, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

            // セッションコンテキストを保存します。
            nn::lcs::SessionContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));

            NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.SystemReceived");

            // 待ちます
            sleep.FixedSleep();

            // 同期を取ってサスペンドします。
            auto suspendResult = nn::lcs::SuspendSession();

            // サスペンドのあとにセッション破棄された場合は成功します。
            if (suspendResult.IsSuccess())
            {
                nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);
            }
            // サスペンドより先にセッション破棄された場合は InvalidState により失敗します。
            else if (nn::lcs::ResultInvalidState::Includes(suspendResult))
            {
                NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultInvalidState, suspendResult);
                ASSERT_EQ(nn::lcs::State_ContentsShareFailed, nn::lcs::GetState());
            }
        }

        NNT_LDN_SYNC("Lup.SimultaneousLeaveSuspend.Leaved");
    }
}  // NOLINT(impl/function_size)

// クライアントが ResumeSession() すると同時にホストが LeaveSession() をしても
// 意図しない状態にならないことを検証します。
//
TEST(Lup, SimultaneousLeaveResume)
{
    NNT_LDN_REQUIRES_NODE_COUNT(2);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(200), nn::TimeSpan::FromMilliSeconds(1000), LupRepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    if (g_TestConfig.nodeIndex == 0)
    {
        // テストケース開始前の同期です。
        NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Start");

        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            nn::ns::SystemUpdateControl control;
            NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
            ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.SessionOpened");

            // クライアントの接続を待機します
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount = 0;
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Joined");

            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.ReadyToStart");

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Started");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.SystemReceived");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Rebooted");

            // 待ちます。
            sleep.FlexSleep();

            // セッションを破棄します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::LeaveSession());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);
        }
        NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Leaved");
    }
    else
    {
        if (!g_TestConfig.resume)
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // テストケース開始前の同期です。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Start");

            // セッションの構築を待機します。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Joined");
            nn::os::SleepThread(StateChangeTimeout);
            ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.ReadyToStart");

            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Started");

            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // システムの受信をします。
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_EulaRequired, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

            // セッションコンテキストを保存します。
            nn::lcs::SessionContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));

            // 全クライアントが同期を取ってサスペンドします。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.SystemReceived");
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SuspendSession());
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);

            // 再起動します。
            nnt::lcs::RequestReboot();
            nnt::lcs::RequestResume();
        }
        else
        {
            {
                nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

                // イベントを取得します。
                nn::os::SystemEventType stateChangeEvent;
                nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // セッションコンテキストを復元します。
                nn::lcs::SessionContext sessionContext;
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::LoadSessionContext(&sessionContext));

                // 全クライアントが同期をとってセッションを再開します。
                NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Rebooted");

                // 待ちます。
                sleep.FixedSleep();

                auto resumeResult = nnt::lcs::TimedResumeSession(sessionContext, JoinTimeout);
                if (!nn::lcs::ResultCommunicationError::Includes(resumeResult))
                {
                    NNT_ASSERT_RESULT_SUCCESS(resumeResult);

                    // セッション再開に成功したときはホストからの切断による失敗を待ちます。
                    nnt::lcs::TimedWaitTransition(nn::lcs::State_ContentsShareFailed, &stateChangeEvent, StateChangeTimeout);
                    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
                }
                else
                {
                    auto state = nn::lcs::GetState();

                    // 再接続後に切断された時は State_ContentsShareFailure になります。
                    if (state == nn::lcs::State_ContentsShareFailed)
                    {
                        ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
                    }
                    // セッションが見つかなかった時は State_Initialized になります。
                    else
                    {
                        ASSERT_EQ(nn::lcs::State_Initialized, state);
                    }
                }
            }

            // セッションから離脱します。
            NNT_LDN_SYNC("Lup.SimultaneousLeaveResume.Leaved");
        }
    }
}  // NOLINT(impl/function_size)

// クライアントが ResumeSession() すると同時にホストが Finalize() をしても
// 意図しない状態にならないことを検証します。
//
TEST(Lup, SimultaneousFinalizeResume)
{
    NNT_LDN_REQUIRES_NODE_COUNT(2);

    // NS, LDN, LCS ライブラリを初期化します。
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;

    // 待ち時間の変動幅を設定します。
    FlexSleeper sleep(nn::TimeSpan::FromMilliSeconds(200), nn::TimeSpan::FromMilliSeconds(1000), LupRepeatCount);

    nn::lcs::Config config;
    if (g_TestConfig.nodeIndex == 0)
    {
        config = nnt::lcs::CreateDefaultHostConfig();
    }
    else
    {
        config = nnt::lcs::CreateDefaultClientConfig();
    }

    if (g_TestConfig.nodeIndex == 0)
    {
        // テストケース開始前の同期です。
        NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Start");

        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            nn::ns::SystemUpdateControl control;
            NNT_ASSERT_RESULT_FAILURE(nn::ns::ResultAlreadyOccupied, control.Occupy());

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // ホストとしてセッションを構築します。
            const auto id = nnt::lcs::Application::Timing01.value;
            const auto settings = nnt::lcs::CreateSessionSettings(nn::lcs::NodeCountMax, id);
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
            ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.SessionOpened");

            // クライアントの接続を待機します
            nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];
            int actualCount = 0;
            while (actualCount < g_TestConfig.nodeCount)
            {
                // クライアントの接続を待機します。
                ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, ScanTimeout + JoinTimeout));
                ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
                nn::os::ClearSystemEvent(&stateChangeEvent);

                // 現在接続中のクライアントを確認します。
                NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetNodes(nodes, &actualCount, nn::lcs::NodeCountMax));

                ASSERT_GE(g_TestConfig.nodeCount, actualCount);
                ASSERT_LE(2, actualCount);
                ASSERT_EQ(nn::lcs::GetMyIndex(), nodes[0].index);
                ASSERT_STREQ(nnt::lcs::DefaultHostUserName, nodes[0].userName);
                for (int i = 1; i < actualCount; ++i)
                {
                    ASSERT_NE(0, nodes[i].index);
                    ASSERT_STREQ(nnt::lcs::DefaultClientUserName, nodes[i].userName);
                }
            }
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Joined");

            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.ReadyToStart");

            // コンテンツ配信を開始します。
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::StartContentsShare());
            ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Started");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.SystemReceived");
            ASSERT_EQ(nn::lcs::State_Transferring, nn::lcs::GetState());

            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Rebooted");

            // 待ちます。
            sleep.FlexSleep();

            // セッションを破棄します。
        }
        NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Leaved");
    }
    else
    {
        if (!g_TestConfig.resume)
        {
            nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

            // イベントを取得します。
            nn::os::SystemEventType stateChangeEvent;
            nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
            ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

            // テストケース開始前の同期です。
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Start");

            // セッションの構築を待機します。
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.SessionOpened");

            // 自分の順番が回ってくるまでスキャンをして待ちます。
            nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
            int count;
            int index;
            do
            {
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(sessions, &count, &index, nn::lcs::ScanResultCountMax, nnt::lcs::DefaultHostUserName, ScanTimeout));
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
            } while (sessions[index].nodeCount < g_TestConfig.nodeIndex);

            // セッションに参加します。
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedJoin(sessions[index], JoinTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Joined");
            nn::os::SleepThread(StateChangeTimeout);
            ASSERT_EQ(nn::lcs::State_Joined, nn::lcs::GetState());
            nn::os::ClearSystemEvent(&stateChangeEvent);

            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.ReadyToStart");

            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Started");

            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&stateChangeEvent, StateChangeTimeout));
            nn::os::ClearSystemEvent(&stateChangeEvent);

            // システムの受信をします。
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_NeedTerminateApplication, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_EulaRequired, nn::lcs::GetSuspendedReason());
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::ResumeContentsShare());

            nnt::lcs::TimedWaitTransition(nn::lcs::State_Suspended, &stateChangeEvent, ShareCompleteTimeout);
            ASSERT_EQ(nn::lcs::SuspendedReason_RebootRequired, nn::lcs::GetSuspendedReason());

            // セッションコンテキストを保存します。
            nn::lcs::SessionContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));
            NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::SaveSessionContext(context));

            // 全クライアントが同期を取ってサスペンドします。
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.SystemReceived");
            NNT_ASSERT_RESULT_SUCCESS(nn::lcs::SuspendSession());
            nnt::lcs::TimedWaitTransition(nn::lcs::State_Initialized, &stateChangeEvent, StateChangeTimeout);

            // 再起動します。
            nnt::lcs::RequestReboot();
            nnt::lcs::RequestResume();
        }
        else
        {
            {
                nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

                // イベントを取得します。
                nn::os::SystemEventType stateChangeEvent;
                nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
                ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

                // セッションコンテキストを復元します。
                nn::lcs::SessionContext sessionContext;
                NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::LoadSessionContext(&sessionContext));

                // 全クライアントが同期をとってセッションを再開します。
                NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Rebooted");

                // 待ちます。
                sleep.FixedSleep();

                auto resumeResult = nnt::lcs::TimedResumeSession(sessionContext, JoinTimeout);
                if (!nn::lcs::ResultCommunicationError::Includes(resumeResult))
                {
                    NNT_ASSERT_RESULT_SUCCESS(resumeResult);

                    // セッション再開に成功したときはホストからの切断による失敗を待ちます。
                    nnt::lcs::TimedWaitTransition(nn::lcs::State_ContentsShareFailed, &stateChangeEvent, StateChangeTimeout);
                    ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
                }
                else
                {
                    auto state = nn::lcs::GetState();

                    // 再接続後に切断された時は State_ContentsShareFailure になります。
                    if (state == nn::lcs::State_ContentsShareFailed)
                    {
                        ASSERT_EQ(nn::lcs::ContentsShareFailureReason_NodeLeaved, nn::lcs::GetContentsShareFailureReason());
                    }
                    // セッションが見つかなかった時は State_Initialized になります。
                    else
                    {
                        ASSERT_EQ(nn::lcs::State_Initialized, state);
                    }
                }
            }

            // セッションから離脱します。
            NNT_LDN_SYNC("Lup.SimultaneousFinalizeResume.Leaved");
        }
    }
}  // NOLINT(impl/function_size)

//
// テストのエントリポイントです。
//
extern "C" void nnMain()
{
    // カスタムコマンドラインオプションの初期値を設定します。
    g_TestConfig.testCount = 0;

    // コマンドライン引数に関する設定です。
    nnt::lcs::CommandLineParserSetting setting = { };
    setting.flag = static_cast<nn::Bit32>(
        nnt::lcs::CommandLineOptionFlag_NodeCount |
        nnt::lcs::CommandLineOptionFlag_NodeIndex |
        nnt::lcs::CommandLineOptionFlag_SceneId |
        nnt::lcs::CommandLineOptionFlag_Resume |
        nnt::lcs::CommandLineOptionFlag_Custom);
    setting.nodeCountMin = 2;
    setting.nodeCountMax = nnt::ldn::SynchronizationClientCountMax;
    setting.parser = [](int* pOutIndex, char* argv[], int argc, int index) -> bool
    {
        if (argv[index][0] == '-' && argv[index][1] == 't')
        {
            char* end;
            g_TestConfig.testCount = std::strtol(argv[index + 1], &end, 10);

            *pOutIndex = index + 2;
        }
        else
        {
            return false;
        }

        return true;
    };

    // コマンドライン引数を解析します。
    int argc = nn::os::GetHostArgc();
    char **argv = nn::os::GetHostArgv();
    ::testing::InitGoogleTest(&argc, argv);
    nnt::lcs::Parse(&g_TestConfig, setting, argc, argv);

    // socket ライブラリと ns ライブラリを初期化しておきます。
    nn::socket::Initialize(g_SocketConfig);
    nnt::lcs::NsInitializer nsInitiailzer;

    // テストの準備です。
    nnt::lcs::Setup(g_TestConfig.sceneId);

    int exitCode;
    if (g_TestConfig.nodeIndex == 0)
    {
        // 他のインスタンスと同期するためにテストサーバを生成します。
        nnt::ldn::HtcsSynchronizationServer server(g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        auto result = server.CreateServer("nn::lcs::Integration::Timing", g_TestConfig.nodeCount - 1);

        // テストを実行します。
        if (result == nnt::ldn::SynchronizationResult_Success)
        {
            g_pSync = &server;
            exitCode = RUN_ALL_TESTS();
            server.DestroyServer();
        }
        else
        {
            exitCode = result;
        }
    }
    else
    {
        // 他のインスタンスと同期するためにテストサーバに接続します。
        nnt::ldn::HtcsSynchronizationClient client(g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        auto result = client.Connect("nn::lcs::Integration::Timing");

        // テストを実行します。
        if (result == nnt::ldn::SynchronizationResult_Success)
        {
            g_pSync = &client;
            exitCode = RUN_ALL_TESTS();
            client.Disconnect();
        }
        else
        {
            exitCode = result;
        }
    }

    nn::socket::Finalize();
    nnt::Exit(exitCode);
}
