﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/lcs/testLcs_Utility.h>
#include <nnt/ldn/testLdn_HtcsSynchronization.h>
#include <nnt/ldn/testLdn_Utility.h>
#include <nnt.h>
#include "Helper.h"

namespace
{
    // LCS の初期化用バッファです。
    NN_ALIGNAS(nn::lcs::RequiredBufferAlignment) char g_Buffer[4 * 1024 * 1024];
    NN_STATIC_ASSERT(nn::lcs::RequiredBufferSize < sizeof(g_Buffer));

    // 同期に使用するバッファです。
    NN_ALIGNAS(4096) char g_SynchronizationBuffer[16 * 1024];

    // 接続待ちのタイムアウト時間です。
    const nn::TimeSpan JoinTimeout = nn::TimeSpan::FromSeconds(5);

    // スキャン待ちのタイムアウト時間です。
    const nn::TimeSpan ScanTimeout = nn::TimeSpan::FromSeconds(4);

    // 状態変化待ちのタイムアウト時間です。
    const nn::TimeSpan StateChangeTimeout = nn::TimeSpan::FromSeconds(1);

} // namespace <unnamed>

SystemDeliveryProtocolTestHelper::SystemDeliveryProtocolTestHelper(
    const nnt::lcs::TestConfig& config, const char* group) NN_NOEXCEPT
    : m_pSync(nullptr),
      m_TestConfig(config)
{
    NN_ASSERT_NOT_NULL(group);
    std::strncpy(m_Group, group, nnt::ldn::SynchronizationGroupNameLengthMax);
    m_Group[nnt::ldn::SynchronizationGroupNameLengthMax] = 0;
}

void SystemDeliveryProtocolTestHelper::RunTest(
    const nnt::lcs::Precondition& cond, const char* prefix, int requiredNodeCount) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(prefix);
    std::strncpy(m_Prefix, prefix, nnt::lcs::PrefixLengthMax);
    m_Prefix[nnt::lcs::PrefixLengthMax] = 0;

    // 他のインスタンスと同期するためにテストグループに参加します。
    if (m_TestConfig.nodeIndex == cond.masterIndex)
    {
        auto* pServer = new nnt::ldn::HtcsSynchronizationServer(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, pServer->CreateServer(
            m_Group, m_TestConfig.nodeCount - 1));
        m_pSync = pServer;
    }
    else
    {
        auto* pClient = new nnt::ldn::HtcsSynchronizationClient(
            g_SynchronizationBuffer, sizeof(g_SynchronizationBuffer));
        ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, pClient->Connect(m_Group));
        m_pSync = pClient;
    }

    // クライアントの端末数の指定は必須ではないため、未入力時は LdnTestBridge から取得します。
    if (m_TestConfig.nodeCount == 0)
    {
        m_TestConfig.nodeCount = static_cast<int8_t>(m_pSync->GetClientCount() + 1);
    }

    // テストを終える際にテストグループから抜けます。
    NN_UTIL_SCOPE_EXIT
    {
        delete m_pSync;
        m_pSync = nullptr;

        // FIXME: 同じグループ名を使い回している場合に、
        // 誤って破棄される直前の古いグループに再度加入してしまうことを防ぎます。
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    };

    // 端末数が必要数に達している場合にテストを実行します。
    if (requiredNodeCount <= m_TestConfig.nodeCount)
    {
        RunTestImpl(cond);
    }
}

void SystemDeliveryProtocolTestHelper::RunTestImpl(
    const nnt::lcs::Precondition& cond) NN_NOEXCEPT
{
    // NS, LDN, LCS ライブラリを初期化します。
    nn::lcs::Config config = m_TestConfig.nodeIndex == 0 ?
        nnt::lcs::CreateDefaultHostConfig() : nnt::lcs::CreateDefaultClientConfig();
    nnt::lcs::NsInitializer nsInitializer;
    nnt::ldn::SystemInitializer ldnInitializer;
    nnt::lcs::Initializer lcsInitializer(g_Buffer, sizeof(g_Buffer), config);

    // イベントを取得します。
    nn::os::SystemEventType stateChangeEvent;
    nn::lcs::AttachStateChangeEvent(&stateChangeEvent);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    // ローカルパッチ配信を開始します。
    if (m_TestConfig.nodeIndex == 0)
    {
        StartHost(cond, &stateChangeEvent);
    }
    else
    {
        StartClient(cond, &stateChangeEvent);
    }
}

//
// ホストとしてローカルコンテンツ配信を開始します。
//
void SystemDeliveryProtocolTestHelper::StartHost(
    const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    auto& stateChangeEvent = *pEvent;

    // セッションを構築します。
    const auto settings = nnt::lcs::CreateSessionSettings(
        nn::lcs::NodeCountMax, cond.application.id);
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    NNT_ASSERT_RESULT_SUCCESS(nn::lcs::OpenSession(settings));
    ASSERT_TRUE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    nn::os::ClearSystemEvent(&stateChangeEvent);
    ASSERT_EQ(nn::lcs::State_Opened, nn::lcs::GetState());
    Sync("SessionOpened");

    // クライアントの接続を待機します。
    Sync("Joined");
}

//
// クライアントとしてローカルコンテンツ配信を開始します。
//
void SystemDeliveryProtocolTestHelper::StartClient(
    const nnt::lcs::Precondition& cond, nn::os::SystemEventType* pEvent) NN_NOEXCEPT
{
    auto& stateChangeEvent = *pEvent;

    // セッションの構築を待機します。
    Sync("SessionOpened");

    // セッションをスキャンします。
    nn::lcs::SessionInfo sessions[nn::lcs::ScanResultCountMax];
    int count;
    int index;
    NNT_ASSERT_RESULT_SUCCESS(nnt::lcs::TimedScan(
        sessions, &count, &index, nn::lcs::ScanResultCountMax,
        nnt::lcs::DefaultHostUserName, ScanTimeout));

    // ホストへの接続を試行しますが、SystemDeliveryProtocolVersion の相違のため失敗します。
    const auto& session = sessions[index];
    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));

    if (cond.nodes[0].system.systemDeliveryProtocolVersion >
        cond.nodes[m_TestConfig.nodeIndex].system.systemDeliveryProtocolVersion)
    {
        NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultLowerSystemVersion,
            nnt::lcs::TimedJoin(session, JoinTimeout));
    }
    else
    {
        NNT_ASSERT_RESULT_FAILURE(nn::lcs::ResultHigherSystemVersion,
            nnt::lcs::TimedJoin(session, JoinTimeout));
    }

    ASSERT_FALSE(nn::os::TryWaitSystemEvent(&stateChangeEvent));
    ASSERT_EQ(nn::lcs::State_Initialized, nn::lcs::GetState());
    Sync("Joined");
}

void SystemDeliveryProtocolTestHelper::Sync(const char* keyword) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(keyword);
    char buffer[nnt::ldn::SynchronizationKeywordLengthMax + 1];
    nn::util::SNPrintf(buffer, sizeof(buffer), "%s.%s", m_Prefix, keyword);
    ASSERT_EQ(nnt::ldn::SynchronizationResult_Success, m_pSync->Synchronize(buffer));
}
