﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/util/util_FormatString.h>
#include <nnt/lcs/testLcs_Precondition.h>

namespace nnt { namespace lcs { namespace
{
    void CopyImpl(Precondition* pOut, const PreconditionSource& source) NN_NOEXCEPT
    {
        // 0 で初期化します。
        auto& cond = *pOut;
        std::memset(&cond, 0, sizeof(Precondition));

        // source の情報を全て cond にコピーします。
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            {
                auto& dst = cond.nodes[i].application;
                const auto& src = source.nodes[i].application;
                dst.version = src.version;
                dst.requiredVersion = src.requiredVersion;
                dst.requiredSystemVersion = src.requiredSystemVersion;
            }
            {
                auto& dst = cond.nodes[i].system;
                const auto& src = source.nodes[i].system;
                dst.version = src.version;
                dst.requiredSystemVersion = src.requiredSystemVersion;
                dst.systemDeliveryProtocolVersion = src.systemDeliveryProtocolVersion;
                dst.applicationDeliveryProtocolVersion = src.applicationDeliveryProtocolVersion;
                dst.hasExFatDriver = src.hasExFatDriver;
                dst.hasSystemDeliveryProtocolBackwardCompatibility =
                    src.hasSystemDeliveryProtocolBackwardCompatibility;
                dst.hasApplicationDeliveryProtocolBackwardCompatibility =
                    src.hasApplicationDeliveryProtocolBackwardCompatibility;
                dst.hasEula = src.hasEula;
            }
            cond.nodes[i].isBuiltInUserStorageNotEnough = source.nodes[i].isBuiltInUserStorageNotEnough;
        }
        cond.application.id = source.application.id;
        cond.applicationForClean.id = source.applicationForClean.id;
        cond.scenario = source.scenario;
    }

    int FindMaster(const Precondition& cond) NN_NOEXCEPT
    {
        int masterIndex = 0;
        for (int i = 1; i < nn::lcs::NodeCountMax; ++i)
        {
            const auto& master = cond.nodes[masterIndex];
            const auto& node = cond.nodes[i];
            if (master.system.version < node.system.version)
            {
                masterIndex = i;
            }
        }
        return masterIndex;
    }

    int FindExFatSender(const Precondition& cond) NN_NOEXCEPT
    {
        int exFatSenderIndex = -1;
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            const auto& node = cond.nodes[i];
            if (node.system.hasExFatDriver)
            {
                if (exFatSenderIndex < 0 ||
                    cond.nodes[exFatSenderIndex].system.version < node.system.version)
                {
                    exFatSenderIndex = i;
                }
            }
        }
        return exFatSenderIndex;
    }

    void SetDisplayVersion(Precondition* pCond) NN_NOEXCEPT
    {
        auto& cond = *pCond;
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            auto& app = cond.nodes[i].application;
            ConvertToDisplayVersion(app.displayVersion, sizeof(app.displayVersion), app.version);
        }
        ConvertToDisplayVersion(
            cond.application.latestDisplayVersion,
            sizeof(cond.application.latestDisplayVersion),
            cond.application.latestVersion);
    }

}}} // namespace nnt::lcs::<unnamed>

namespace nnt { namespace lcs
{
    void PreconditionSource::SetApplicationVersion(int version) NN_NOEXCEPT
    {
        SetApplicationVersion(version, version, version);
    }

    void PreconditionSource::SetApplicationVersion(int host, int client) NN_NOEXCEPT
    {
        SetApplicationVersion(host, client, client);
    }

    void PreconditionSource::SetApplicationVersion(int host, int first, int others) NN_NOEXCEPT
    {
        nodes[0].application.version = host;
        nodes[1].application.version = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].application.version = others;
        }
    }

    void PreconditionSource::SetApplicationRequiredVersion(int version) NN_NOEXCEPT
    {
        SetApplicationRequiredVersion(version, version, version);
    }

    void PreconditionSource::SetApplicationRequiredVersion(int host, int client) NN_NOEXCEPT
    {
        SetApplicationRequiredVersion(host, client, client);
    }

    void PreconditionSource::SetApplicationRequiredVersion(
        int host, int first, int others) NN_NOEXCEPT
    {
        nodes[0].application.requiredVersion = host;
        nodes[1].application.requiredVersion = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].application.requiredVersion = others;
        }
    }

    void PreconditionSource::SetApplicationRequiredSystemVersion(
        int version) NN_NOEXCEPT
    {
        SetApplicationRequiredSystemVersion(version, version, version);
    }

    void PreconditionSource::SetApplicationRequiredSystemVersion(
        int host, int client) NN_NOEXCEPT
    {
        SetApplicationRequiredSystemVersion(host, client, client);
    }

    void PreconditionSource::SetApplicationRequiredSystemVersion(
        int host, int first, int others) NN_NOEXCEPT
    {
        nodes[0].application.requiredSystemVersion = host;
        nodes[1].application.requiredSystemVersion = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].application.requiredSystemVersion = others;
        }
    }

    void PreconditionSource::SetSystemVersion(int version) NN_NOEXCEPT
    {
        SetSystemVersion(version, version, version);
    }

    void PreconditionSource::SetSystemVersion(int host, int client) NN_NOEXCEPT
    {
        SetSystemVersion(host, client, client);
    }

    void PreconditionSource::SetSystemVersion(int host, int first, int others) NN_NOEXCEPT
    {
        nodes[0].system.version = host;
        nodes[1].system.version = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].system.version = others;
        }
    }

    void PreconditionSource::SetSystemRequiredSystemVersion(
        int version) NN_NOEXCEPT
    {
        SetSystemRequiredSystemVersion(version, version, version);
    }

    void PreconditionSource::SetSystemRequiredSystemVersion(
        int host, int client) NN_NOEXCEPT
    {
        SetSystemRequiredSystemVersion(host, client, client);
    }

    void PreconditionSource::SetSystemRequiredSystemVersion(
        int host, int first, int others) NN_NOEXCEPT
    {
        nodes[0].system.requiredSystemVersion = host;
        nodes[1].system.requiredSystemVersion = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].system.requiredSystemVersion = others;
        }
    }

    void PreconditionSource::SetSystemDeliveryProtocolVersion(
        int version) NN_NOEXCEPT
    {
        SetSystemDeliveryProtocolVersion(version, version, version);
    }

    void PreconditionSource::SetSystemDeliveryProtocolVersion(
        int host, int client) NN_NOEXCEPT
    {
        SetSystemDeliveryProtocolVersion(host, client, client);
    }

    void PreconditionSource::SetSystemDeliveryProtocolVersion(
        int host, int first, int others) NN_NOEXCEPT
    {
        nodes[0].system.systemDeliveryProtocolVersion = host;
        nodes[1].system.systemDeliveryProtocolVersion = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].system.systemDeliveryProtocolVersion = others;
        }
    }

    void PreconditionSource::SetApplicationDeliveryProtocolVersion(
        int version, bool hasCompatibility) NN_NOEXCEPT
    {
        SetApplicationDeliveryProtocolVersion(version, version, version, hasCompatibility);
    }

    void PreconditionSource::SetApplicationDeliveryProtocolVersion(
        int host, int client, bool hasCompatibility) NN_NOEXCEPT
    {
        SetApplicationDeliveryProtocolVersion(host, client, client, hasCompatibility);
    }

    void PreconditionSource::SetApplicationDeliveryProtocolVersion(
        int host, int first, int others, bool hasCompatibility) NN_NOEXCEPT
    {
        // バージョン番号を設定します。
        nodes[0].system.applicationDeliveryProtocolVersion = host;
        nodes[1].system.applicationDeliveryProtocolVersion = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].system.applicationDeliveryProtocolVersion = others;
        }

        // 互換性の有無を設定します。
        if (!hasCompatibility)
        {
            int latestVersion = std::max(std::max(host, first), others);
            for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
            {
                if (nodes[i].system.applicationDeliveryProtocolVersion == latestVersion)
                {
                    nodes[i].system.hasApplicationDeliveryProtocolBackwardCompatibility = false;
                }
            }
        }
    }

    void PreconditionSource::SetExFatDriverEnabled(bool enabled) NN_NOEXCEPT
    {
        SetExFatDriverEnabled(enabled, enabled, enabled);
    }

    void PreconditionSource::SetExFatDriverEnabled(bool host, bool client) NN_NOEXCEPT
    {
        SetExFatDriverEnabled(host, client, client);
    }

    void PreconditionSource::SetExFatDriverEnabled(bool host, bool first, bool others) NN_NOEXCEPT
    {
        nodes[0].system.hasExFatDriver = host;
        nodes[1].system.hasExFatDriver = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].system.hasExFatDriver = others;
        }
    }

    void PreconditionSource::SetEulaContained(bool hasEula) NN_NOEXCEPT
    {
        SetEulaContained(hasEula, hasEula, hasEula);
    }

    void PreconditionSource::SetEulaContained(bool host, bool client) NN_NOEXCEPT
    {
        SetExFatDriverEnabled(host, client, client);
    }

    void PreconditionSource::SetEulaContained(bool host, bool first, bool others) NN_NOEXCEPT
    {
        nodes[0].system.hasEula = host;
        nodes[1].system.hasEula = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].system.hasEula = others;
        }
    }

    void PreconditionSource::SetBuiltInUserStorageNotEnough(bool isFilled) NN_NOEXCEPT
    {
        SetBuiltInUserStorageNotEnough(isFilled, isFilled, isFilled);
    }

    void PreconditionSource::SetBuiltInUserStorageNotEnough(bool host, bool client) NN_NOEXCEPT
    {
        SetBuiltInUserStorageNotEnough(host, client, client);
    }

    void PreconditionSource::SetBuiltInUserStorageNotEnough(bool host, bool first, bool others) NN_NOEXCEPT
    {
        nodes[0].isBuiltInUserStorageNotEnough = host;
        nodes[1].isBuiltInUserStorageNotEnough = first;
        for (int i = 2; i < nn::lcs::NodeCountMax; ++i)
        {
            nodes[i].isBuiltInUserStorageNotEnough = others;
        }
    }

    PreconditionSource CreateDefaultPreconditionSource() NN_NOEXCEPT
    {
        PreconditionSource cond = {};
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            auto& node = cond.nodes[i];
            auto& system = node.system;
            system.systemDeliveryProtocolVersion = 1;
            system.applicationDeliveryProtocolVersion = 1;
            system.hasExFatDriver = true;
            system.hasSystemDeliveryProtocolBackwardCompatibility = true;
            system.hasApplicationDeliveryProtocolBackwardCompatibility = true;
            system.hasEula = false;
        }
        cond.scenario = Scenario_Normal;
        return cond;
    }

    Precondition Calculate(const PreconditionSource& source) NN_NOEXCEPT
    {
        // 入力情報をコピーします。
        Precondition cond;
        CopyImpl(&cond, source);

        // マスターになる端末と exFAT ドライバの送信者となる端末を検索します。
        cond.masterIndex = static_cast<int8_t>(FindMaster(cond));
        const auto& master = cond.nodes[cond.masterIndex];
        cond.system.latestVersion = master.system.version;
        cond.exFatSenderIndex = static_cast<int8_t>(FindExFatSender(cond));
        if (0 <= cond.exFatSenderIndex)
        {
            const auto& exFatSender = cond.nodes[cond.exFatSenderIndex];
            cond.system.latestVersionExFat = exFatSender.system.version;
        }

        // 最新バージョンのパッチを検索します。
        auto& app = cond.application;
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            const auto& node = source.nodes[i];
            if (app.latestVersion < node.application.version)
            {
                app.latestVersion = node.application.version;
                app.minimumRequiredSystemVersion = node.application.requiredSystemVersion;
            }
            else if (app.latestVersion == node.application.version)
            {
                app.minimumRequiredSystemVersion = std::min(
                    app.minimumRequiredSystemVersion, node.application.requiredSystemVersion);
            }
        }

        // Display Version を計算します。
        SetDisplayVersion(&cond);

        // パッチを受信するために必要なバージョンを計算します。
        if (master.system.hasApplicationDeliveryProtocolBackwardCompatibility)
        {
            cond.system.protocolRequiredSystemVersion = 0U;
        }
        else
        {
            cond.system.protocolRequiredSystemVersion = master.system.version;
            for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
            {
                const auto& node = cond.nodes[i];
                if (node.system.applicationDeliveryProtocolVersion ==
                    master.system.applicationDeliveryProtocolVersion &&
                    node.system.version < cond.system.protocolRequiredSystemVersion)
                {
                    cond.system.protocolRequiredSystemVersion = node.system.version;
                }
            }
        }
        cond.system.requiredSystemVersion = std::max(std::max(
            cond.application.minimumRequiredSystemVersion,
            master.system.requiredSystemVersion),
            cond.system.protocolRequiredSystemVersion
        );

        // システム配信の必要性を計算します。
        // TODO :  i == 2 で 3 台目の端末で固定してしまっているので、SIGLO-76573 で対応予定。
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            auto& node = cond.nodes[i];
            node.system.isLupEnabled =
                (!node.system.hasExFatDriver) ||
                (cond.system.requiredSystemVersion <= cond.system.latestVersionExFat);
            node.system.isLupRequired =
                (node.application.version < cond.application.latestVersion &&
                 node.system.version < cond.application.minimumRequiredSystemVersion) ||
                (!master.system.hasApplicationDeliveryProtocolBackwardCompatibility &&
                 node.system.applicationDeliveryProtocolVersion <
                 master.system.applicationDeliveryProtocolVersion) ||
                (node.system.version < master.system.requiredSystemVersion &&
                 node.application.version < cond.application.latestVersion) ||
                (cond.application.latestVersion <= node.application.version &&
                 node.system.version < master.system.requiredSystemVersion &&
                 node.system.isLupEnabled && i == 2);
        }

        // パッチ配信が可能な組み合わせか否かを計算します。
        uint32_t sharableApplicationVersion = 0;
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            const auto& node = cond.nodes[i];
            if (((!node.system.isLupRequired) &&
                (node.system.version >= master.system.requiredSystemVersion)) ||
                (node.system.isLupRequired && node.system.isLupEnabled))
            {
                if (sharableApplicationVersion < node.application.version)
                {
                    sharableApplicationVersion = node.application.version;
                }
            }
        }
        cond.isPatchShareEnabled = sharableApplicationVersion == cond.application.latestVersion;

        // セッションで配信しようとするパッチのバージョンバージョンを計算します。
        // TODO :  不要な判定をしているかもしれないので、SIGLO-76573 で検証予定。
        uint32_t targetApplicationVersion = 0;
        for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
        {
            const auto& node = cond.nodes[i];
            if (cond.system.protocolRequiredSystemVersion <
                node.system.applicationDeliveryProtocolVersion)
            {
                if (targetApplicationVersion < node.application.version)
                {
                    targetApplicationVersion = node.application.version;
                }
            }
        }
        cond.application.targetVersion = targetApplicationVersion;

        return cond;
    }

    void ConvertToDisplayVersion(char* buffer, size_t bufferSize, int version) NN_NOEXCEPT
    {
        if (version <= 1)
        {
            std::memset(buffer, 0, bufferSize);
        }
        else
        {
            nn::util::SNPrintf(buffer, bufferSize, "%d.0", version);
        }
    }

}} // namespace nnt::lcs
