﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include "testLMem_Util.h"

#include <random>

// =============
// static 関数
// =============

namespace {

const size_t g_ThreadStackSize = 8192;
NN_ALIGNAS(4096) char g_ThreadStack[g_ThreadStackSize];
nn::os::ThreadType g_Thread;
nn::os::EventType g_Event;

/**
 * @brief   ExpHeap を繰り返し作成します。
 */
void CreateExpHeapConstantly(void* arg)
{
    NN_UNUSED(arg);

    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    void* heapAddr;
    nn::lmem::HeapHandle heapHandle;

    heapAddr = std::malloc(HeapSize);

    // 繰り返しヒープを作成
    for(int i = 0; i < 1024; ++i)
    {
        heapHandle = nn::lmem::CreateExpHeap(heapAddr, HeapSize, nn::lmem::CreationOption_NoOption);
        ASSERT_NOT_NULL(heapHandle);     // ヒープ作成に成功したか
    }

    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heapAddr);

    nn::os::SignalEvent(&g_Event);
}

/**
 * @brief       VisitExpHeapAllBlocks() のテストで使われるコールバック関数です。
 * @param[in]   pBlock  メモリブロックへのポインタ
 * @param[in]   heap    メモリブロックを持つヒープ
 * @param[in]   pParam  受け取った文字列
 * @details     全てのメモリブロックに対して WriteNumericalSequence() を呼び値を代入します。
 */
void WriteVisitor(void* pBlock, nn::lmem::HeapHandle heap, uintptr_t pParam)
{
    NN_UNUSED(heap);
    char* str = reinterpret_cast<char*>(pParam);
    // 指定した文字列を受け取れているか
    EXPECT_STREQ("param test", str);

    size_t size = nn::lmem::GetExpHeapBlockSize(pBlock);
    EXPECT_TRUE(WriteNumericalSequence(pBlock, size));
}

/**
 * @brief       VisitExpHeapAllBlocks() のテストで使われるコールバック関数です。
 * @param[in]   pBlock  メモリブロックへのポインタ
 * @param[in]   heap    メモリブロックを持つヒープ
 * @param[in]   pParam  受け取った文字列
 * @details     全てのメモリブロックに対して HasNumericalSequence() を呼び値をチェックします。
 */
void ReadVisitor(void* pBlock, nn::lmem::HeapHandle heap, uintptr_t pParam)
{
    NN_UNUSED(heap);
    char* str = reinterpret_cast<char*>(pParam);
    // 指定した文字列を受け取れているか
    EXPECT_STREQ("param test", str);

    size_t size = nn::lmem::GetExpHeapBlockSize(pBlock);
    EXPECT_TRUE(HasNumericalSequence(pBlock, size));
}

}   // unnamed namespace


// =============
// テスト
// =============

// 単純なヒープの作成
// ヒープを作成します。ヒープの作成が成功（ハンドルが nullptr でない）なら OK
TEST(ExpHeapBasicTest, SimpleCreateHeap)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    void* heap;

    heap = std::malloc(HeapSize);

    // ヒープの作成
    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(heap, HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);     // ヒープ作成に成功したか
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// ヒープの作成と単一メモリブロックの確保
// メモリブロックの確保が、正しいアドレス領域に意図した状態で行われていれば、 OK
TEST(ExpHeapBasicTest, CreateAndAllocate)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    void* pMemoryAddress;

    // ヒープの作成とメモリ確保
    heapHandle = nn::lmem::CreateExpHeap(heap, HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);  // メモリ確保に成功したか

    // ヒープとメモリブロックのチェック
    EXPECT_TRUE(CheckExpHeapBlock(heapHandle, pMemoryAddress, nn::lmem::ErrorOption_NoOption));
    EXPECT_TRUE(CheckExpHeap(heapHandle, nn::lmem::ErrorOption_NoOption));

    // デフォルトのアロケート方向である
    EXPECT_EQ(nn::lmem::GetExpHeapAllocationDirectionOfBlock(pMemoryAddress), nn::lmem::AllocationDirection_Front);
    // デフォルトのグループIDである
    EXPECT_EQ(nn::lmem::GetExpHeapGroupId(heapHandle), 0);

    // 確保したメモリブロックのサイズが、指定したサイズにアライメントを付加した範囲内に収まっている
    EXPECT_GE(nn::lmem::GetExpHeapBlockSize(pMemoryAddress), AllocateSize);
    EXPECT_LE(nn::lmem::GetExpHeapBlockSize(pMemoryAddress), AllocateSize + nn::lmem::DefaultAlignment);

    // ヒープの空き領域が正しい範囲内に収まっている
    EXPECT_GE(nn::lmem::GetExpHeapTotalFreeSize(heapHandle), static_cast<size_t>(0));
    EXPECT_LE(nn::lmem::GetExpHeapTotalFreeSize(heapHandle), HeapSize - AllocateSize);

    // 確保可能なメモリブロックのサイズが正しい範囲内に収まっている
    EXPECT_GE(nn::lmem::GetExpHeapAllocatableSize(heapHandle, nn::lmem::DefaultAlignment), static_cast<size_t>(0));
    EXPECT_LE(nn::lmem::GetExpHeapAllocatableSize(heapHandle, nn::lmem::DefaultAlignment), HeapSize - AllocateSize);

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// メモリブロックのリサイズ
// 確保したメモリブロックを半分のサイズにした後、元に戻します。
// リサイズが失敗せず、リサイズしたメモリブロックに R/W できることを確認したら、OK
TEST(ExpHeapBasicTest, ResizeBlock)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    void* pMemoryAddress;

    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);

    // Read / Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, AllocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, AllocateSize));

    // メモリブロックのサイズを元の半分に変更
    EXPECT_NE(nn::lmem::ResizeExpHeapBlock(heapHandle, pMemoryAddress, AllocateSize / 2), static_cast<size_t>(0));

    // Read テスト
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, AllocateSize / 2));

    // メモリブロックのサイズを変化させないが、 ResizeExpHeapBlock() を呼ぶ
    EXPECT_TRUE(nn::lmem::ResizeExpHeapBlock(heapHandle, pMemoryAddress, AllocateSize / 2) != 0);

    // メモリブロックのサイズを現在の2倍（元の大きさ）に変更
    EXPECT_TRUE(nn::lmem::ResizeExpHeapBlock(heapHandle, pMemoryAddress, AllocateSize) != 0);

    // Read / Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, AllocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, AllocateSize));

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// 複数メモリブロックの確保と visitor 関数を用いた全メモリブロックに対する処理
// visitor 関数が確保した全てのメモリブロックで呼ばれていれば、 OK
TEST(ExpHeapBasicTest, Visitor)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 32;
    void* heap;
    const int BlockNum = 4;
    void* pMemoryAddresses[BlockNum];

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;

    heapHandle = nn::lmem::CreateExpHeap(heap, HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    // メモリブロックを複数確保
    for(int i =0; i < BlockNum; i++)
    {
        pMemoryAddresses[i] = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
        ASSERT_NOT_NULL(pMemoryAddresses[i]);
    }

    // 全てのメモリブロックに対して、 WriteVisitor と ReadVisitor を呼び、
    // 引数の受け渡しチェックとメモリブロック操作の確認を行う
    const char* str = "param test";
    nn::lmem::VisitExpHeapAllBlocks(heapHandle, WriteVisitor, reinterpret_cast<uintptr_t>(str));
    nn::lmem::VisitExpHeapAllBlocks(heapHandle, ReadVisitor, reinterpret_cast<uintptr_t>(str));

    for(int i =0; i < BlockNum; i++)
    {
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[i]);
    }

    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// 確保されたメモリブロックのサイズが正しくなっているか
// ヒープ領域に確保できない無効な領域が発生しないか
// 各サイズ取得と、確保時に得られるアドレスから求められる算出結果に矛盾がなければ、OK
TEST(ExpHeapBasicTest, Size)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    const int MinAlign = 4;
    void* heap;

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(heap, HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, MinAlign);
    ASSERT_NOT_NULL(pMemoryAddress);
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);

    // ヒープのヘッダ + メモリブロックのヘッダのサイズを求める
    // （pMemoryAddressには確保されたメモリの先頭アドレスが入る）
    size_t headSize = reinterpret_cast<uintptr_t>(pMemoryAddress) - reinterpret_cast<uintptr_t>(heap);

    void* pMemoryAddress1 = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize / 2, MinAlign);
    ASSERT_NOT_NULL(pMemoryAddress1);
    void* pMemoryAddress2 = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize / 2, MinAlign);
    ASSERT_NOT_NULL(pMemoryAddress2);
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress1);
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress2);

    // メモリブロックのヘッダサイズを求める
    size_t memoryBlockHeadSize = reinterpret_cast<uintptr_t>(pMemoryAddress2) - reinterpret_cast<uintptr_t>(pMemoryAddress1) - AllocateSize / 2;

    // 何も確保されていない状態でのサイズチェック
    size_t allocatableSize = nn::lmem::GetExpHeapAllocatableSize(heapHandle, MinAlign);
    size_t totalSize = nn::lmem::GetTotalSize(heapHandle);
    size_t totalFreeSize = nn::lmem::GetExpHeapTotalFreeSize(heapHandle);

    EXPECT_EQ(totalFreeSize, allocatableSize);
    EXPECT_EQ(totalSize, totalFreeSize + headSize);

    // メモリブロックの確保
    pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, MinAlign);

    // メモリが確保された状態でのサイズチェック
    allocatableSize = nn::lmem::GetExpHeapAllocatableSize(heapHandle, MinAlign);
    totalFreeSize = nn::lmem::GetExpHeapTotalFreeSize(heapHandle);
    EXPECT_EQ(totalFreeSize, allocatableSize);
    EXPECT_EQ(totalSize, totalFreeSize + headSize + memoryBlockHeadSize + AllocateSize);
    EXPECT_EQ(nn::lmem::GetExpHeapBlockSize(pMemoryAddress), AllocateSize);

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// 正しいグループIDが各メモリブロックに割り当てられているか
// グループIDが正しく設定できていれば、OK
TEST(ExpHeapBasicTest, GroupId)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 32;
    void* heap;
    const int AddressCount = 4;
    void* pMemoryAddresses[AddressCount];

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    for(int i = 0; i < AddressCount; ++i)
    {
        // メモリを確保した際のメモリブロックのグループIDは、
        // 確保時のヒープのグループIDと同じものになる
        nn::lmem::SetExpHeapGroupId(heapHandle, static_cast<nn::Bit16>(i));
        pMemoryAddresses[i] = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
        ASSERT_NOT_NULL(pMemoryAddresses[i]);
    }

    // グループID のチェック
    for(int i =0; i < AddressCount; i++)
    {
        nn::Bit16 groupId = nn::lmem::GetExpHeapGroupIdOfBlock(pMemoryAddresses[i]);
        EXPECT_EQ(groupId, static_cast<nn::Bit16>(i));
    }

    for(int i =0; i < AddressCount; i++)
    {
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[i]);
    }

    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// メモリが正しい方向から確保できているか
// 確保時のアライメントの正負によって、GetExpHeapAllocationDirectionOfBlock() の値が変われば、OK
TEST(ExpHeapBasicTest, AllocationDirection)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 64;
    void* heap;

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    // アライメントに負の値を入れると、逆方向から確保する
    void* pMemoryAddress1 = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, nn::lmem::DefaultAlignment);
    ASSERT_NOT_NULL(pMemoryAddress1);
    void* pMemoryAddress2 = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, -nn::lmem::DefaultAlignment);
    ASSERT_NOT_NULL(pMemoryAddress2);

    EXPECT_EQ(nn::lmem::GetExpHeapAllocationDirectionOfBlock(pMemoryAddress1), nn::lmem::AllocationDirection_Front);
    EXPECT_EQ(nn::lmem::GetExpHeapAllocationDirectionOfBlock(pMemoryAddress2), nn::lmem::AllocationDirection_Rear);

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress1);
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress2);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// ヒープ領域の縮小 (Adjust) が正常に行えるか
// AdjustExpHeap() によって得られる縮小された実際のサイズと、
// アドレスと確保サイズから算出した理論上縮小されるはずのサイズが一致すれば、OK
TEST(ExpHeapBasicTest, Adjust)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    // 末尾からの Adjust が成功する場合
    {
        nn::lmem::HeapHandle heapHandle;
        heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
        ASSERT_NOT_NULL(heapHandle);
        void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
        ASSERT_NOT_NULL(pMemoryAddress);
        size_t headSize = reinterpret_cast<uintptr_t>(pMemoryAddress) - reinterpret_cast<uintptr_t>(heap);

        // Adjust のテスト
        // ヒープの空き領域が解放され、現在確保されている領域とヒープの領域が一致するようになる
        nn::lmem::MemoryRange cutSize = nn::lmem::AdjustExpHeap(heapHandle);
        EXPECT_EQ(cutSize.size, HeapSize - (nn::lmem::GetExpHeapBlockSize(pMemoryAddress) + headSize));
        EXPECT_EQ(nn::lmem::GetTotalSize(heapHandle), nn::lmem::GetExpHeapBlockSize(pMemoryAddress) + headSize);

        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
        nn::lmem::DestroyExpHeap(heapHandle);
    }

    // 末尾からの Adjust が失敗する場合
    {
        nn::lmem::HeapHandle heapHandle;
        heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
        ASSERT_NOT_NULL(heapHandle);
        // 末尾からメモリブロックを確保
        void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, -nn::lmem::DefaultAlignment);
        ASSERT_NOT_NULL(pMemoryAddress);

        // Adjust のテスト
        // Adjust に失敗した場合は、 MemoryRange の size に 0 が入る
        nn::lmem::MemoryRange cutSize = nn::lmem::AdjustExpHeap(heapHandle);
        EXPECT_EQ(cutSize.size, 0);

        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
        nn::lmem::DestroyExpHeap(heapHandle);
    }

    // メモリを確保していない時の Adjust
    {
        nn::lmem::HeapHandle heapHandle;
        heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
        ASSERT_NOT_NULL(heapHandle);

        // ヒープ共通ヘッダのサイズを求める
        size_t commonHeapHeadSize = reinterpret_cast<uintptr_t>(nn::lmem::GetStartAddress(heapHandle)) - reinterpret_cast<uintptr_t>(heap);

        // Adjust のテスト
        // 未使用領域が全てヒープ領域でなくなるので、共通ヘッダ以外の領域がなくなる
        nn::lmem::MemoryRange cutSize = nn::lmem::AdjustExpHeap(heapHandle);
        EXPECT_EQ(cutSize.size, HeapSize - commonHeapHeadSize);
        EXPECT_EQ(nn::lmem::GetTotalSize(heapHandle), commonHeapHeadSize);

        nn::lmem::DestroyExpHeap(heapHandle);
    }
    std::free(heap);
}

// ヒープが壊れていないかどうかチェックすることができるか
// 正常なヒープで CheckExpHeap() が true を返すなら、OK
TEST(ExpHeapBasicTest, Check)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    // 拡張ヒープの作成とメモリ領域の確保
    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_ZeroClear);
    ASSERT_NOT_NULL(heapHandle);
    void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);
    size_t headSize = reinterpret_cast<uintptr_t>(pMemoryAddress) - reinterpret_cast<uintptr_t>(heap);

    // Adjust のテスト
    nn::lmem::MemoryRange newSize = nn::lmem::AdjustExpHeap(heapHandle);
    EXPECT_EQ(newSize.size, HeapSize - (nn::lmem::GetExpHeapBlockSize(pMemoryAddress) + headSize));
    EXPECT_EQ(nn::lmem::GetTotalSize(heapHandle), nn::lmem::GetExpHeapBlockSize(pMemoryAddress) + headSize);

    // 領域縮小後のヒープが壊れていないかテスト
    EXPECT_TRUE(nn::lmem::CheckExpHeap(heapHandle, nn::lmem::ErrorOption_NoOption));

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// メモリの確保と解放でおかしなフラグメンテーション（利用不可能領域）が発生しないか
// 確保と解放を繰り返した後で、フリーリストが壊れない、かつ利用不可能領域が発生しなければ、OK
TEST(ExpHeapBasicTest, Fragmentation)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;
    const int Seed = 12345;
    const int MaxRandomAllocateSize = 0x0100;
    std::mt19937 engine(Seed);
    std::uniform_int_distribution<uint32_t> distribution(0, MaxRandomAllocateSize - 1);

    NN_LOG("Seed : %d\n", Seed);

    heap = std::malloc(HeapSize);

    // 拡張ヒープの作成
    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);


    const int BlockNum = 10;
    void* p[BlockNum];

    // 各メモリアライメントで確保と解放を繰り返す
    for(int align = 4; align <= 128; align <<= 1)
    {
        NN_LOG("===%d===\n", align);
        for(int i = 0; i < BlockNum; i++)
        {
            p[i] = nn::lmem::AllocateFromExpHeap(heapHandle, distribution(engine), align);
        }
        nn::lmem::DumpExpHeap(heapHandle);
        for(int i = 0; i < BlockNum; i++)
        {
            nn::lmem::FreeToExpHeap(heapHandle, p[i]);
        }
        ASSERT_TRUE(nn::lmem::CheckExpHeap(heapHandle, nn::lmem::ErrorOption_NoOption));
    }

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// アロケートの方法を変更する
// アロケートの方法を First Fit と Best Fit に切り替え可能で、
// アロケート方法によって確保される場所もかわることがわかれば、OK
TEST(ExpHeapBasicTest, AllocationMode)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    void* heap;

    heap = std::malloc(HeapSize);

    // ヒープの作成
    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    // デフォルトは AllocationMode_FirstFit が指定されている
    EXPECT_EQ(nn::lmem::GetExpHeapAllocationMode(heapHandle), nn::lmem::AllocationMode_FirstFit);

    // わざと断片化したメモリを作成する
    // ブロック0: 256byte ブロック1～3: 128byte
    const size_t AllocateSize = 128;
    const int BlockNum = 4;
    void* pMemoryAddresses[BlockNum];
    pMemoryAddresses[0] = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize * 2);
    ASSERT_NOT_NULL(pMemoryAddresses[0]);
    for(int i = 1; i < BlockNum; ++i)
    {
        pMemoryAddresses[i] = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
        ASSERT_NOT_NULL(pMemoryAddresses[i]);
    }
    // ブロック0と2を解放する
    // 次に First Fit で 128byte の領域を確保する場合、 ブロック0 と同じアドレスで確保され、
    // Best Fit で 128byte の領域を確保する場合、 ブロック2 と同じアドレスで確保されるはず
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[0]);
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[2]);

    // First Fit で確保されているか
    void* pCandidate = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pCandidate);
    EXPECT_EQ(reinterpret_cast<uintptr_t>(pCandidate), reinterpret_cast<uintptr_t>(pMemoryAddresses[0]));
    nn::lmem::FreeToExpHeap(heapHandle, pCandidate);

    nn::lmem::SetExpHeapAllocationMode(heapHandle, nn::lmem::AllocationMode_BestFit);
    EXPECT_EQ(nn::lmem::GetExpHeapAllocationMode(heapHandle), nn::lmem::AllocationMode_BestFit);

    // Best Fit で確保されているか
    pCandidate = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pCandidate);
    EXPECT_EQ(reinterpret_cast<uintptr_t>(pCandidate), reinterpret_cast<uintptr_t>(pMemoryAddresses[2]));
    nn::lmem::FreeToExpHeap(heapHandle, pCandidate);

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[1]);
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[3]);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// メモリ確保時のアライメントで生じる断片化領域の再利用
// 断片化領域を再利用する場合、正しく再利用ができていれば、OK
TEST(ExpHeapBasicTest, UseMarginOfAlignment)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    // ヒープの作成
    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    // デフォルトは 断片化領域の再利用はしないように設定されている
    EXPECT_FALSE(nn::lmem::GetExpHeapUseMarginOfAlignment(heapHandle));

    // 大きいサイズの断片化領域が生まれるように確保アドレスを調整する
    const int MinAlign = 4;
    const int MaxAlign = 128;
    size_t paddingSize = 0;
    size_t memoryBlockHeadSize;
    void* pPaddingAddress;

    // メモリブロックのヘッダサイズを求める
    {
        void* pMemoryAddress1 = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize / 2, MinAlign);
        ASSERT_NOT_NULL(pMemoryAddress1);
        void* pMemoryAddress2 = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize / 2, MinAlign);
        ASSERT_NOT_NULL(pMemoryAddress2);
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress1);
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress2);
        memoryBlockHeadSize = reinterpret_cast<uintptr_t>(pMemoryAddress2) - reinterpret_cast<uintptr_t>(pMemoryAddress1) - AllocateSize / 2;
    }

    do
    {
        paddingSize++;
        pPaddingAddress = nn::lmem::AllocateFromExpHeap(heapHandle, paddingSize, MinAlign);
        ASSERT_NOT_NULL(pPaddingAddress);
        nn::lmem::FreeToExpHeap(heapHandle, pPaddingAddress);
    } while ((reinterpret_cast<uintptr_t>(pPaddingAddress) + paddingSize + memoryBlockHeadSize ) % 128 != 1);

    // 断片化領域を再利用しない場合
    {
        pPaddingAddress = nn::lmem::AllocateFromExpHeap(heapHandle, paddingSize, MinAlign);
        ASSERT_NOT_NULL(pPaddingAddress);
        void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, MaxAlign);
        ASSERT_NOT_NULL(pMemoryAddress);

        // この時点で、 pPaddingAddress と pMemoryAddress の間に再利用できる断片化領域が発生している

        void* pMemoryAddressForCheck = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize / 2, MinAlign);
        ASSERT_NOT_NULL(pMemoryAddressForCheck);
        // 断片化領域を再利用しない場合は、pMemoryAddress より後ろの領域に確保する
        EXPECT_TRUE(reinterpret_cast<uintptr_t>(pMemoryAddressForCheck) > reinterpret_cast<uintptr_t>(pMemoryAddress));
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddressForCheck);
        nn::lmem::FreeToExpHeap(heapHandle, pPaddingAddress);
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    }

    // 断片化領域を再利用する場合
    {
        nn::lmem::SetExpHeapUseMarginOfAlignment(heapHandle, true);
        pPaddingAddress = nn::lmem::AllocateFromExpHeap(heapHandle, paddingSize, MinAlign);
        ASSERT_NOT_NULL(pPaddingAddress);
        void* pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, MaxAlign);
        ASSERT_NOT_NULL(pMemoryAddress);

        void* pMemoryAddressForCheck = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize / 2, MinAlign);
        ASSERT_NOT_NULL(pMemoryAddressForCheck);
        // 断片化領域を再利用する場合は、pMemoryAddress と pPaddingAddress の間の領域に確保する
        EXPECT_TRUE(reinterpret_cast<uintptr_t>(pMemoryAddressForCheck) < reinterpret_cast<uintptr_t>(pMemoryAddress) &&
                    reinterpret_cast<uintptr_t>(pMemoryAddressForCheck) > reinterpret_cast<uintptr_t>(pPaddingAddress));
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddressForCheck);
        nn::lmem::FreeToExpHeap(heapHandle, pPaddingAddress);
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    }

    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// メモリブロックの確保失敗
// 現在のフリー領域より大きなメモリの確保に失敗すれば、OK
TEST(ExpHeapBasicTest, AllocationFailure)
{
    const int BlockNum = 64;
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = HeapSize / (BlockNum / 2);
    const int MinAlign = 4;
    void* heap;
    void* pMemoryAddresses[BlockNum];   // 少し余裕を持って配列を確保

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);

    // AllocateSize で確保できなくなるまで確保する
    int i = 0;
    while(nn::lmem::GetExpHeapAllocatableSize(heapHandle, MinAlign) > AllocateSize)
    {
        pMemoryAddresses[i] = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, MinAlign);
        ASSERT_NOT_NULL(pMemoryAddresses[i]);
        i++;
    }

    NN_LOG("Allocate %d times.\n", i);

    pMemoryAddresses[i] = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize, MinAlign);
    ASSERT_NULL(pMemoryAddresses[i]);
    i--;

    while(i >= 0)
    {
        nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddresses[i]);
        i--;
    }
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);

}

// ヒープ作成の繰り返し
// ヒープを同じアドレスで繰り返し呼んでも問題ないことを確認する
TEST(ExpHeapBasicTest, CreateHeapConstantly)
{
    int threadPriority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread());
    if(threadPriority < nn::os::LowestThreadPriority)
    {
        threadPriority++;
    }
    auto result = nn::os::CreateThread(&g_Thread, CreateExpHeapConstantly, NULL, g_ThreadStack, g_ThreadStackSize, threadPriority);
    ASSERT_TRUE(result.IsSuccess());

    nn::os::InitializeEvent(&g_Event, false, nn::os::EventClearMode_AutoClear);

    const int timeOut = 20;     // 20 秒でタイムアウトとする
    nn::os::StartThread(&g_Thread);

    ASSERT_TRUE(nn::os::TimedWaitEvent(&g_Event, nn::TimeSpan::FromSeconds(timeOut)));

    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
    nn::os::FinalizeEvent(&g_Event);
}

// ヒープの繰り返し前の確保状況のクリア
// Create 後に以前のヒープ状況がクリアされていることを確認する
TEST(ExpHeapBasicTest, CreateHeapClear)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    void* heap;

    heap = std::malloc(HeapSize);

    // ヒープの作成
    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);
    size_t allocatableSize = nn::lmem::GetExpHeapAllocatableSize(heapHandle, nn::lmem::DefaultAlignment);

    void* ptr = nn::lmem::AllocateFromExpHeap(heapHandle, 32);
    ASSERT_NOT_NULL(ptr);
    EXPECT_GT(allocatableSize, nn::lmem::GetExpHeapAllocatableSize(heapHandle, nn::lmem::DefaultAlignment));

    // Destroy せずに再度 Create
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_NoOption);
    ASSERT_NOT_NULL(heapHandle);
    // ヒープは初期化されているはず
    EXPECT_EQ(allocatableSize, nn::lmem::GetExpHeapAllocatableSize(heapHandle, nn::lmem::DefaultAlignment));

    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// =============
// テスト（Debug ビルド、Develop ビルドでのみ有効）
// =============

// メモリ確保時のオプションとメモリフィルは、Release ビルド以外の時のみ有効になる
#if defined(NN_TEST_MEM_DEBUG)

// ヒープ作成時のオプションの有効性テスト
// オプションで CreationOption_ZeroClear が指定されている場合、
// メモリ確保時に 0 フィルされていれば、OK
TEST(ExpHeapBasicTest, Option)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;
    void* pMemoryAddress;

    // メモリ確保時に 0 フィル、とヒープ確保時、メモリ確保時、メモリ解放時に指定値でフィルするオプションを設定
    // CreationOption_ZeroClear と CreationOption_DebugFill が両方呼ばれている場合、
    // メモリ確保時のフィルは CreationOption_ZeroClear が優先されます。
    int option = nn::lmem::CreationOption_ZeroClear | nn::lmem::CreationOption_DebugFill;

    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, option);
    ASSERT_NOT_NULL(heapHandle);

    pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);

    // 確保時のメモリが 0 フィルされているか
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, AllocateSize, 0));

    // Read / Write テスト
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, AllocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, AllocateSize));

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// ユーザが指定したメモリフィルの値が正しく適用されているか
// ヒープ作成、確保、解放でそれぞれ指定の値でメモリフィルされていれば、OK
TEST(ExpHeapBasicTest, DebugFill)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;
    const uint32_t fillUnallocated = 0x01234567;
    const uint32_t fillAllocate = 0x456789ab;
    const uint32_t fillFree  = 0x89abcdef;

    heap = std::malloc(HeapSize);

    nn::lmem::HeapHandle heapHandle;

    // DebugFill で埋める値を設定
    nn::lmem::SetFillValue(nn::lmem::FillType_Unallocated, fillUnallocated);
    nn::lmem::SetFillValue(nn::lmem::FillType_Allocate, fillAllocate);
    nn::lmem::SetFillValue(nn::lmem::FillType_Free, fillFree);

    // DebugFill で埋める値を確認
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Unallocated), fillUnallocated);
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Allocate), fillAllocate);
    EXPECT_EQ(nn::lmem::GetFillValue(nn::lmem::FillType_Free), fillFree);

    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_DebugFill);
    ASSERT_NOT_NULL(heapHandle);

    // DebugFill（未使用時）テスト
    // 開始アドレスをヘッダの領域の後ろに補正
    void* pMemoryAddress = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(heapHandle) + nn::lmem::GetTotalSize(heapHandle) - nn::lmem::GetExpHeapTotalFreeSize(heapHandle));
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, nn::lmem::GetExpHeapTotalFreeSize(heapHandle), fillUnallocated));

    // DebugFill（確保時）テスト
    pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, AllocateSize, fillAllocate));

    // DebugFill（解放時）テスト
    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, AllocateSize, fillFree));

    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

// CreationOption_ZeroClear を指定時に新たに確保したメモリが 0 フィルされているかどうか
// CreationOption_ZeroClear を指定した場合、リサイズ時の増加分のメモリも 0 フィルされる
// メモリ確保とリサイズで適切に 0 フィルがされていれば、OK
TEST(ExpHeapBasicTest, ZeroClear)
{
    const size_t HeapSize = nn::os::MemoryBlockUnitSize;
    const size_t AllocateSize = 128;
    void* heap;

    heap = std::malloc(HeapSize);

    void* pMemoryAddress;

    nn::lmem::HeapHandle heapHandle;
    heapHandle = nn::lmem::CreateExpHeap(reinterpret_cast<void*>(heap), HeapSize, nn::lmem::CreationOption_ZeroClear);
    ASSERT_NOT_NULL(heapHandle);

    pMemoryAddress = nn::lmem::AllocateFromExpHeap(heapHandle, AllocateSize);
    ASSERT_NOT_NULL(pMemoryAddress);

    // 確保時のメモリが 0 フィルされているか
    EXPECT_TRUE(HasEqualNum(pMemoryAddress, AllocateSize, 0));

    // Read / Write テスト
    // Write することにより 0 フィルを書き換える
    EXPECT_TRUE(WriteNumericalSequence(pMemoryAddress, AllocateSize));
    EXPECT_TRUE(HasNumericalSequence(pMemoryAddress, AllocateSize));

    // リサイズ後にメモリが 0 フィルされているか
    size_t size = nn::lmem::ResizeExpHeapBlock(heapHandle, pMemoryAddress, AllocateSize * 2);
    EXPECT_EQ(size, AllocateSize * 2);
    void* additionAddr = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(pMemoryAddress) + AllocateSize);
    EXPECT_TRUE(HasEqualNum(additionAddr, AllocateSize, 0));

    nn::lmem::FreeToExpHeap(heapHandle, pMemoryAddress);
    nn::lmem::DestroyExpHeap(heapHandle);
    std::free(heap);
}

#endif  // #if defined(NN_TEST_MEM_DEBUG)
