﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace jit { namespace testvm {

enum class InstructionKind : uint16_t
{
    End,
    Nop,     // ++pc
    Imm,     // Push(imm); ++pc
    Read,    // x = Read(imm); Push(x); ++pc
    Write,   // x = Pop(); Write(x, imm); ++pc
    Pop,     // Pop(); ++pc
    Dup,     // tmp = Pop(); Push(tmp); Push(tmp); ++pc
    Swap,    // x = Pop(); y = Pop(); Push(x); Push(y); ++pc
    Add,     // x = Pop(); y = Pop(); Push(x + y); ++pc
    Mul,     // x = Pop(); y = Pop(); Push(x * y); ++pc
    Neg,     // x = Pop(); Push(-x); ++pc
    Not,     // x = Pop(); Push((x == 0) ? 1 : 0); ++pc
    JmpR,    // pc += Pop()
    JmpRIf0, // d = Pop(); tmp = Pop(); pc += (tmp == 0) ? ds : 1
    Call,    // d = Pop(); Push(pc + 1); pc = d
    Ret,     // newPc = Pop(); pc = newPc

    Label,
    PushL,
};

struct Instruction
{
    InstructionKind kind;
    int16_t imm;

    Instruction() NN_NOEXCEPT
    {
    }

    NN_IMPLICIT Instruction(InstructionKind kind_, int16_t x = 0) NN_NOEXCEPT
        : kind(kind_)
        , imm(x)
    {
    }

    NN_IMPLICIT Instruction(int16_t imm_) NN_NOEXCEPT
        : kind(InstructionKind::Imm)
        , imm(imm_)
    {
    }

    static Instruction Read(int16_t n) NN_NOEXCEPT
    {
        return {InstructionKind::Read, n};
    }

    static Instruction Write(int16_t n) NN_NOEXCEPT
    {
        return {InstructionKind::Write, n};
    }

    static Instruction Label(int16_t name) NN_NOEXCEPT
    {
        return {InstructionKind::Label, name};
    }

    static Instruction PushLabel(int16_t name) NN_NOEXCEPT
    {
        return {InstructionKind::PushL, name};
    }
};

struct MachineState
{
    int32_t* stack;
    uint32_t stackCountMax;
    uint32_t stackCount;
};

struct Program
{
    Instruction* instructions;
    uint32_t instructionCount;
};

struct MachineContext
{
    int32_t pc;
    MachineState state;
    Program program;
};

void ResolveLabel(Program* pProgram) NN_NOEXCEPT;

typedef void (*RunTestVm)(MachineContext*);

}}}
