﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/os.h>
#include <nn/init.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Tcb.h>
#include <nn/svc/svc_Result.h>

#include "../Common/Utility.h"

namespace nn { namespace {

uint64_t MapProcessCodeMemory(svc::Handle processHandle, uint64_t bufferAddr, uint64_t codeSize) NN_NOEXCEPT
{
    jit::AslrAllocator aslrAllocator(processHandle);
retry:
    auto toAddr = aslrAllocator.GetAslrRegion(codeSize);
    /*
        svc::MapProcessCodeMemory
        - 疑似ハンドルは渡せない → DuplicateHandle で本ハンドルに修正 (実装は内部で自己 IPC)
        - 本番では別プロセスになるため、問題ないはず
        - fromAddr は MemoryState_Normal を要求するため BSS 領域は指定できない → ヒープから確保
    */
    NN_RESULT_TRY(svc::MapProcessCodeMemory(processHandle, toAddr, bufferAddr, codeSize))
        NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
        {
            goto retry;
        }
    NN_RESULT_END_TRY
    return toAddr;
}

void UnmapProcessCodeMemory(svc::Handle processHandle, uint64_t codeAddress, uint64_t bufferAddr, uint64_t codeSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapProcessCodeMemory(processHandle, codeAddress, bufferAddr, codeSize));
}

uintptr_t MapProcessMemory(svc::Handle processHandle, uint64_t codeAddr, uint64_t codeSize) NN_NOEXCEPT
{
    jit::AslrAllocator aslrAllocator(processHandle);
retry:
    auto toAddr = aslrAllocator.GetAslrRegion(codeSize);
    NN_RESULT_TRY(svc::MapProcessMemory(toAddr, processHandle, codeAddr, codeSize))
        NN_RESULT_CATCH(svc::ResultInvalidCurrentMemory)
        {
            goto retry;
        }
    NN_RESULT_END_TRY
    return toAddr;
}

void UnmapProcessMemory(uintptr_t toAddr, svc::Handle processHandle, uint64_t codeAddr, uint64_t codeSize) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(svc::UnmapProcessMemory(toAddr, processHandle, codeAddr, codeSize));
}

int g_Int;

void G(void* p)
{
    *static_cast<int*>(p) = 100;
}

void F(void (*g)(void*), void* p)
{
    g(p);
}

template <typename F>
void IterateCacheLine(uintptr_t addr, size_t size, F f) NN_NOEXCEPT
{
    uintptr_t cacheTypeRegister;
    asm volatile ("mrs %0, ctr_el0" : "=r" (cacheTypeRegister));
    size_t cacheLineSize = 4 << ((cacheTypeRegister >> 16) & 0xF);
    auto startAddr = reinterpret_cast<uintptr_t>(addr) & ~(cacheLineSize - 1);
    auto endAddr = reinterpret_cast<uintptr_t>(addr) + size;
    for (auto p = startAddr; p < endAddr; p += cacheLineSize)
    {
        f(p);
    }
}

void Main() NN_NOEXCEPT
{
    auto processHandle = jit::DuplicateHandle(svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    NN_UTIL_SCOPE_EXIT
    {
        svc::CloseHandle(processHandle);
    };

    auto codeSize = 1024 * 1024;
    auto buffer = nn::init::GetAllocator()->Allocate(codeSize, 4096);
    NN_UTIL_SCOPE_EXIT
    {
        nn::init::GetAllocator()->Free(buffer);
    };
    auto bufferAddr = reinterpret_cast<uintptr_t>(buffer);

    auto codeAddr = MapProcessCodeMemory(processHandle, bufferAddr, codeSize);
    NN_UTIL_SCOPE_EXIT
    {
        UnmapProcessCodeMemory(processHandle, codeAddr, bufferAddr, codeSize);
    };

    auto mappedCodeAddr = MapProcessMemory(processHandle, codeAddr, codeSize);
    NN_UTIL_SCOPE_EXIT
    {
        UnmapProcessMemory(mappedCodeAddr, processHandle, codeAddr, codeSize);
    };
    std::memset(reinterpret_cast<void*>(mappedCodeAddr), 0, codeSize);

    NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(processHandle, codeAddr, codeSize, svc::MemoryPermission_ReadExecute));
    NN_UTIL_SCOPE_EXIT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(processHandle, codeAddr, codeSize, svc::MemoryPermission_None));
    };

    for (int i = 0; i < codeSize / 4096; ++i)
    {
        /*
            書き換え範囲に対し MemoryPermission を一旦 None に落としている。
            これは、命令キャッシュの invalidate のため。
            専用の命令キャッシュ invalidate API があれば、そちらを使用すれば十分。
        */
        auto fOffset = 4096 * i;
        auto fSize = 4096;

        // 関数 F のコードをコピーして動的生成コードとしている。サイズが 4096B に収まっている仮定。
        std::memcpy(reinterpret_cast<void*>(mappedCodeAddr + fOffset), reinterpret_cast<void*>(&F), fSize);

        // 書き換え側
        //データキャッシュ store
        IterateCacheLine(reinterpret_cast<uintptr_t>(mappedCodeAddr + fOffset), fSize, [](uintptr_t p)
        {
            asm volatile ("dc cvau, %0"::"r"(p):"memory");
        });
        // ish メモリバリア
        asm volatile ("dsb ish":::"memory");

        // 実行側
        // 命令キャッシュ無効化
        IterateCacheLine(mappedCodeAddr + fOffset, fSize, [](uintptr_t p)
        {
            asm volatile ("ic ivau, %0"::"r"(p):"memory");
        });
        // ish メモリバリア
        asm volatile ("dsb ish":::"memory");
        // パイプラインフラッシュ
        asm volatile ("isb":::"memory");

        auto genF = reinterpret_cast<decltype(&F)>(codeAddr + fOffset);

        NN_ABORT_UNLESS_RESULT_SUCCESS(svc::SetProcessMemoryPermission(processHandle, codeAddr + fOffset, fSize, svc::MemoryPermission_ReadExecute));

        // 動的生成コードから別モジュールのシンボルは参照できないため、
        // メインモジュールの &G や &g_Int を参照するには、明示的に渡す必要がある。
        g_Int = 0;
        genF(G, &g_Int);
        NN_SDK_ASSERT(g_Int == 100);
    }
}

}}

extern "C" void nnMain()
{
    nn::Main();
}
