﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MomentModeState.h"

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

MomentModeState::MomentModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle)
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_MomentProcessorState()
    , m_MomentProcessorConfig()
    , m_ViewMode(ViewMode::ViewMode_Black)
{
    nn::irsensor::GetMomentProcessorDefaultConfig(&m_MomentProcessorConfig);

    nn::irsensor::MomentProcessorConfig* pMomentConfig = &m_MomentProcessorConfig;

    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pMomentConfig->irCameraConfig,
        nn::irsensor::MomentProcessorExposureTimeMin,
        nn::irsensor::MomentProcessorExposureTimeMax
    );

    m_ReadWriteMenu.emplace_back("Preprocess",
        [pMomentConfig](std::stringstream& sstr) {
            sstr << (pMomentConfig->preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize ?
                "Binarize" :
                "Cutoff");
        },
        [pMomentConfig](int8_t delta) {
            NN_UNUSED(delta);
            pMomentConfig->preprocess =
                (pMomentConfig->preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize ?
                    nn::irsensor::MomentProcessorPreprocess_Cutoff :
                    nn::irsensor::MomentProcessorPreprocess_Binarize);
        }
    );
    m_ReadWriteMenu.emplace_back("Intensity Threshold",
        [pMomentConfig](std::stringstream& sstr) {
            sstr << pMomentConfig->preprocessIntensityThreshold;
        },
        [pMomentConfig](int8_t delta) {
            pMomentConfig->preprocessIntensityThreshold += delta * 5;
            pMomentConfig->preprocessIntensityThreshold =
                std::min(std::max(pMomentConfig->preprocessIntensityThreshold, 0), nn::irsensor::IrCameraIntensityMax);
        }
    );

    AddCommonReadOnlyMenu(&m_ReadOnlyMenu, &m_MomentProcessorState.samplingNumber, &m_MomentProcessorState.ambientNoiseLevel);
    AddStatisticsMenu(&m_ReadOnlyMenu, &m_Statistics.m_PacketDropPercentage);
}

void MomentModeState::Start()
{
    if (m_ViewMode == ViewMode_Black)
    {
        m_MomentProcessorConfig.irCameraConfig.exposureTime = nn::TimeSpan::FromMicroSeconds(500);
        m_MomentProcessorConfig.irCameraConfig.gain = 1;
        m_MomentProcessorConfig.irCameraConfig.lightTarget = nn::irsensor::IrCameraLightTarget_AllObjects;
    }
    else if (m_ViewMode == ViewMode_Red)
    {
        m_MomentProcessorConfig.irCameraConfig.exposureTime = nn::TimeSpan::FromMicroSeconds(115);
        m_MomentProcessorConfig.irCameraConfig.gain = 1;
        m_MomentProcessorConfig.irCameraConfig.lightTarget = nn::irsensor::IrCameraLightTarget_AllObjects;
    }
    m_MomentProcessorConfig.windowOfInterest.x = 140;
    m_MomentProcessorConfig.windowOfInterest.y = 105;
    m_MomentProcessorConfig.windowOfInterest.width = 40;
    m_MomentProcessorConfig.windowOfInterest.height = 30;

    nn::irsensor::RunMomentProcessor(m_IrCameraHandle, m_MomentProcessorConfig);
}

void MomentModeState::Update()
{
    int count;
    nn::Result result = nn::irsensor::GetMomentProcessorStates(&m_MomentProcessorState, &count, 1, m_IrCameraHandle);
    HandleResult(result);
    m_Statistics.Update(&m_MomentProcessorState.samplingNumber, IrSensorModeStatistics::ExpectedMomentModeFramerate);

    nn::irsensor::MomentStatistic statistic;
    statistic = nn::irsensor::CalculateMomentRegionStatistic(&m_MomentProcessorState, m_MomentProcessorConfig.windowOfInterest, 0, 0, nn::irsensor::MomentProcessorBlockRowCount, nn::irsensor::MomentProcessorBlockColumnCount);

    auto brightestValue = 0.0f;
    for (auto i = 0; i < nn::irsensor::MomentProcessorBlockColumnCount * nn::irsensor::MomentProcessorBlockRowCount; i++)
    {
        if (brightestValue < m_MomentProcessorState.blocks[i].averageIntensity)
        {
            brightestValue = m_MomentProcessorState.blocks[i].averageIntensity;
        }
    }
    m_MomentResult.Update(statistic.averageIntensity, brightestValue);
}


namespace
{

void WriteMomentProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::MomentProcessorState* pMomentState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pMomentState);
    NN_UNUSED(screenIndex);
#if 0
    const nn::util::Uint8x4 White ={ { 255, 255, 255, 255 } };
    const nn::util::Uint8x4 Red ={ { 255, 0, 0, 255 } };

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // Disable Depth Disable
    pPrimitiveRenderer->SetDepthStencilState(pCommandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest);

    const float xStart = -0.8f, xEnd = -0.2f;
    float yStart, yEnd;
    if (screenIndex == 0)
        yStart = 0.1f, yEnd = 0.9f;
    else
        yStart = -0.9f, yEnd = -0.1f;

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    size_t idx = 0;
    for (size_t y = 0; y < nn::irsensor::MomentProcessorBlockRowCount; ++y)
    {
        for (size_t x = 0; x < nn::irsensor::MomentProcessorBlockColumnCount; ++x)
        {
            nn::util::Vector3fType corner;
            uint8_t averageIntensity = static_cast<uint8_t>(pMomentState->blocks[idx].averageIntensity);

            // 各面を描画
            nn::util::VectorSet(&corner,
                xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount),
                yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount), 0.0f);

            nn::util::Vector3fType size;
            nn::util::VectorSet(&size,
                (xEnd - xStart) / static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount),
                (yEnd - yStart) / static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount), 0.0f);

            nn::util::Uint8x4 gridColor ={ { averageIntensity, averageIntensity, averageIntensity, averageIntensity } };
            pPrimitiveRenderer->SetColor(gridColor);
            pPrimitiveRenderer->DrawQuad(
                pCommandBuffer,
                corner,
                size);

            // 重心点を描画
            nn::util::VectorSet(&corner,
                xStart + (xEnd - xStart) * pMomentState->blocks[idx].centroid.x / static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * pMomentState->blocks[idx].centroid.y / static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            pPrimitiveRenderer->SetColor(Red);
            pPrimitiveRenderer->DrawSphere(
                pCommandBuffer,
                nns::gfx::PrimitiveRenderer::Surface_Solid,
                nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                corner,
                4.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

            idx++;
        }
    }

    // グリッドを描画
    pPrimitiveRenderer->SetLineWidth(0.0f);

    for (size_t x = 0; x <= nn::irsensor::MomentProcessorBlockColumnCount; ++x)
    {
        float xValue = xStart + (xEnd - xStart) * x / static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount);
        nn::util::VectorSet(&begin, xValue, yStart, 0.0f);
        nn::util::VectorSet(&end, xValue, yEnd, 0.0f);
        pPrimitiveRenderer->SetColor(White);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
    for (size_t y = 0; y <= nn::irsensor::MomentProcessorBlockRowCount; ++y)
    {
        float yValue = yStart + (yEnd - yStart) * y / static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount);
        nn::util::VectorSet(&begin, xStart, yValue, 0.0f);
        nn::util::VectorSet(&end, xEnd, yValue, 0.0f);
        pPrimitiveRenderer->SetColor(White);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
#endif
}

}

void MomentModeState::Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index)
{
    WriteMomentProcessorState(pPrimitiveRenderer, pCommandBuffer, &m_MomentProcessorState, index);
}

void MomentModeState::Record(const RecordInfo& recordInfo, const PathInfo& pathInfo, int clusterCountMax, GraphicsSystem* pGraphicsSystem)
{
    NN_UNUSED(recordInfo);
    NN_UNUSED(pathInfo);
    NN_UNUSED(clusterCountMax);
    NN_UNUSED(pGraphicsSystem);
}

void MomentModeState::Reset()
{

}

void MomentModeState::SetViewMode(ViewMode viewMode)
{
    m_ViewMode = viewMode;
}


