﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/irsensor.h>
#include <nnt.h>
#include "../Common/testIrsensor_Util.h"

using namespace ::nn::irsensor;

namespace nnt { namespace irsensor {
    IrCameraHandle IrSensorTest::s_Handles[NpadIdCountMax];
    int IrSensorTest::s_HandleCount;
}}

namespace
{
// エージングテストの試行回数
const int AgingTestCount = 100;
// データ取得の試行回数
const int GetStateCount = 100;
// コマンド呼び出しの定常インターバル
const auto CommandInterval = nn::TimeSpan::FromMilliSeconds(15);
// 起動直後のインターバル
const auto InitialInterval = nn::TimeSpan::FromSeconds(2);


class AgingTest : public ::nnt::irsensor::IrSensorTest, public ::testing::WithParamInterface<int>
{
};

INSTANTIATE_TEST_CASE_P(
    MomentTransition,
    AgingTest,
    ::testing::Values(AgingTestCount));

TEST_P(AgingTest, MomentTransition)
{
    auto testCount = GetParam();
    nn::os::SleepThread(InitialInterval);

    // 少なくとも1台1以上のIR センサー使用可能なデバイスが接続されたかどうか。
    bool isConnected = false;

    for (auto i = 0; i < testCount; ++i)
    {
        for (auto j = 0; j < s_HandleCount; ++j)
        {
            const auto& handle = s_Handles[j];

            auto status = GetIrCameraStatus(handle);
            if (status == IrCameraStatus_Available)
            {
                isConnected = true;
                MomentProcessorConfig config;
                GetMomentProcessorDefaultConfig(&config);

                RunMomentProcessor(handle, config);

                for (auto k = 0; k < GetStateCount; ++k)
                {
                    // 15ms のインターバル
                    nn::os::SleepThread(CommandInterval);

                    MomentProcessorState state;
                    nn::Result result = GetMomentProcessorState(&state, handle);
                    if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
                    {
                        // 少なくとも GetStateCount 回目のデータは取得できていることを確認する
                        EXPECT_NE(GetStateCount - 1, k);
                        continue;
                    }
                    NNT_EXPECT_RESULT_SUCCESS(result);
                    EXPECT_LE(0, state.samplingNumber);
                    //NN_LOG("    [GetStateLoop:%d] samplingCount:%d\n", k, state.samplingNumber);
                }

                StopImageProcessor(handle);

                nn::os::SleepThread(CommandInterval);
            }
        }
        NN_LOG("[Loop:%d] MomentAging\n", i);
    }
    // 1台も接続がなかった場合は失敗
    ASSERT_TRUE(isConnected);
}

} // namespace
