﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  nninitSetMallocRegion 動作確認テスト
//-----------------------------------------------------------------------------

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_SdkText.h>
#include <nn/TargetConfigs/build_Compiler.h>

#if defined(NN_BUILD_CONFIG_COMPILER_VC)
    // 日本語以外の環境で表示される文字コードエンコーディングに関する警告の抑制
    #pragma warning( disable : 4566 )
#endif

#include <cstdlib>
#include <cstring>
#include <malloc.h>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_COMPILER_GCC)
extern "C" void* aligned_alloc(size_t alignment, size_t size);
#endif

namespace nnt { namespace init { namespace startup {

namespace {
    void* g_MallocBegin;
    void* g_MallocEnd;
}

#define NNT_ABORT_IF_NOT_ALIGNED32(value)  \
        NN_ABORT_UNLESS((static_cast<uintptr_t>(value) & (sizeof(uint32_t) - 1)) == 0)

//-----------------------------------------------------------------------------

extern "C" void nninitStartup()
{
    const size_t MallocRegionSize = 4 * 1024 * 1024;

    nn::os::SetMemoryHeapSize( 32 * 1024 * 1024 );

    uintptr_t address;
    auto result = nn::os::AllocateMemoryBlock( &address, MallocRegionSize );
    EXPECT_TRUE( result.IsSuccess() );

    // malloc 用の領域を確保
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MallocRegionSize );
    g_MallocBegin = reinterpret_cast<void*>(address);
    g_MallocEnd   = reinterpret_cast<void*>(address + MallocRegionSize);

    NN_LOG("Malloc region begin = 0x%p\n", g_MallocBegin);
    NN_LOG("Malloc region end   = 0x%p\n", g_MallocEnd);
}

#if defined(NN_BUILD_CONFIG_COMPILER_GCC) || \
    defined(NN_BUILD_CONFIG_COMPILER_CLANG)
//-----------------------------------------------------------------------------
//  start ～ size 分の領域が data8 の値で埋まっていることをチェックする。
//  start および size は 32bit にアライメントされたものを渡すべし。
//
bool CheckMemoryRegion32(void* start, size_t size, uint8_t data8)
{
    NNT_ABORT_IF_NOT_ALIGNED32(reinterpret_cast<uintptr_t>(start));
    NNT_ABORT_IF_NOT_ALIGNED32(size);

    uint32_t data32 = (static_cast<uint32_t>(data8) << 24) |
                      (static_cast<uint32_t>(data8) << 16) |
                      (static_cast<uint32_t>(data8) <<  8) |
                      (static_cast<uint32_t>(data8) <<  0);

    uint32_t* address    = reinterpret_cast<uint32_t*>(start);
    uint32_t* endAddress = address + (size / sizeof(uint32_t));

    while (address < endAddress)
    {
        if (*address != data32)
        {
            return false;
        }
        ++address;
    }
    return true;
}

//-----------------------------------------------------------------------------
// malloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testMalloc)
{
    const size_t mallocSize = 5000;

    NN_LOG(NN_TEXT("malloc の呼出し\n"));
    void* address = malloc( mallocSize );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("malloc の獲得領域を 0x55 で埋めておく\n"));
    std::memset(address, 0x55, mallocSize);

    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// aligned_alloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testAlignedAlloc)
{
    const size_t alignedAllocSize = 5000;
    const size_t alignedAllocAlignment = 128;

    NN_LOG(NN_TEXT("aligned_alloc の呼出し\n"));
    void* address = aligned_alloc( alignedAllocAlignment, alignedAllocSize );
    EXPECT_TRUE( address != NULL );
    EXPECT_TRUE( (reinterpret_cast<uintptr_t>(address) % alignedAllocAlignment) == 0 );

    NN_LOG(NN_TEXT("aligned_alloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("aligned_alloc の獲得領域を 0xAA で埋めておく\n"));
    std::memset(address, 0xAA, alignedAllocSize);

    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// calloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testCalloc)
{
    const size_t callocSize = 5000;

    NN_LOG(NN_TEXT("calloc の呼出し\n"));
    void* address = calloc( callocSize / 4, 4 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("calloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("calloc の獲得領域が 0 初期化されているか\n"));
    EXPECT_TRUE( CheckMemoryRegion32(address, callocSize, 0) );

    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// realloc の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testRealloc)
{
    const size_t reallocSize = 5000;

    NN_LOG(NN_TEXT("malloc の呼出し\n"));
    void* address = malloc( reallocSize );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("malloc の獲得領域を 0x33 で埋める\n"));
    memset(address, 0x33, reallocSize);


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを小さくする）\n"));
    address = realloc( address, reallocSize / 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("realloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("realloc の新しい領域が 0x33 で埋まっているか\n"));
    NN_LOG("address[0]=0x%08x\n", *reinterpret_cast<uint32_t*>(address));
    EXPECT_TRUE( CheckMemoryRegion32(address, reallocSize / 2, 0x33) );


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを大きくする）\n"));
    address = realloc( address, reallocSize * 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("realloc の獲得アドレスが正しいか address=0x%p\n"), address);
    EXPECT_TRUE( g_MallocBegin <= address && address < g_MallocEnd );

    NN_LOG(NN_TEXT("realloc の少なくとも前半部分が 0x33 で埋まっているか\n"));
    NN_LOG("address[0]=0x%08x\n", *reinterpret_cast<uint32_t*>(address));
    EXPECT_TRUE( CheckMemoryRegion32(address, reallocSize / 2, 0x33) );


    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}

//-----------------------------------------------------------------------------
// malloc_usable_size の呼出し、確保した領域のチェック、free の呼出し
//
TEST(nninitSetMallocRegionTest, testMallocUsableSize)
{
    const size_t mallocSize = 5000;

    NN_LOG(NN_TEXT("malloc の呼出し\n"));
    void* address = malloc( mallocSize );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc_usable_size の呼出し\n"));
    size_t usableSize = malloc_usable_size( address );

    NN_LOG(NN_TEXT("malloc_usable_size の獲得サイズが正しいか usableSize=%zu\n"), usableSize);
    EXPECT_TRUE( usableSize >= mallocSize );


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを大きくする）\n"));
    address = realloc( address, mallocSize * 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc_usable_size の呼出し\n"));
    usableSize = malloc_usable_size( address );

    NN_LOG(NN_TEXT("malloc_usable_size の獲得サイズが正しいか usableSize=%zu\n"), usableSize);
    EXPECT_TRUE( usableSize >= mallocSize * 2 );


    NN_LOG(NN_TEXT("realloc の呼出し（サイズを小さくする）\n"));
    address = realloc( address, mallocSize / 2 );
    EXPECT_TRUE( address != NULL );

    NN_LOG(NN_TEXT("malloc_usable_size の呼出し\n"));
    usableSize = malloc_usable_size( address );

    NN_LOG(NN_TEXT("malloc_usable_size の獲得サイズが正しいか usableSize=%zu\n"), usableSize);
    EXPECT_TRUE( usableSize >= mallocSize / 2 );


    NN_LOG(NN_TEXT("free の呼出し\n"));
    free( address );
    EXPECT_TRUE( true );
}


//-----------------------------------------------------------------------------
// new の呼出し
//
TEST(nninitSetMallocRegionTest, testNew)
{
    NN_LOG(NN_TEXT("new の呼出し\n"));
    uint32_t* p = new uint32_t[1024 * 1024 / sizeof(uint32_t)];

    NN_LOG(NN_TEXT("new による獲得アドレスが正しいか p=0x%p\n"), p);
    EXPECT_TRUE( g_MallocBegin <= p && p < g_MallocEnd );

    delete[] p;
}

//-----------------------------------------------------------------------------
// GCC/Clang 時の GetAllocatorTotalFreeSize() と GetAllocatorAllocatableSize() の呼び出し
//
TEST(nninitSetMallocRegionTest, testGetSizeGccClang)
{
    nn::mem::StandardAllocator allocator;
    nn::mem::StandardAllocator* pInitAllocator;

    // nninitStartup() で指定しているサイズと同サイズの StandardAllocator を作成
    const size_t MallocRegionSize = 4 * 1024 * 1024;
    uintptr_t address;
    auto result = nn::os::AllocateMemoryBlock(&address, MallocRegionSize);
    ASSERT_TRUE(result.IsSuccess());

    allocator.Initialize(reinterpret_cast<void*>(address), MallocRegionSize);

    const int repeat = 128;
    void* pAllocatedMalloc[repeat] = { nullptr };
    void* pAllocatedSa[repeat] = { nullptr };
    pInitAllocator = nn::init::GetAllocator();

    size_t prevAllocatableSize = pInitAllocator->GetAllocatableSize();
    size_t prevFreeSize = pInitAllocator->GetTotalFreeSize();

    // init のアロケータと StandardAllocator で挙動が同じことを確認
    for (int i = 0; i < repeat; ++i)
    {
        size_t allocSize = (i + 1) * 16;
        if (i % 2)
        {
            pAllocatedMalloc[i] = malloc(allocSize);
        }
        else
        {
            pAllocatedMalloc[i] = pInitAllocator->Allocate(allocSize);
        }
        pAllocatedSa[i] = allocator.Allocate(allocSize);
        ASSERT_TRUE(pAllocatedMalloc[i] != NULL);
        ASSERT_TRUE(pAllocatedSa[i] != NULL);
        ASSERT_EQ(malloc_usable_size(pAllocatedMalloc[i]), allocator.GetSizeOf(pAllocatedSa[i]));
        ASSERT_EQ(pInitAllocator->GetSizeOf(pAllocatedMalloc[i]), allocator.GetSizeOf(pAllocatedSa[i]));
        ASSERT_GE(prevAllocatableSize, pInitAllocator->GetAllocatableSize());
        ASSERT_GE(prevFreeSize, pInitAllocator->GetTotalFreeSize());
        prevAllocatableSize = pInitAllocator->GetAllocatableSize();
        prevFreeSize = pInitAllocator->GetTotalFreeSize();
    }
}

#elif defined(NN_BUILD_CONFIG_COMPILER_VC)
//-----------------------------------------------------------------------------
// Windows 時の GetAllocatorTotalFreeSize() と GetAllocatorAllocatableSize() の呼び出し
//
TEST(nninitSetMallocRegionTest, testGetSizeWindows)
{
    NN_LOG(NN_TEXT("Windows 時に GetAllocator() で NULL が返るか\n"));
    ASSERT_TRUE(nn::init::GetAllocator() == NULL);
}

//-----------------------------------------------------------------------------
#endif  // GCC or CLANG

}}} // namespace nnt::init::startup

