﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_ConditionVariable.h>

namespace nnt { namespace sfutil {

template <typename T>
class CounterSynchronizer
{
private:

    T m_Value;
    bool m_Synchronized;
    mutable nn::os::Mutex m_Mutex;
    mutable nn::os::ConditionVariable m_Condition;

public:

    explicit CounterSynchronizer(T value = T()) NN_NOEXCEPT
        : m_Value(value)
        , m_Synchronized(true)
        , m_Mutex(false)
    {
    }

    void Reset(T value = T()) NN_NOEXCEPT
    {
        this->m_Value = value;
    }

    void Synchronize(T value) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        while (!(m_Value == value))
        {
            m_Condition.Wait(m_Mutex);
        }
        this->m_Synchronized = true;
        m_Condition.Broadcast();
    }

    void Set(T value) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(m_Synchronized);
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        this->m_Synchronized = false;
        this->m_Value = value;
        m_Condition.Broadcast();
        while (!m_Synchronized)
        {
            m_Condition.Wait(m_Mutex);
        }
    }

};

}}
