﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{RenderThreadPool.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This file defines a class that handles running
 *  multiple threads in parallel that create command
 *  buffers to render objects from a work queue.
 */

#pragma once

#include <nn/os/os_Thread.h>
#include <nn/os/os_Mutex.h>
#include <vector>
#include <nvn\nvn.h>
#include "RingBufferManagers.h"
#include "MemoryPool.h"

class GraphicsObject;
class RenderThreadPool;

/*
 * CommandBufferData
 * -----------------
 * Data needed for each thread for each command
 * buffer.
 */
struct CommandBufferData
{
    NVNcommandBuffer           m_CommandBuffer;
    MemoryPool*                m_MemoryPool;
    void*                      m_pControlPool;
    CompletionTracker*         m_pCompletionTracker;
    CommandBufferMemoryManager m_MemoryManager;
};

/*
* TutorialThreadState
* ----------
* Denotes the state that each thread is in
* and what action it should take.
*/
enum TutorialThreadState
{
    TUTORIAL_THREAD_STATE_RUN  = 0,
    TUTORIAL_THREAD_STATE_WAIT = 1,
    TUTORIAL_THREAD_STATE_STOP = 2,
    TUTORIAL_THREAD_STATE_NUM_STATES,
};

/*
* ThreadData
* ----------
* Command buffer data and a pointer to the
* parent thread pool for each thread.
*/
struct ThreadData
{
    RenderThreadPool*   m_pParentPool;
    CommandBufferData*  m_pCommandBufferData;
    TutorialThreadState m_ThreadState;
    nn::os::MutexType*  m_ThreadStateCheckMutex;
};

class RenderThreadPool
{
    public:
        RenderThreadPool();
        ~RenderThreadPool();

        void Init(int numThreads, NVNdevice* pDevice);
        void Clean();
        void Run(std::vector<NVNcommandHandle>* pOutput, std::vector<GraphicsObject*>* pWork, NVNqueue* pQueue);
        void Wait();
        void Stop();

        std::pair<int, GraphicsObject*> GetNextWorkItem();
        void WriteThreadOutput(int index, const NVNcommandHandle& handle);

    private:
        std::vector<nn::os::ThreadType>         m_ThreadHandles;
        std::vector<GraphicsObject*>*           m_pWorkQueue;
        std::vector<NVNcommandHandle>*          m_pOutput;
        std::vector<CommandBufferData>          m_CommandBuffers;
        int                                     m_WorkIndex;
        nn::os::MutexType                       m_WorkMutex;
        nn::os::MutexType                       m_WriteMutex;
        std::vector<nn::os::MutexType>          m_ThreadStateMutex;
        char*                                   m_pThreadStackMemory;
        bool                                    m_Running;
        std::vector<ThreadData>                 m_ThreadData;
};
