﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdio>
#include <vector>
#include <string>
#include <future>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nnt/ncmutil/ncmutil_InstallUtils.h>
#include <nnt/result/testResult_Assert.h>

namespace nnt { namespace nsutil {

class ApplicationInstaller : private nnt::ncmutil::SubmissionPackageFileInstaller
{
public:
    using SubmissionPackageFileInstaller::SetupBisWorkingDirectory;
    using SubmissionPackageFileInstaller::TearDownBisWorkingDirectory;

    std::vector<nn::ncm::ApplicationId> GetApplicationIdList(const std::vector<nn::ncm::ApplicationContentMetaKey>& contentMetaList) NN_NOEXCEPT
    {
        std::vector<nn::ncm::ApplicationId> idList;
        for (const auto& appKey : contentMetaList)
        {
            bool foundAlready = false;
            for (auto id : idList)
            {
                if (id.value == appKey.applicationId.value)
                {
                    foundAlready = true;
                    break;
                }
            }

            if (!foundAlready)
            {
                idList.push_back(appKey.applicationId);
            }
        }

        return idList;
    }

    nn::Result InstallAll(std::vector<nn::ncm::ApplicationContentMetaKey>* outInstalled, const char* directoryPath, nn::ncm::StorageId storage) NN_NOEXCEPT
    {
        nn::fs::DirectoryHandle dir;
        NN_RESULT_DO(nn::fs::OpenDirectory(&dir, directoryPath, nn::fs::OpenDirectoryMode_File));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(dir); };

        std::vector<nn::ncm::ApplicationContentMetaKey> installed;
        while(NN_STATIC_CONDITION(true))
        {
            int64_t readCount;
            nn::fs::DirectoryEntry entry;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readCount, &entry, dir, 1));
            if(readCount == 0)
            {
                break;
            }

            auto name = std::string(entry.name);
            auto ext = std::string(".nsp");
            if(name.substr(name.size() - ext.size()) == ext)
            {
                auto nspPath = std::string(directoryPath) + "/" + name;

                std::vector<nn::ncm::ApplicationContentMetaKey> installedOne;
                NN_RESULT_DO(InstallOne(&installedOne, nspPath.c_str(), storage));
                std::copy(installedOne.begin(),installedOne.end(),std::back_inserter(installed));
            }
        }

        *outInstalled = installed;

        NN_RESULT_SUCCESS;
    }

    nn::Result InstallOne(std::vector<nn::ncm::ApplicationContentMetaKey>* outInstalled, const char* filePath, nn::ncm::StorageId storage) NN_NOEXCEPT
    {
        NN_LOG("Install %s.\n", filePath);

        std::vector<nn::ncm::ApplicationContentMetaKey> installed;

        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

        static char g_Buffer[16 * 1024 * 1024];
        nn::ns::ApplicationInstallTask task;
        NN_RESULT_DO(task.Initialize(file, storage, g_Buffer, sizeof(g_Buffer)));
        NN_RESULT_DO(task.Prepare());
        auto progress = task.GetProgress();
        NN_LOG("Prepared. progress.installedSize %lld, progress.totalSize %lld\n", progress.installedSize, progress.totalSize);

        int offset = 0;
        for (;;)
        {
            nn::ncm::ApplicationContentMetaKey key[16];
            const int Count = sizeof(key) / sizeof(key[0]);
            int outCount;
            NN_RESULT_DO(task.ListApplicationContentMetaKey(&outCount, key, Count, offset));
            for (int i = 0; i < outCount; i++)
            {
                NN_LOG("  ApplicationId 0x%016llx, ContentMetaType %u, ContentMetaId 0x%016llx, ContentMetaVersion %u\n",
                          key[i].applicationId.value, key[i].key.type, key[i].key.id, key[i].key.version);
                installed.push_back(key[i]);
            }

            if (outCount < Count)
            {
                break;
            }

            offset += outCount;
        }
        NN_RESULT_DO(task.Execute());
        progress = task.GetProgress();
        NN_LOG("Installed. progress.installedSize %lld, progress.totalSize %lld\n", progress.installedSize, progress.totalSize);
        NN_RESULT_DO(task.Commit());

        *outInstalled = installed;

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteAllContents(nn::Bit64 id, nn::ncm::StorageId storageId)
    {
        nn::ncm::ContentMetaDatabase db;
        nn::ncm::ContentStorage storage;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
        NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, storageId));
        nn::ncm::ContentManagerAccessor accessor(&db, &storage);
        NN_RESULT_DO(accessor.DeleteAll(id));
        NN_RESULT_DO(db.Commit());

        NN_RESULT_SUCCESS;
    }

    nn::Result GetFileHash(nn::crypto::Sha256Generator* outValue, char* path)
    {
        nn::fs::FileHandle file;
        NN_RESULT_DO(nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };
        static const int BufferSize = 16 * 1024 * 1024;
        std::unique_ptr<char> buffer(new char[BufferSize]);

        outValue->Initialize();
        int64_t offset = 0;
        for (;;)
        {
            size_t read;
            NN_RESULT_DO(nn::fs::ReadFile(&read, file, offset, buffer.get(), BufferSize));
            if (read == 0)
            {
                break;
            }

            offset += read;
            outValue->Update(buffer.get(), read);
        }

        NN_RESULT_SUCCESS;
    }
};

}}
