﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/nn_Allocator.h>

// Siglo 環境での警告抑止用
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(push)
#pragma warning(disable : 4244)
#pragma warning(disable : 4668)
#pragma warning(disable : 4702)
#endif

#include <rapidjson/document.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
#pragma warning(pop)
#endif

namespace nnt{ namespace graphics{

/*  データのサンプル
{
  "header": {
    "name": "filename.lyt",
    "date": "2016-08-19 10:08:25",
    "environment": "Generic-Win64",
    "cpu_average_load": 11100,
    "gpu_average_load": 11100,
    "initialization_load": 9050
  },
  "cpu_load": {
  "values": [
      {
        "x" : 0,
        "y" : 11100
      },
      {
        "x" : 1,
        "y" : 11100
      },
      {
        "x" : 2,
        "y" : 11100
      }
    ]
  },
  "gpu_load": {
  "values": [
      {
        "x" : 0,
        "y" : 11100
      },
      {
        "x" : 1,
        "y" : 11100
      },
      {
        "x" : 2,
        "y" : 11100
      }
    ]
  },
  "memory_max_usage": {
  "values": [
      {
        "x" : 0,
        "y" : 123456
      },
      {
        "x" : 1,
        "y" : 123456
      },
      {
        "x" : 2,
        "y" : 123000
      }
    ]
  }
}
*/


//!
//! @brief  パフォーマンス計測結果です。
//!
class PerformanceProfileData
{
public:
    enum FrameValueType
    {
        FrameValueType_CpuLoad,     //!< CPU の負荷情報
        FrameValueType_GpuLoad,     //!< GPU の負荷情報
        FrameValueType_MemoryUsage, //!< メモリ使用量の情報
        FrameValueTypeMax
    };

    //!
    //! @brief  1 フレームに対応する値を保持するクラス
    //!
    class FrameValue
    {
    public:
        //! コンストラクタ
        FrameValue()
            : m_Frame(0)
            , m_Value(0)
        {
        }

        //! デストラクタ
        ~FrameValue()
        {
        }

        //!
        //! @brief  フレームを設定します。
        //!
        //! @param[in]  frame   設定するフレーム番号。
        //!
        void SetFrame(int frame)
        {
            m_Frame = frame;
        }

        //!
        //! @brief  値を設定します。
        //!
        //! @param[in]  frame   設定する値。
        //!
        void SetValue(int64_t value)
        {
            m_Value = value;
        }

        //!
        //! @brief  フレームを取得します。
        //!
        //! @return フレーム
        //!
        int GetFrame() const
        {
            return m_Frame;
        }

        //!
        //! @brief  値を取得します。
        //!
        //! @return 値
        //!
        int64_t GetValue() const
        {
            return m_Value;
        }
    private:
        uint32_t    m_Frame;
        int64_t     m_Value;
    };
public:
    //!
    //! @brief  コンストラクタ
    //!
    PerformanceProfileData();

    //!
    //! @brief  デストラクタ
    //!
    ~PerformanceProfileData();

    //!
    //! @brief  初期化処理。データ出力用に初期化します。
    //!
    //! @param[in]  maxDataCount    蓄積可能なデータの最大数。
    //! @param[in]  pAlloc          メモリ確保関数。
    //! @param[in]  pDealloc        メモリ開放関数。
    //!
    void Initialize(int maxDataCount, nn::AllocateFunction pAlloc, nn::DeallocateFunction pDealloc);

    //!
    //! @brief  初期化処理です。
    //!
    //! @details    渡された JSON ファイルを解析して内部表現形式のデータとして保持します。
    //!
    //! @param[in]  pJsonText   解析する json 形式のテキストデータです。
    //! !param[in]  pAlloc      内部で使用するアロケータです。
    //! !param[in]  pDealloc    内部で使用するアロケータです。
    //!
    //! @return 初期化に成功すると true、失敗すると false が返ります。
    //!
    bool Initialize(const char* pJsonText, nn::AllocateFunction pAlloc, nn::DeallocateFunction pDealloc);

    //!
    //! @brief  終了処理
    //!
    void Finalize();

    //!
    //! @brief  計測データの名前を設定します。設定された名前はヘッダとして出力されます。
    //!
    //! @param[in]  pName   計測データの名前
    //!
    void SetName(const char* pName);

    //!
    //! @brief  計測データの日付を設定します。設定された日付はヘッダとして出力されます。
    //!
    //! @param[in]  pDate   日付。YYYY-MM-DD HH:HH:MM の形式にフォーマットされている必要があります。
    //!
    void SetDate(const char* pDate);

    //!
    //! @brief  データを計測した環境に関するテキストを設定します。設定されたテキストはヘッダとして出力されます。
    //!
    //! @param[in]  pText   環境の名称テキスト
    //!
    void SetEnvironmentText(const char* pText);

    //!
    //! @brief  データに関するコメントテキストを設定します。設定されたテキストはヘッダとして出力されます。
    //!
    //! @param[in]  pText   コメントテキスト
    //!
    void SetCommentText(const char* pText);

    //!
    //! @brief  CPU 負荷の情報を記録する。
    //!
    //! @param[in]  frame   CPU 負荷を記録するフレーム番号です。
    //! @param[in]  value   CPU 負荷の数値(us)です。
    //!
    void SetCpuValue(int frame, int64_t value);

    //!
    //! @brief  GPU 負荷の情報を記録する。
    //!
    //! @param[in]  frame   GPU 負荷を記録するフレーム番号です。
    //! @param[in]  value   GPU 負荷の数値(us)です。
    //!
    void SetGpuValue(int frame, int64_t value);


    //!
    //! @brief  メモリ使用量の情報を記録する。
    //!
    //! @param[in]  frame   メモリ使用量を記録するフレーム番号です。
    //! @param[in]  value   メモリ使用量(byte)です。
    //!
    void SetMemoryUsageValue(int frame, int64_t value);

    //!
    //! @brief  初期化負荷の情報を記録する。
    //!
    //! @param[in]  value   初期化時の負荷(us)です。
    //!
    void SetInitializationLoadValue(int64_t value)
    {
        m_InitializationLoad = value;
    }

    //!
    //! @brief  平均値を取得します。
    //!
    //! @param[in]  type    取得する平均値の種類。
    //!
    //! @return 平均値
    //!
    int64_t GetAverageValue(FrameValueType type) const
    {
        return m_AverageValue[type];
    }

    //!
    //! @brief  フレームごとのデータへのポインタを取得します。
    //!
    //! @param[in]  type    取得するデータの種類。
    //!
    //! @return フレームごとのデータ
    //!
    const FrameValue* GetFrameValues(FrameValueType type) const
    {
        return m_pFrameValues[type];
    }

    //!
    //! @brief  フレームごとのデータの数を取得します。
    //!
    //! @param[in]  type    取得するデータの種類。
    //!
    //! @return フレームごとのデータ数
    //!
    const int GetFrameValueCount(FrameValueType type) const
    {
        return m_FrameValueCount[type];
    }

    //!
    //! @brief  初期化時の負荷の値を取得します。
    //!
    //! @return 負荷の値
    //!
    int64_t GetInitializationLoadValue() const
    {
        return m_InitializationLoad;
    }

    //!
    //! @brief  蓄積されている計測結果の情報をファイルへ出力する。
    //!
    //! @param  pFilePath   出力するファイルのパス。
    //!
    //! @return データが正しく書き込めた場合は true、書き込みに失敗した場合は false を返します。
    //!
    bool Write(const char* pFilePath);

protected:
private:
    void AddElement(FrameValueType type, int frame, int64_t value);
    void CalculateAverageValue(FrameValueType type);

    void ReadString(char** ppWrite, rapidjson::Value& value, const char* pName);
    void ReadInt64(int64_t* pWrite, rapidjson::Value& value, const char* pName);
    void ReadFrameValue(FrameValue** ppWrite, uint32_t* pCount, rapidjson::Document& doc, const char* pRootTagName);

    nn::AllocateFunction    m_pAllocator;
    nn::DeallocateFunction  m_pDeallocator;

    char*       m_pName;
    char*       m_pDate;
    char*       m_pEnvironment;
    char*       m_pComment;

    int64_t     m_InitializationLoad;

    FrameValue*     m_pFrameValues[FrameValueTypeMax];
    uint32_t        m_FrameValueCount[FrameValueTypeMax];
    uint32_t        m_AvaiableElementCountMax[FrameValueTypeMax];
    int64_t         m_AverageValue[FrameValueTypeMax];
};


}}

