﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <iostream>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nnt/gfx/testGfx_PngCommon.h>

namespace nnt {
namespace gfx {

/**
* @brief PNG入出力
*
* @details
*/
class NntGfxPngIO
{
public:

    /**
     * @brief カラーマスク
     *
     * @details カラータイプの組み合わせ用
     */
    enum PNG_COLOR_MASK
    {
        PNG_COLOR_MASK_PALETTE = 1, //!< パレット使用
        PNG_COLOR_MASK_COLOR   = 2, //!< カラー
        PNG_COLOR_MASK_ALPHA   = 4  //!< αチャンネル
    };

    /**
     * @brief カラータイプ
     *
     * @details 0 = グレイスケール
                3 = パレット
                2 = RGB
                6 = RGB+アルファ
                4 = グレイスケール+アルファ
     */
    enum PNG_COLOR_TYPE
    {
        PNG_COLOR_TYPE_GRAY         = 0,                                                //!< グレイスケール
        PNG_COLOR_TYPE_PALETTE      = (PNG_COLOR_MASK_COLOR | PNG_COLOR_MASK_PALETTE),  //!< パレット
        PNG_COLOR_TYPE_RGB          = (PNG_COLOR_MASK_COLOR),                           //!< RGB
        PNG_COLOR_TYPE_RGB_ALPHA    = (PNG_COLOR_MASK_COLOR | PNG_COLOR_MASK_ALPHA),    //!< RGB+アルファ
        PNG_COLOR_TYPE_GRAY_ALPHA   = PNG_COLOR_MASK_ALPHA                              //!< グレイスケール+アルファ
    };

    /**
     * @brief PNGファイル読み込み
     *
     * @param[out] pImage        イメージデータ
     * @param[in]  pPath         ファイルパス
     * @param[in]  pHeapHandle   ヒープハンドル
     * @return     入力に成功したときtrueを返します。
    */
    static bool ReadPng( uint8_t** pImage, const char* pPath, nn::lmem::HeapHandle pHeapHandle );

    /**
     * @brief PNGファイル読み込み
     *
     * @param[out] pImage        イメージデータ
     * @param[in]  pPath         ファイルパス
     * @param[in]  pHeapHandle   ヒープハンドル
     * @param[out] ihdr          PNGのIHDRデータ
     * @return     入力に成功したときtrueを返します。
    */
    static bool ReadPng( uint8_t** pImage, const char* pPath, nn::lmem::HeapHandle pHeapHandle, PNG_IHDR& ihdr );

    /**
     * @brief PNGファイル書き込み
     *
     * @param[in] pPath           ファイルパス
     * @param[in] pImage          イメージデータ
     * @param[in] pHeapHandle     ヒープハンドル
     * @param[in] width           横幅
     * @param[in] height          縦幅
     * @param[in] bitDepth        縦幅
     * @param[in] colorType       カラータイプ
     * @param[in] compressType    圧縮手法
     * @param[in] filterType      フィルター手法
     * @param[in] interlaceType   インターレース手法
     * @return     保存に成功したときtrueを返します。
    */
    static bool WritePng( const char* pPath, const uint8_t* pImage, nn::lmem::HeapHandle pHeapHandle, int width, int height, uint8_t bitDepth, uint8_t colorType, uint8_t compressType, uint8_t filterType, uint8_t interlaceType );

    /**
     * @brief PNGファイル書き込み
     *
     * @param[in] pPath           ファイルパス
     * @param[in] pImage          イメージデータ
     * @param[in] ihdr            PNGのIHDRデータ
     * @param[in] pHeapHandle     ヒープハンドル
     * @return     保存に成功したときtrueを返します。
    */
    static bool WritePng( const char* pPath, const uint8_t* pImage, PNG_IHDR ihdr, nn::lmem::HeapHandle pHeapHandle );


private:

    /**
     * @brief 8bitデータをストリームに出力
     *
     * @param[in] pStream 出力ストリーム
     * @param[in] pData   データ
     * @param[in] pCrc    CRC
    */
    static void Output8( std::ostream* pStream, const void* pData, uint32_t* pCrc = NULL );

    /**
     * @brief 32bitデータをストリームに出力
     *
     * @param[in] pStream 出力ストリーム
     * @param[in] pData   データ
     * @param[in] pCrc    CRC
    */
    static void Output32( std::ostream* pStream, const void* pData, uint32_t* pCrc = NULL );


    /**
     * @brief バイト列データをストリームに出力
     *
     * @param[in] pStream 出力ストリーム
     * @param[in] pData   データ
     * @param[in] size    データサイズ
     * @param[in] pCrc    CRC
    */
    static void OutputBytes( std::ostream* pStream, const void* pData, size_t size, uint32_t* pCrc = NULL );


    /**
     * @brief 圧縮率を上げるためピクセルデータにフィルターをかける
     *
     * @param[in] pInput  入力データ
     * @param[in] width   画像の横幅
     * @param[in] height  画像の縦幅
     * @param[in] pOutput 出力データ
    */
    static void CompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput );


    /**
     * @brief 8bitデータをストリームから入力
     *
     * @param[in] pStream 入力ストリーム
     * @return 入力データ
    */
    static uint8_t Input8( uint8_t** pStream );


    /**
     * @brief 32bitデータをストリームから入力
     *
     * @param[in] pStream 入力ストリーム
     * @return 入力データ
    */
    static uint32_t Input32( uint8_t** pStream );

    /**
     * @brief バイト列データをストリームから入力
     *
     * @param[in] pStream 入力ストリーム
     * @param[in] pData   データ
     * @param[in] size    データサイズ
    */
    static void InputBytes( uint8_t** pStream, void* pData, size_t size );


    /**
     * @brief フィルタを適用したピクセルデータを復元
     *
     * @param[in] pInput        入力データ
     * @param[in] width         画像の横幅
     * @param[in] height        画像の縦幅
     * @param[in] pixelBytes    1ピクセルあたりのバイト数
     * @param[in] pOutput       出力データ
    */
    static void DecompressFilter( const uint8_t* pInput, int width, int height, uint8_t pixelBytes, uint8_t* pOutput );

};


}
}
