﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <memory>
#include <mutex>

#include <nn/nn_Log.h>
#include <nn/nn_Allocator.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_MutexApi.h>

#include <nnt/nntest.h>

namespace nnt { namespace fs { namespace util {

struct StaticMutex
{
    nn::os::MutexType _mutex;

    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&_mutex);
    }

    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&_mutex);
    }
};

/**
    @brief global new/delete が実行されたことを示すフラグの値を取得します。
*/
bool IsGlobalNewDeleteCalled() NN_NOEXCEPT;

/**
    @brief global new/delete が実行されたことを示すフラグを設定します。
*/
void SetGlobalNewDeleteFlag() NN_NOEXCEPT;

/**
    @brief global new/delete が実行されたことを示すフラグを解除します。
*/
void ResetGlobalNewDeleteFlag() NN_NOEXCEPT;

void SetStackTraceDumpOnGlobalNewDeleteCallEnabled(bool isEnabled) NN_NOEXCEPT;

bool IsStackTraceDumpOnGlobalNewDeleteCallEnabled() NN_NOEXCEPT;

std::unique_lock<StaticMutex> GetScopedLockForGlobalNewDeleteChecker() NN_NOEXCEPT;

/**
    @brief nnt::fs::util::Allocate(), nnt::fs::util::Deallocate() で利用するヒープ領域を設定します。
           この関数を実行しなかった場合は、libnnt_fsUtil ライブラリ内で確保した TestLibraryHeapSize
           分の領域が使用されます。
*/
const int  TestLibraryHeapSize = 512 * 1024;
void InitializeTestLibraryHeap(void* pHeap, const int heapSize) NN_NOEXCEPT;

/**
    @brief 既に確保されているメモリーサイズを取得します。
           TestFsのメモリーリークチェック用です。
*/
int64_t GetStartAllocateSize() NN_NOEXCEPT;

/**
    @brief ヒープの確保・解放の累積サイズをリセットします
*/
void ResetAllocateCount() NN_NOEXCEPT;
/**
    @brief ヒープの確保・解放の累積サイズを比較し、相違がある場合にはtrueを返します。
*/
bool CheckMemoryLeak() NN_NOEXCEPT;

/**
    @brief 指定したヒープサイズを開始位置としヒープの確保・解放の累積サイズを比較し、相違がある場合にはtrueを返します。
*/
bool CheckMemoryLeak(const int64_t startAllocated) NN_NOEXCEPT;

// 内部使用
void SetAllocationPointerOffset(int offset) NN_NOEXCEPT;
void* Allocate(const size_t size) NN_NOEXCEPT;
void Deallocate(void* p, const size_t size) NN_NOEXCEPT;

void* StdAllocate(const size_t size) NN_NOEXCEPT;
void StdDeallocate(void* p, const size_t size) NN_NOEXCEPT;

/**
    @brief 暫定の文字列型定義です。
*/
template <class T>
struct Allocator : public std::allocator<T>
{
    Allocator(){}
    Allocator(const Allocator&){}
    template<class U>
    Allocator(const Allocator<U>&){}

    template<class U>
    struct rebind
    {
        typedef Allocator<U> other;
    };

    T* allocate(const size_t size, const T* hint = 0) NN_NOEXCEPT
    {
        NN_UNUSED(hint);
        return static_cast<T*>(Allocate(sizeof(T) * size));
    }
    void deallocate(T* p, const size_t size) NN_NOEXCEPT
    {
        Deallocate(p, sizeof(T) * size);
    }
};

/**
    @brief STDライブラリ側で使用するアローケーターテンプレート定義です。
           メモリーリークチェック対象としないように分離しています。
*/
template <class T>
struct StdAllocator : public std::allocator<T>
{
    StdAllocator(){}
    StdAllocator(const StdAllocator&){}
    template<class U>
    StdAllocator(const StdAllocator<U>&){}

    template<class U>
    struct rebind
    {
        typedef StdAllocator<U> other;
    };

    T* allocate(const size_t size, const T* hint = 0) NN_NOEXCEPT
    {
        NN_UNUSED(hint);
        return static_cast<T*>(StdAllocate(sizeof(T) * size));
    }
    void deallocate(T* p, const size_t size) NN_NOEXCEPT
    {
        StdDeallocate(p, sizeof(T) * size);
    }
};


/**
    @brief nnt::fs::util::Allocate(), nnt::fs::util::Deallocate() を使うアロケータです。
*/
nn::MemoryResource* GetTestLibraryAllocator() NN_NOEXCEPT;

}}}
