﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <random>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/buffers/fs_BufferManagerUtility.h>
#include <nn/result/result_HandlingUtility.h>

namespace nnt { namespace fs { namespace util {

    class RandomFailureBufferManager : public nn::fssystem::IBufferManager
    {
    public:
        RandomFailureBufferManager(nn::fssystem::IBufferManager* pBaseBufferManager, std::mt19937* pMt) NN_NOEXCEPT
            : m_pBaseBufferManager(pBaseBufferManager),
            m_pMt(pMt),
            m_Divider(0),
            m_DividerOriginal(0)
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pBaseBufferManager);
            NN_SDK_REQUIRES_NOT_NULL(m_pMt);
        }

        virtual ~RandomFailureBufferManager() NN_NOEXCEPT NN_OVERRIDE {}

    public:
        void SetDivider(int divider) NN_NOEXCEPT
        {
            m_DividerOriginal = divider;
            m_Divider = m_DividerOriginal;
            m_AllocationCount = 0;
        }

        int GetAllocationCount() NN_NOEXCEPT
        {
            return m_AllocationCount;
        }

        virtual const std::pair<uintptr_t, size_t> DoAllocateBuffer(
            size_t size,
            const BufferAttribute& bufAttr
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_SDK_ASSERT_NOT_EQUAL(0, m_DividerOriginal);
            NN_SDK_ASSERT_NOT_EQUAL(0, m_Divider);

            ++m_AllocationCount;

            std::pair<uintptr_t, size_t> buffer(0, 0);
            if( std::uniform_int_distribution<>(1, m_Divider)(*m_pMt) <= 1 )
            {
                auto pContext = nn::fssystem::buffers::GetBufferManagerContext();
                if( pContext == nullptr || !pContext->IsNeedBlocking() )
                {
                    // スタックが巻き戻ると予測してリセット
                    // 確実にテストを終わらせるために確率を下げる
                    m_DividerOriginal *= 2;
                    m_Divider = m_DividerOriginal;
                    m_AllocationCount = 0;
                }
            }
            else
            {
                if( m_Divider >= 3 )
                {
                    // 深いところでも浅いところと同確率で発生させたい
                    --m_Divider;
                }
                buffer = m_pBaseBufferManager->AllocateBuffer(size, bufAttr);
            }
            return buffer;
        }

        virtual void DoDeallocateBuffer(uintptr_t address, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            m_pBaseBufferManager->DeallocateBuffer(address, size);
        }

        virtual CacheHandle DoRegisterCache(
            uintptr_t address,
            size_t size,
            const BufferAttribute& bufAttr
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseBufferManager->RegisterCache(address, size, bufAttr);
        }

        virtual const std::pair<uintptr_t, size_t> DoAcquireCache(
            CacheHandle handle
        ) NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseBufferManager->AcquireCache(handle);
        }

        virtual size_t DoGetTotalSize() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseBufferManager->GetTotalSize();
        }

        virtual size_t DoGetFreeSize() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseBufferManager->GetFreeSize();
        }

        virtual size_t DoGetTotalAllocatableSize() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_pBaseBufferManager->GetTotalAllocatableSize();
        }

        virtual size_t DoGetFreeSizePeak() const NN_NOEXCEPT NN_OVERRIDE
        {
            return 0;
        }

        virtual size_t DoGetTotalAllocatableSizePeak() const NN_NOEXCEPT NN_OVERRIDE
        {
            return 0;
        }

        virtual size_t DoGetRetriedCount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return 0;
        }

        virtual void DoClearPeak() NN_NOEXCEPT NN_OVERRIDE
        {
        }

    private:
        IBufferManager* m_pBaseBufferManager;
        std::mt19937* m_pMt;
        int m_Divider;
        int m_DividerOriginal;
        int m_AllocationCount;
    };

}}}
