﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_SubStorage.h>

namespace nnt { namespace fs { namespace util {

/*
* @brief    電源断のテストを行うためのストレージ
*
* @details
*       指定した回数の書き込みが行われると、
*       それ以降に読み書きができなくなるストレージです。
*/
class PowerInterruptionStorage : public nn::fs::IStorage
{
    NN_DISALLOW_COPY(PowerInterruptionStorage);
public:
    typedef nn::fs::ResultSdCardAccessFailed ResultPowerInterruptionOccurred;

public:
    //! このストレージの状態
    enum class State
    {
        NotInitialized, //!< 初期化前状態
        Waiting,        //!< 待機状態              何度書き込まれても、電源断状態に遷移しません。
        Counting,       //!< 書き込みカウント状態  指定回数書き込みが行われると、電源断状態に遷移します。
        RunOut          //!< 電源断状態            書き込み、読み込みが行われるとエラーを返します。
    };

public:
    /**
    * @brief        デフォルトコンストラクタ
    *
    * @details      あとで初期化する必要があります
    */
    PowerInterruptionStorage() NN_NOEXCEPT
      : m_BaseStorage(),
        m_WritableCount(0),
        m_WrittenCount(0),
        m_State(State::NotInitialized),
        m_Mutex(false)
    {
    }

    /**
    * @brief        初期化を行うコンストラクタ
    *
    * @param[in]    storage         下に敷かれるサブストレージ
    *
    * @details      待機状態になります。
    */
    explicit PowerInterruptionStorage(nn::fs::SubStorage storage) NN_NOEXCEPT
      : m_BaseStorage(storage),
        m_WritableCount(0),
        m_WrittenCount(0),
        m_State(State::Waiting),
        m_Mutex(false)
    {
    }

    /**
    * @brief        初期化
    *
    * @param[in]    storage         下に敷かれるサブストレージ
    *
    * @details      状態は待機状態になります。
    */
    void Initialize(nn::fs::SubStorage storage) NN_NOEXCEPT
    {
        m_BaseStorage = storage;
        m_WritableCount = 0;
        m_WrittenCount = 0;
        m_State = State::Waiting;
    }

    /**
    * @brief        下位ストレージの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultPowerInterruptionOccurred   読み書き不可となっています
    *
    * @details      読み書き不可となっている場合、読み込みできません
    */
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        {
            std::lock_guard<nn::os::Mutex> locker(m_Mutex);
            if( IsPowerInterruptionOccurred() )
            {
                return ResultPowerInterruptionOccurred();
            }
        }
        return m_BaseStorage.Read(offset, buffer, size);
    }

    /**
    * @brief        バッファの内容を下位ストレージに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultPowerInterruptionOccurred   読み書き不可となっています
    *
    * @details      読み書き不可となっている場合、書き込みできません
    */
    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        {
            std::lock_guard<nn::os::Mutex> locker(m_Mutex);
            UpdateStateByWrite();
            if( IsPowerInterruptionOccurred() )
            {
                return ResultPowerInterruptionOccurred();
            }
        }
        return m_BaseStorage.Write(offset, buffer, size);
    }

    /**
    * @brief        下位ストレージをフラッシュします。
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultPowerInterruptionOccurred   読み書き不可となっています
    *
    * @details      読み書き不可となっている場合、フラッシュできません
    */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        {
            std::lock_guard<nn::os::Mutex> locker(m_Mutex);
            UpdateStateByWrite();
            if( IsPowerInterruptionOccurred() )
            {
                return ResultPowerInterruptionOccurred();
            }
        }
        return m_BaseStorage.Flush();
    }

    /**
    * @brief        下位ストレージのサイズを返します
    *
    * @param[out]   outValue        下位ストレージのサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultPowerInterruptionOccurred   読み書き不可となっています
    *
    * @details      読み書き不可となっている場合、サイズを取得できません
    */
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        {
            std::lock_guard<nn::os::Mutex> locker(m_Mutex);
            if( IsPowerInterruptionOccurred() )
            {
                return ResultPowerInterruptionOccurred();
            }
        }
        return m_BaseStorage.GetSize(outValue);
    }

    /**
    * @brief        下位ストレージのサイズを設定します
    *
    * @param[out]   value           下位ストレージのサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultPowerInterruptionOccurred   読み書き不可となっています
    *
    * @details      読み書き不可となっている場合、サイズを取得できません
    */
    virtual nn::Result SetSize(int64_t value) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        {
            std::lock_guard<nn::os::Mutex> locker(m_Mutex);
            UpdateStateByWrite();
            if( IsPowerInterruptionOccurred() )
            {
                return ResultPowerInterruptionOccurred();
            }
        }
        return m_BaseStorage.SetSize(value);
    }

    /**
    * @brief        下位ストレージに範囲処理を行います
    *
    * @param[out]   outBuffer       範囲処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     処理 ID
    * @param[in]    offset          開始位置
    * @param[in]    size            終了位置までのサイズ
    * @param[in]    inBuffer        範囲処理に渡すバッファ
    * @param[in]    inBufferSize    範囲処理に渡すバッファのサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultPowerInterruptionOccurred   読み書き不可となっています
    *
    * @details      読み書き不可となっている場合、範囲処理を実行できません
    */
    virtual nn::Result OperateRange(
                void* outBuffer,
                size_t outBufferSize,
                nn::fs::OperationId operationId,
                int64_t offset,
                int64_t size,
                const void* inBuffer,
                size_t inBufferSize
            ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        {
            std::lock_guard<nn::os::Mutex> locker(m_Mutex);
            UpdateStateByWrite();
            if( IsPowerInterruptionOccurred() )
            {
                return ResultPowerInterruptionOccurred();
            }
        }
        return m_BaseStorage.OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

    /**
    * @brief        電源断が発生するまでの書き込み可能回数をセットします
    *
    * @param[in]    remainWritableCount       残り書き込み可能回数
    *
    * @details      初期化でもセットできます
    *               今までに書き込まれた回数はリセットされます
    */
    void StartCounting(int remainWritableCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        m_WritableCount = remainWritableCount;
        m_WrittenCount = 0;
        m_State = State::Counting;
    }

    /**
    * @brief        電源断が発生したか
    *
    * @return       電源断が発生したか否かを返します
    * @retval       true    電源断されています
    * @retval       false   電源断されていません
    *
    * @details      電源断が発生したか
    */
    bool IsPowerInterruptionOccurred() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        return m_State == State::RunOut;
    }

    /**
    * @brief    書き込み回数のチェックを終了します。
    *
    * @details  待機状態へ遷移させます
    */
    void StopCounting() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        m_WritableCount = 0;
        m_WrittenCount = 0;
        m_State = State::Waiting;
    }
private:
    /**
    * @brief    書き込み回数をインクリメントする
    *
    * @details  書き込み回数が指定した値に達したら状態遷移させる
    */
    void UpdateStateByWrite()
    {
        NN_SDK_REQUIRES_NOT_EQUAL(m_State, State::NotInitialized);
        if( m_State == State::Counting )
        {
            ++m_WrittenCount;
            if( m_WrittenCount > m_WritableCount )
            {
                m_State = State::RunOut;
            }
        }
    }

private:
    nn::fs::SubStorage m_BaseStorage;   //!< 下位ストレージ
    int m_WritableCount;                //!< 書き込み可能回数
    int m_WrittenCount;                 //!< 書き込まれた回数
    State m_State;                      //!< このストレージの状態
    nn::os::Mutex m_Mutex;              //!< マルチスレッド処理用ミューテックス
};
}}}
