﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SubStorage.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

namespace nnt { namespace fs { namespace util {

/**
* @brief    読み書きにおけるアライメントをチェックするストレージ
*
* @details  読み書きにおけるアライメントをチェックするストレージ
*/
class AlignCheckStorage : public nn::fs::IStorage
{
public:
    /**
    * @brief        コンストラクタ
    *
    * @details      あとで初期化を行う必要があります
    */
    AlignCheckStorage() NN_NOEXCEPT
        : m_BaseStorage(),
          m_AlignBlockSize(0)
    {
    }

    //! デストラクタ
    virtual ~AlignCheckStorage() NN_NOEXCEPT NN_OVERRIDE {}

    /**
    * @brief        コンストラクタ
    *
    * @param[in]    storage         ベースストレージ
    * @param[in]    alignBlockSize  アライメントチェックするブロックサイズ
    *
    * @details      初期化も行う
    */
    AlignCheckStorage(nn::fs::SubStorage storage, size_t alignBlockSize) NN_NOEXCEPT
        : m_BaseStorage(),
          m_AlignBlockSize(0)
    {
        Initialize(storage, alignBlockSize);
    }

    /**
    * @brief        初期化
    *
    * @param[in]    storage         ベースストレージ
    * @param[in]    alignBlockSize  アライメントチェックするブロックサイズ
    *
    * @pre          alignBlockSize > 0
    * @pre          alignBlockSize が 2 の乗数
    *
    * @details      初期化
    */
    void Initialize(nn::fs::SubStorage storage, size_t alignBlockSize) NN_NOEXCEPT
    {
        ASSERT_LT(0, static_cast<int64_t>(alignBlockSize));
        ASSERT_TRUE(nn::util::ispow2(alignBlockSize));
        m_BaseStorage = storage;
        m_AlignBlockSize = alignBlockSize;
    }

    /**
    * @brief        ファイナライズ
    *
    * @details      ファイナライズ
    */
    void Finalize() NN_NOEXCEPT
    {
        m_BaseStorage = nn::fs::SubStorage();
        m_AlignBlockSize = 0;
    }

    /**
    * @brief    アライメントをチェックしてから読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込むバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return   関数の処理結果を返します。
    * @retval   ResultInvalidAlignment      offset % m_AlignBlockSize != 0
    *
    * @details  offset がブロックサイズでアライメントされているかどうかチェックします。
    *           満たさない場合はエラーリザルトを返します
    */
    virtual nn::Result Read(
                int64_t offset,
                void* buffer,
                size_t size
            ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !nn::util::is_aligned<int64_t>(offset, m_AlignBlockSize) )
        {
            return nn::fs::ResultInvalidAlignment();
        }
        return m_BaseStorage.Read(offset, buffer, size);
    }

    /**
    * @brief    アライメントをチェックしてから読み込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return   関数の処理結果を返します。
    * @retval   ResultInvalidAlignment      offset % m_AlignBlockSize != 0
    *
    * @details  offset がブロックサイズでアライメントされているかどうかチェックします。
    *           満たさない場合はエラーリザルトを返します
    */
    virtual nn::Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( ! nn::util::is_aligned<int64_t>(offset, m_AlignBlockSize) )
        {
            return nn::fs::ResultInvalidAlignment();
        }
        return m_BaseStorage.Write(offset, buffer, size);
    }

    /**
    * @brief    アライメントをチェックしてから範囲処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲処理の ID
    * @param[in]    offset          範囲処理開始位置
    * @param[in]    size            範囲処理を行う範囲
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return   関数の処理結果を返します。
    * @retval   ResultInvalidAlignment      offset % m_AlignBlockSize != 0
    *
    * @details  offset がブロックサイズでアライメントされているかどうかチェックします。
    *           満たさない場合はエラーリザルトを返します
    */
    virtual nn::Result OperateRange(
            void* outBuffer,
            size_t outBufferSize,
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize
        ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( !nn::util::is_aligned<int64_t>(offset, m_AlignBlockSize) )
        {
            return nn::fs::ResultInvalidAlignment();
        }
        return m_BaseStorage.OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

    virtual nn::Result GetSize(int64_t* outSize) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseStorage.GetSize(outSize);
    }
    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseStorage.SetSize(size);
    }
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_BaseStorage.Flush();
    }

private:
    nn::fs::SubStorage m_BaseStorage;
    size_t m_AlignBlockSize;
};

}}}
