﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <limits>
#include <algorithm>
#include <atomic>

#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/fs_IStorage.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "testFs_util_SafeMemoryStorage.h"

namespace nnt { namespace fs { namespace util {

//! アクセスカウント
class AccessCountData
{
private:
    std::atomic_int m_ReadTimes;         //!< 読み込み回数
    std::atomic_int m_WriteTimes;        //!< 書き込み回数
    std::atomic_int m_FlushTimes;        //!< フラッシュ回数
    std::atomic_int m_InvalidateTimes;   //!< キャッシュ無効化回数

public:
    AccessCountData() NN_NOEXCEPT
    {
        Reset();
    }

    /**
    * @brief    アクセスカウントデータをリセットします。
    */
    void Reset() NN_NOEXCEPT
    {
        m_ReadTimes = 0;
        m_WriteTimes = 0;
        m_FlushTimes = 0;
        m_InvalidateTimes = 0;
    }

    /**
    * @brief    読み込み回数をインクリメントします。
    */
    void IncrementReadTimes() NN_NOEXCEPT
    {
        m_ReadTimes++;
    }

    /**
    * @brief    現在の読み込み回数を取得します。
    *
    * @return   読み込み回数を返します。
    */
    int GetReadTimes() const NN_NOEXCEPT
    {
        return m_ReadTimes.load();
    }

    /**
    * @brief    書き込み回数をインクリメントします。
    */
    void IncrementWriteTimes() NN_NOEXCEPT
    {
        m_WriteTimes++;
    }

    /**
    * @brief    現在の書き込み回数を取得します。
    *
    * @return   書き込み回数を返します。
    */
    int GetWriteTimes() const NN_NOEXCEPT
    {
        return m_WriteTimes.load();
    }

    /**
    * @brief    フラッシュ回数をインクリメントします。
    */
    void IncrementFlushTimes() NN_NOEXCEPT
    {
        m_FlushTimes++;
    }

    /**
    * @brief    現在のフラッシュ回数を取得します。
    *
    * @return   フラッシュ回数を返します。
    */
    int GetFlushTimes() const NN_NOEXCEPT
    {
        return m_FlushTimes.load();
    }

    /**
    * @brief    キャッシュ無効化回数をインクリメントします。
    */
    void IncrementInvalidateTimes() NN_NOEXCEPT
    {
        m_InvalidateTimes++;
    }

    /**
    * @brief    現在のキャッシュ無効化回数を取得します。
    *
    * @return   キャッシュ無効化回数を返します。
    */
    int GetInvalidateTimes() const NN_NOEXCEPT
    {
        return m_InvalidateTimes.load();
    }
};

/**
* @brief アクセスカウント付きストレージラッパーです。
*/
class AccessCountWrapperStorage : public nn::fs::IStorage
{
public:
    explicit AccessCountWrapperStorage(nn::fs::IStorage* pStorage) NN_NOEXCEPT
    : m_pStorage(pStorage)
    {
        m_AccessCountData.Reset();
    }

    /**
    * @brief        マッピングテーブル経由でデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        // 読み込み回数カウントアップ
        m_AccessCountData.IncrementReadTimes();

        return m_pStorage->Read(offset, buffer, size);
    }

    /**
    * @brief        マッピングテーブル経由でデータを書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size > 0 )
        {
            // 書き込み回数カウントアップ
            m_AccessCountData.IncrementWriteTimes();
        }
        return m_pStorage->Write(offset, buffer, size);
    }

    /**
    * @brief        書き込みのフラッシュを行ないます。
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        // フラッシュ指定書き込み回数カウントアップ
        m_AccessCountData.IncrementFlushTimes();

        return m_pStorage->Flush();
    }

    /**
    * @brief        ストレージのサイズを取得します。
    *
    * @param[out]   outValue    取得したサイズ格納先
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->GetSize(outValue);
    }

    /**
    * @brief   ストレージサイズを変更します。
    *
    * @param[in] size 変更後のサイズ
    *
    * @return   関数の処理結果を返します。
    */
    virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(m_pStorage);
        return m_pStorage->SetSize(size);
    }

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
    * @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]    operationId     範囲指定処理の種類
    * @param[in]    offset          範囲指定処理開始位置
    * @param[in]    size            範囲指定処理を行うデータサイズ
    * @param[in]    inBuffer        範囲指定処理に渡すバッファ
    * @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual nn::Result OperateRange(
                           void* outBuffer,
                           size_t outBufferSize,
                           nn::fs::OperationId operationId,
                           int64_t offset,
                           int64_t size,
                           const void* inBuffer,
                           size_t inBufferSize
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( operationId == nn::fs::OperationId::Invalidate )
        {
            m_AccessCountData.IncrementInvalidateTimes();
        }
        return m_pStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

public:
    /**
    * @brief    アクセスカウントデータをリセットします。
    */
    void ResetAccessCounter() NN_NOEXCEPT
    {
        m_AccessCountData.Reset();
    }

    /**
    * @brief    現在の読み込み回数を取得します。
    *
    * @return   読み込み回数を返します。
    */
    int GetReadTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetReadTimes();
    }

    /**
    * @brief    現在の書き込み回数を取得します。
    *
    * @return   書き込み回数を返します。
    */
    int GetWriteTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetWriteTimes();
    }

    /**
    * @brief    現在のフラッシュ回数を取得します。
    *
    * @return   フラッシュ回数を返します。
    */
    int GetFlushTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetFlushTimes();
    }

    /**
    * @brief    現在のキャッシュ無効化回数を取得します。
    *
    * @return   キャッシュ無効化回数を返します。
    */
    int GetInvalidateTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetInvalidateTimes();
    }

private:
    nn::fs::IStorage* const m_pStorage;   //!< ストレージ
    AccessCountData m_AccessCountData;          //!< アクセスカウントデータ
};

/**
* @brief アクセスカウント付きメモリストレージオブジェクトです。
*/
class AccessCountedMemoryStorage : public SafeMemoryStorage, public nn::fs::detail::Newable
{
public:
    AccessCountedMemoryStorage() NN_NOEXCEPT
    : m_IsRandomError(false),
      m_bVerbose(false)
    {
        m_AccessCountData.Reset();
    }

    virtual ~AccessCountedMemoryStorage() NN_NOEXCEPT
    {
    }

    /**
    * @brief        下位ストレージの offset から size バイト読み込み buffer にコピーします。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Read(
                           int64_t offset,
                           void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_IsRandomError)
        {
            if (m_Random.Get32(5000) < 10)
            {
                return GetRandomReadErrorResult();
            }
        }

        // 読み込み回数カウントアップ
        m_AccessCountData.IncrementReadTimes();

        return SafeMemoryStorage::Read(offset, buffer, size);
    }

    /**
    * @brief 下位ストレージの offset 以降に buffer を size バイト分コピーします。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Write(
                           int64_t offset,
                           const void* buffer,
                           size_t size
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_IsRandomError)
        {
            if (m_Random.Get32(5000) < 10)
            {
                return GetRandomWriteErrorResult();
            }
        }

        if (size > 0)
        {
            // 書き込み回数カウントアップ
            m_AccessCountData.IncrementWriteTimes();
        }

        return SafeMemoryStorage::Write(offset, buffer, size);
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        // フラッシュ指定書き込み回数カウントアップ
        m_AccessCountData.IncrementFlushTimes();
        return SafeMemoryStorage::Flush();
    }

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual nn::Result OperateRange(
                           void* outBuffer,
                           size_t outBufferSize,
                           nn::fs::OperationId operationId,
                           int64_t offset,
                           int64_t size,
                           const void* inBuffer,
                           size_t inBufferSize
                       ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( operationId == nn::fs::OperationId::Invalidate )
        {
            m_AccessCountData.IncrementInvalidateTimes();
        }
        return SafeMemoryStorage::OperateRange(
            outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

    using IStorage::OperateRange;

public:
    /**
    * @brief    アクセスカウントデータをリセットします。
    */
    void ResetAccessCounter() NN_NOEXCEPT
    {
        m_AccessCountData.Reset();
    }

    /**
    * @brief    現在の読み込み回数を取得します。
    *
    * @return   読み込み回数を返します。
    */
    int GetReadTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetReadTimes();
    }

    /**
    * @brief    現在の書き込み回数を取得します。
    *
    * @return   書き込み回数を返します。
    */
    int GetWriteTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetWriteTimes();
    }

    /**
    * @brief    現在のフラッシュ回数を取得します。
    *
    * @return   フラッシュ回数を返します。
    */
    int GetFlushTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetFlushTimes();
    }

    /**
    * @brief    現在のキャッシュ無効化回数を取得します。
    *
    * @return   キャッシュ無効化回数を返します。
    */
    int GetInvalidateTimes() const NN_NOEXCEPT
    {
        return m_AccessCountData.GetInvalidateTimes();
    }

    void SetRandomError(bool isEnable) NN_NOEXCEPT
    {
        m_IsRandomError = isEnable;
    }

    static nn::Result GetRandomReadErrorResult() NN_NOEXCEPT
    {
        // SaveDataFileSystem では返しそうもないエラーを返しときます。
        return nn::fs::ResultSdCardAccessFailed();
    }

    static nn::Result GetRandomWriteErrorResult() NN_NOEXCEPT
    {
        // SaveDataFileSystem では返しそうもないエラーを返しときます。
        return nn::fs::ResultSdCardAccessFailed();
    }

    void SetVerbose(bool bVerbose) NN_NOEXCEPT
    {
        m_bVerbose = bVerbose;
    }


private:
    AccessCountData m_AccessCountData;      //!< アクセスカウントデータ
    bool m_IsRandomError;                   //!< ランダムなタイミングでエラーを発生させるかどうか
    bool m_bVerbose;
    char reserved[2] NN_IS_UNUSED_MEMBER;
    nnt::fs::util::Random m_Random;
};

}}} // namespace nnt::fs::util
