﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os.h>
#include <nn/fs/fs_Directory.h>

namespace nnt {
namespace codec {
namespace util {

class TesterManager
{
private:

    // テスタスレッドを管理する構造体
    struct TesterThreadType
    {
        nn::os::ThreadType handle;
        nn::os::MultiWaitHolderType holder;
        char* stackBase;
        bool isRunning;
    };

public:

    typedef void* (*AllocatorType) (std::size_t size);
    typedef void (*DeallocatorType) (void* p);

    TesterManager() NN_NOEXCEPT
        : m_ThreadCount(0)
        , m_pThreadStruct(nullptr)
        , m_ThreadStructSize(0)
        , m_ThreadStackBase(nullptr)
        , m_ThreadStackSize(0)
        , m_MultiWait()
        , m_IsInitialized(false)
    {}

    void Initialize(int threadCount, AllocatorType allocator) NN_NOEXCEPT;
    void Finalize(DeallocatorType deallocator) NN_NOEXCEPT;

    bool Run(nn::fs::DirectoryEntry& entry, nn::os::ThreadFunction function,int coreNo) NN_NOEXCEPT;

    inline bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:

    void InitializeTesterThread(struct TesterThreadType*, nn::os::ThreadFunction function, nn::fs::DirectoryEntry* pEntry, int coreNo) NN_NOEXCEPT;
    void FinalizeTesterThread(struct TesterThreadType*) NN_NOEXCEPT;

    int GetAvailableThreadNumber(const struct TesterThreadType* pThread, int threadCount) const NN_NOEXCEPT;
    bool IsRunningAll(const struct TesterThreadType* pThread, int threadCount) const NN_NOEXCEPT;

    inline bool IsRunning(const struct TesterThreadType* pThread) const NN_NOEXCEPT
    {
        return pThread->isRunning;
    }

    static const size_t KiB = 1024;
    static const size_t ThreadStackSize = 32 * KiB; // 1スレッドのスタックサイズ

    int m_ThreadCount; // スレッド数

    struct TesterThreadType* m_pThreadStruct; // スレッド管理構造体の先頭ポインタ
    std::size_t m_ThreadStructSize;

    char* m_ThreadStackBase;       // スレッドスタックのベースポインタ
    std::size_t m_ThreadStackSize; // スレッドスタック合計サイズ

    nn::os::MultiWaitType m_MultiWait; // 多重待オブジェクト
    bool    m_IsInitialized;

};

}}} // nnt::codec::util
