﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <string>

#include <nnt.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/audio.h>

namespace
{

template<typename T> static inline void safeDelete(T* ptr);
template<typename T> static inline void safeFree(T* ptr);

}

namespace nnt {
namespace audio {
namespace util {

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
bool IsFirmwareSettingEnabled(const char* key) NN_NOEXCEPT;
#endif

nn::audio::AudioRendererParameter GetAudioRendererParameterForWaveComparison() NN_NOEXCEPT;
nn::audio::AudioRendererParameter GetAudioRendererParameterForDefault() NN_NOEXCEPT;

class ScopedAudioRenderer
{
private:
    void* m_WorkBuffer;
    void* m_ConfigBuffer;
    nn::audio::AudioRendererHandle m_Handle;
    nn::audio::AudioRendererConfig m_Config;
    nn::mem::StandardAllocator m_Allocator;

public:
    NN_IMPLICIT ScopedAudioRenderer(const nn::audio::AudioRendererParameter& parameter = GetAudioRendererParameterForDefault(), nn::os::SystemEvent* pSystemEvent = nullptr) NN_NOEXCEPT;
    ~ScopedAudioRenderer() NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;
    nn::audio::AudioRendererHandle GetHandle() const NN_NOEXCEPT;
    nn::audio::AudioRendererConfig& GetConfig() NN_NOEXCEPT;
};

////////////////////////////////////////////////////////////////////////////////
// Audio File Utility class
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// wave file util
////////////////////////////////////////////////////////////////////////////////

class WaveFile
{
    // TODO: Need to consider big endian case.

public:
    enum WaveEndian
    {
        WaveEndian_Invalid = 0,
        WaveEndian_Little,
        WaveEndian_Big
    };

    enum WaveFileResult
    {
        WaveFileResult_Success = 0,
        WaveFileResult_InvalidDataFormat,
        WaveFileResult_BufferSizeTooSmall,
        WaveFileResult_UnexpectedError,
    };

    WaveFile();
    virtual ~WaveFile();

    // parameter accessors
    int GetSampleRate() const;
    int GetChannelCount() const;
    int GetBitRate() const;
    size_t GetSampleDataSize() const;

    /**
    * @brief       Open a wave file to write.
    * @param[in]   filename
    * @param[in]   sampleRate
    * @param[in]   channels
    * @param[in]   bitRate
    * @return      true if succeeded.
    */
    bool Open(const char* filename, uint32_t sampleRate, uint16_t channels, uint16_t bitRate);

    /**
    * @brief       Open a wave file to read.
    * @param[in]   filename
    * @return      true if succeeded.
    */
    bool Open(const char* filename);

    /**
    * @brief       Close a wave file.
    */
    virtual void Close();

    /**
    * @brief       Write sample data.
    * @param[in]   buf            sample buffer
    * @param[in]   dataSize       data buffer size [byte]
    * @return      return dataSize if succeeded. return 0 if failed.
    * @detail
    * passed data should be interleaved if multi channel source.
    */
    size_t Write(const int8_t* buf, const size_t dataSize);

    /**
    * @brief       Read sample data.
    * @param[in]   buf            sample buffer
    * @param[in]   readSize       data size to read [byte]
    * @return      return readSize if succeeded. return 0 if failed.
    * @detail
    */
    size_t Read(int8_t* buf, const size_t readSize);

    /**
     * @brief      Set current Read/Write position.
     * @param[in] offset [byte]
     * @returns (void):
     */
     void Seek(const uint32_t offset);

    /**
    * @brief       Get hash value of this file.
    * @return      hash value.
    * @detail
    */
    uint32_t GetHash(); // TODO: implement


    static const size_t GetRequiredFileSize(int32_t sampleRate, int32_t channelCount, int32_t bitRate, nn::TimeSpan duration);

    // for debug
    void DumpInfo();

private:

#pragma pack(1) // to avoid unexpected padding
    struct RIFFHeader
    {
        uint32_t  RIFF;     // RIFF
        uint32_t  riffSize;
        uint32_t  riffKind; // WAVE
    };

    struct FormatChunk
    {
        uint32_t    fmtId; // fmt
        int32_t     chunkSize;
        uint16_t    id;
        int16_t     channelCount;
        int32_t     samplingFreq;
        int32_t     bytePerSec;
        int16_t     blockSize;
        int16_t     bitsPerSample;
        // skip these fields since only linear PCM supported.
        // int16_t     fmtExtSize;
        // int8_t      cbuf;
    };

    struct DataChunk
    {
        uint32_t dataId;
        int32_t  Nbyte;
    };

    struct WaveFileFormat
    {
        RIFFHeader  riff;
        FormatChunk fmt;
        DataChunk   data;
    };
#pragma pack()

    nn::fs::FileHandle m_Handle;
    nn::fs::OpenMode m_OpenMode;
    uint32_t m_SampleRate;
    uint16_t m_ChannelCount;
    uint16_t m_BitRate;
    uint32_t m_DataStartOffset;
    uint64_t m_CurrentReadWriteOffset;
    int64_t m_SampleDataSize;
    int64_t m_FileSize;

    WaveEndian m_Endian;
    int8_t m_WavFormat[sizeof(WaveFile::WaveFileFormat)];
    void SetWavInfo( int8_t* buf, const int32_t samplingFreq, const int16_t bits, const int16_t numChannels, const int64_t size, const WaveEndian endian );
    WaveFileResult GetWaveInfo( int8_t* buf,
                     uint32_t bufferSize,
                     uint16_t* nChannel,
                     uint32_t* samplingFreq,
                     uint16_t* bitsPerSample,
                     uint32_t* dataStartOffset,
                     int64_t* dataSize,
                     WaveEndian* endian);

    uint32_t CalcRiffChunkSize(uint32_t dataSize);
};

////////////////////////////////////////////////////////////////////////////////
// Audio Output Capture
////////////////////////////////////////////////////////////////////////////////
class Recorder
{
public:

    struct RecordItem
    {
        size_t size;
        int8_t* buffer;

        RecordItem(size_t _size, int8_t* _buffer)
            : size(_size)
            , buffer(_buffer) {}
    };

    Recorder();
    virtual ~Recorder();

    /**
    * @brief       Setup recorder setting.
    * @param[in]   param              Current AudioRenderer's parameter
    * @param[in]   pAux
    * @param[in]   pWaveFile
    * @param[in]   bufferSize         internal buffersize
    * @param[in]   recordingDuration  recording time period. [Sec]
    * @return      true if succeeded.
    * @pre
    * - pAux is already initialized.
    * - pWaveFile is already opened.
    * -
    */
    nn::Result Prepare(const nn::audio::AudioRendererParameter* param,
                       nn::audio::AuxType* pAux,
                       WaveFile* pWaveFile,
                       size_t bufferSize,
                       nn::TimeSpan recordingDuration,
                       void* recorderBufferr,
                       size_t recorderBufferSize);

    void EnableLeadSilenceTrimming(bool enable);

    virtual nn::Result Start();

    /**
    * @brief       write down sample data.
    * @detail
    * user must call this function so as not to interrupt audio frame.
    */
    virtual bool Record();

    /**
    * @brief       Stop recording.
    */
    virtual void Stop();



private:
    int32_t m_ChannelCount;
    int32_t m_FrameSampleCountPerChannel;
    int32_t m_FrameSampleCount;
    int32_t* m_ReadBuffer;
    size_t m_ReadBufferSize;
    size_t m_RecordingDurationSize;

    nn::audio::AuxType* m_pAux;
    WaveFile* m_pWaveFile;

    // for worker thread
    nn::os::ThreadType    m_WorkerThread;
    bool m_WorkerBooting;
    bool m_WorkerRunning;
    bool m_TrimLeadSilence;
    static const int MessageQueueSize = 512;
    uintptr_t m_MessageQueueBuffer[ MessageQueueSize ];
    nn::os::MessageQueueType m_MessageQueue;
    bool SendBufferToWorker(int8_t* buffer, size_t count);
    static void WorkerThreadFunction(void *arg);

    size_t m_TotalSendDataSize;

    int8_t* m_RecorderBuffer;
    size_t m_RecorderBufferSize;
    size_t m_RecorderBufferPosition;
};

////////////////////////////////////////////////////////////////////////////////
// Verification Rules
////////////////////////////////////////////////////////////////////////////////
class Checker
{
public:
    enum WaveFormCheckError
    {
        WaveFormCheckError_NoError = 0,
        WaveFormCheckError_SampleRate,
        WaveFormCheckError_ChannelCount,
        WaveFormCheckError_BitRate,
        WaveFormCheckError_DataSize,
        WaveFormCheckError_FileAccess,
        WaveFormCheckError_WaveForm,
    };

    WaveFormCheckError ParamCheck(const WaveFile* A, const WaveFile* B);
    WaveFormCheckError SimpleCheck(WaveFile* A, WaveFile* B, int32_t* status, int diff = 0);
    static std::string ResultString(WaveFormCheckError error, int status);
};


////////////////////////////////////////////////////////////////////////////////
// Test Foundation
////////////////////////////////////////////////////////////////////////////////

class SimpleFsUtility
{
private:
    char m_MountPathName[nn::fs::MountNameLengthMax + 1];

public:
    enum PathType
    {
        PathType_Directory,
        PathType_File
    };

    SimpleFsUtility();
    virtual ~SimpleFsUtility();
    nn::Result InitializeFileSystem(const char* name, const char* mountPath);
    nn::Result CreateDirectory(const char* path, PathType type);
    nn::Result PrepareNewFile(const char* filename, const int64_t filesize);
};

class MinimalRenderer
{
private:
    nn::audio::AudioRendererHandle m_handle;
    nn::audio::AudioRendererConfig m_config;
    nn::audio::SubMixType m_SubMix;
    nn::audio::FinalMixType m_finalMix;
    nn::audio::DeviceSinkType m_deviceSink;

    void* m_workBuffer;
    void* m_configBuffer;
    size_t m_workBufferSize;
    size_t m_configBufferSize;

    int m_channelCount;
    int8_t m_MainOutIndex[6];

    nn::mem::StandardAllocator m_Allocator;
    nn::os::SystemEvent m_Event;

public:
    MinimalRenderer(void* address, size_t size);

    virtual ~MinimalRenderer();
    void Initialize(nn::audio::AudioRendererParameter& params);
    void Start();
    void Stop();
    void Update();
    void Wait();
    void ExecuteRendering();

    nn::audio::SubMixType* GetSubMix();
    nn::audio::FinalMixType* GetFinalMix();
    nn::audio::AudioRendererConfig* GetConfig();
    nn::audio::AudioRendererHandle* GetHandle();
};

////////////////////////////////////////////////////////////////////////////////
// Utility Functions
////////////////////////////////////////////////////////////////////////////////
size_t GetWaveSampleSize(const std::string &SourceFile);
size_t LoadSourceSamples(const std::string &SourceFile, void* data, size_t size, int* sampleRate);
void SimpleVerify(const std::string &TargetFile, const std::string &GoldenFile, int diff = 0);
void GetMountPath(std::string& mountPath); // TODO: fs の実装が安定するまでの仮関数
const char* GetBuildType();
const char* GetTargetName();
std::string GetResultDataDirectory();


} // namespace nnt
} // namespace audio
} // namespace util
