﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/image/image_JpegDecoder.h>

#include <nn/image/image_JpegEncoder.h>
#include <image_LibjpegHelper.h>

#include "testImageJpeg_Io.h"

#include <cstdlib>
#include <cstring>

#include <nnt/nntest.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

namespace t = nnt::image::jpeg;

namespace
{

void Encode(
    size_t* pOutSize, const t::io::Buffer& outJpeg,
    const t::io::Buffer& pixel, nn::image::PixelFormat pxFormat, const nn::image::Dimension& dim,
    int quality, nn::image::JpegSamplingRatio ratio) NN_NOEXCEPT
{
    nn::image::JpegEncoder encoder;
    encoder.SetPixelData(pixel.GetPointer(), pxFormat, dim, 1);
    encoder.SetQuality(quality);
    encoder.SetSamplingRatio(ratio);

    ASSERT_EQ(nn::image::JpegStatus_Ok, encoder.Analyze());
    t::io::Buffer work(encoder.GetAnalyzedWorkBufferSize());

    ASSERT_EQ(nn::image::JpegStatus_Ok, encoder.Encode(
        pOutSize, outJpeg.GetPointer(), outJpeg.GetSize(),
        work.GetPointer(), work.GetSize()));
}
void Decode(
    const t::io::Buffer& pixel, const t::io::Buffer& jpeg,
    nn::image::PixelFormat pxFormat, int denom) NN_NOEXCEPT
{
    nn::image::JpegDecoder decoder;
    decoder.SetImageData(jpeg.GetPointer(), jpeg.GetSize());
    decoder.SetPixelFormat(pxFormat);
    decoder.SetResolutionDenominator(denom);

    ASSERT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());
    auto dim = decoder.GetAnalyzedDimension();
    NN_UNUSED(dim);
    t::io::Buffer work(decoder.GetAnalyzedWorkBufferSize());

    ASSERT_EQ(nn::image::JpegStatus_Ok, decoder.Decode(
        pixel.GetPointer(), pixel.GetSize(), 1, work.GetPointer(), work.GetSize()));
}

void TestDecodeSequential(
    const t::io::Buffer* jpegArray, int arrayLength, int offset,
    nn::image::PixelFormat pxFormat, int denom) NN_NOEXCEPT
{
    NN_ASSERT(arrayLength > offset);

    auto& head = jpegArray[offset];

    nn::image::JpegDecoder decoder;
    decoder.SetImageData(head.GetPointer(), head.GetSize());
    decoder.SetPixelFormat(pxFormat);
    decoder.SetResolutionDenominator(denom);

    ASSERT_EQ(nn::image::JpegStatus_Ok, decoder.Analyze());
    auto dim = decoder.GetAnalyzedDimension();
    t::io::Buffer work(decoder.GetAnalyzedWorkBufferSize());

    size_t pxSize = dim.width * dim.height * (pxFormat == nn::image::PixelFormat_Rgb24? 3: 4);
    t::io::Buffer out(pxSize);
    t::io::Buffer pixel(pxSize);

    for (int i = 0; i < arrayLength; ++ i)
    {
        auto index = (offset + i) % arrayLength;
        auto& jpeg = jpegArray[index];
        ASSERT_EQ(nn::image::JpegStatus_Ok, nn::image::JpegDecoder::DecodeWithPreconfiguredDecoder(
            out.GetPointer(), out.GetSize(), 1,
            jpeg.GetPointer(), jpeg.GetSize(),
            decoder,
            work.GetPointer(), work.GetSize()));

        Decode(pixel, jpeg, pxFormat, denom);
        ASSERT_EQ(0, std::memcmp(pixel.GetPointer(), out.GetPointer(), out.GetSize()));
    }
}

template <int ArrayLength>
struct TestConf
{
    nn::image::PixelFormat pxFormat;
    nn::image::Dimension dimension;
    struct Enc
    {
        int quality[ArrayLength];
        nn::image::JpegSamplingRatio ratio;
    } enc;
    struct Dec
    {
        int denominator;
    } dec;
};

template <int ArrayLength>
void RunTest(const TestConf<ArrayLength>& test) NN_NOEXCEPT
{
    std::srand(0);

    auto pxBytes = test.dimension.width * test.dimension.height * (test.pxFormat == nn::image::PixelFormat_Rgb24? 3: 4);
    t::io::Buffer pixel(pxBytes);
    t::io::Buffer jpegArray[ArrayLength];
    for (auto i = 0; i < ArrayLength; ++ i)
    {
        // カラーバッファの作成
        for (auto j = 0; j < pxBytes; ++ j)
        {
            auto px = static_cast<uint8_t>(std::rand() % 256);
            reinterpret_cast<uint8_t*>(pixel.GetPointer())[j] = px;
        }
        // JPEG データの作成
        size_t size;
        t::io::Buffer tmp(10 * 1024 * 1024);
        Encode(&size, tmp, pixel, test.pxFormat, test.dimension, test.enc.quality[i], test.enc.ratio);
        jpegArray[i].SetSize(size);
        std::memcpy(jpegArray[i].GetPointer(), tmp.GetPointer(), size);
    }

    for (int i = 0; i < ArrayLength; ++ i)
    {
        TestDecodeSequential(jpegArray, ArrayLength, i, test.pxFormat, test.dec.denominator);
    }
}

TEST(ImageJpegDecoder, Sequential_A)
{
    TestConf<10> conf = {
        nn::image::PixelFormat_Rgb24, {256, 256},
        {{1, 100, 96, 92, 88, 84, 80, 76, 72, 68}, nn::image::JpegSamplingRatio_444}, // enc
        {1} // dec
    };
    RunTest(conf);
}

TEST(ImageJpegDecoder, Sequential_B)
{
    TestConf<10> conf = {
        nn::image::PixelFormat_Rgba32, {1024, 1024},
        {{1, 100, 96, 92, 88, 84, 80, 76, 72, 68}, nn::image::JpegSamplingRatio_420}, // enc
        {2} // dec
    };
    RunTest(conf);
}

TEST(ImageJpegDecoder, Sequential_C)
{
    TestConf<10> conf = {
        nn::image::PixelFormat_Rgba32, {1024, 33},
        {{1, 100, 96, 92, 88, 84, 80, 76, 72, 68}, nn::image::JpegSamplingRatio_422}, // enc
        {1} // dec
    };
    RunTest(conf);
}


} // ~namespace <anonymous>
