﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/image/image_JpegEncoder.h>
#include <image_LibjpegHelper.h>

#include "testImageJpeg_Io.h"
#include "testImageJpeg_LibjpegApi.h"

#include <cstdlib>
#include <cstring>

#include <nnt/nntest.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>


#define NNT_IMAGE_ENABLE_JPEG_ENCODER_ASSERTION
#define NNT_IMAGE_ENABLE_JPEG_ENCODER_ENCODE
#define NNT_IMAGE_ENABLE_JPEG_ENCODER_RGB24

namespace t = nnt::image::jpeg;

namespace
{
bool ErrorOk(int errorCode, int reason)
{
    NN_UNUSED(reason);
    return errorCode == 0;
}

void SerializeColorBuffer(const t::io::Buffer& dst, const t::io::Buffer& src, nn::image::Dimension dim)
{
    auto alignedWidth = dst.GetSize() / sizeof(uint32_t) / dim.height;
    for (auto y = 0; y < dim.height; y++)
    {
        auto dstLine = reinterpret_cast<uint8_t*>(&dst.GetPointerAs<uint32_t>()[y * alignedWidth]);
        auto srcLine = &src.GetPointerAs<uint32_t>()[y * dim.width];
        for (auto x = 0; x < dim.width; x++)
        {
            int pxIndex = x * sizeof(nn::Bit32);
            dstLine[pxIndex + 0] = static_cast<uint8_t>((srcLine[x] & 0xFF000000) >> 24); // R
            dstLine[pxIndex + 1] = static_cast<uint8_t>((srcLine[x] & 0x00FF0000) >> 16); // G
            dstLine[pxIndex + 2] = static_cast<uint8_t>((srcLine[x] & 0x0000FF00) >>  8); // B
            dstLine[pxIndex + 3] = static_cast<uint8_t>((srcLine[x] & 0x000000FF));
        }
    }
};
void SerializeColorBufferRgb24(const t::io::Buffer& dst, const t::io::Buffer& src, nn::image::Dimension dim)
{
    auto alignedWidth = dst.GetSize() / 3 / dim.height;
    for (auto y = 0; y < dim.height; y++)
    {
        auto dstLine = &dst.GetPointerAs<uint8_t>()[y * alignedWidth * 3];
        auto srcLine = &src.GetPointerAs<uint32_t>()[y * dim.width];
        for (auto x = 0; x < dim.width; x++)
        {
            int pxIndex = x * 3;
            dstLine[pxIndex + 0] = static_cast<uint8_t>((srcLine[x] & 0xFF000000) >> 24); // R
            dstLine[pxIndex + 1] = static_cast<uint8_t>((srcLine[x] & 0x00FF0000) >> 16); // G
            dstLine[pxIndex + 2] = static_cast<uint8_t>((srcLine[x] & 0x0000FF00) >>  8); // B
        }
    }
};

struct TestCaseJpegEncoder
{
    const char *label;
    const t::io::Buffer* pxData;
    nn::image::Dimension dim;
    int quality;
    int samplingRatio;
    nn::image::ExifBuilder *pExifBuilder;
    int alignment;
};

void GenerateTruth(
    size_t* pOutActualSize,
    t::io::Buffer& jpegBuf,
    t::io::Buffer& exifBuf,
    const TestCaseJpegEncoder &test) NN_NOEXCEPT
{
    if (test.pExifBuilder != nullptr)
    {
        exifBuf.SetSize(test.pExifBuilder->GetAnalyzedOutputSize());
        test.pExifBuilder->Build(exifBuf.GetPointer(), exifBuf.GetSize(), test.dim);
        NN_LOG("    - Exif data: %d bytes\n", exifBuf.GetSize());
    }

    // testImageJpeg_Main.cpp でテスト済みの libjpeg 補助関数を使って正解データを生成
    nnt::image::jpeg::TestArgEncode arg = {
        ErrorOk, test.quality == -1? 100: test.quality, 0, 0};
    switch (test.samplingRatio)
    {
    case -1: // Implicit
    case nn::image::JpegSamplingRatio_420:
        arg.yHSample = 2;
        arg.yVSample = 2;
        break;
    case nn::image::JpegSamplingRatio_422:
        arg.yHSample = 2;
        arg.yVSample = 1;
        break;
    case nn::image::JpegSamplingRatio_444:
        arg.yHSample = 1;
        arg.yVSample = 1;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    jpegBuf.SetSize(10 * 1024 * 1024);

    t::io::Buffer workBuf(10 * 1024 * 1024);
    nnt::image::jpeg::Dimension testDim = {(uint16_t)test.dim.width, (uint16_t)test.dim.height};
    ASSERT_TRUE(
        nnt::image::jpeg::CompressBinary(
            pOutActualSize,
            jpegBuf.GetPointer(), jpegBuf.GetSize(),
            test.pxData->GetPointerAs<uint32_t>(),
            testDim,
            arg,
            workBuf.GetPointer(), workBuf.GetSize()));
};

bool CompareJpegWithExif(
    const t::io::Buffer& jpegBufTruth,
    size_t actualJpegSize,
    const t::io::Buffer& exifBufTruth,
    const t::io::Buffer& encodedJpeg)
{
    // バイナリの内容を比較
    NN_LOG("    - Comparing JPEG...\n");
    auto diffSoi = memcmp(jpegBufTruth.GetPointer(), encodedJpeg.GetPointer(), 2);
    EXPECT_EQ(0, diffSoi);
    auto diffPicture = memcmp(
            jpegBufTruth.GetPointer() + 2 + 18, // 2...SOI, 18...JFIF
            encodedJpeg.GetPointer() + 2 + 10 + exifBufTruth.GetSize(), // 2...SOI, 10...APP1(2)+SegmentLength(2)+Signature(6)
            actualJpegSize - 2 - 18); // 2...SOI, 18...JFIF
    EXPECT_EQ(0, diffPicture);
    NN_LOG("    - Comparing Exif...\n");
    uint8_t app1Data[] = {
        0xFF, 0xE1,
        static_cast<uint8_t>(((exifBufTruth.GetSize() + 8) & 0xFF00) >> 8),
        static_cast<uint8_t>(((exifBufTruth.GetSize() + 8) & 0xFF)),
        'E', 'x', 'i', 'f', 0x00, 0x00};
    auto diffApp1 = memcmp(app1Data, encodedJpeg.GetPointer() + 2, sizeof(app1Data));
    EXPECT_EQ(0, diffApp1);
    auto diffExif = memcmp(exifBufTruth.GetPointer(), encodedJpeg.GetPointer() + 2 + 10, exifBufTruth.GetSize());
    EXPECT_EQ(0, diffExif);

    return true
        && diffSoi == 0
        && diffPicture == 0
        && diffApp1 == 0
        && diffExif == 0;
}

}

#if defined(NNT_IMAGE_ENABLE_JPEG_ENCODER_ASSERTION)

TEST(ImageJpegEncoder, AssertionTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing nn::image::JpegEncoder pre-condition check\n");

    const uint32_t pxData[] = {
        0xFFFFFFFF, 0xFF0000FF, 0x00FF00FF, 0x0000FFFF,
        0xFFFFFFFF, 0xFFFF00FF, 0x00FFFFFF, 0xFF00FFFF,
        0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF, 0xFFFFFFFF,
        0xFFFFFFFF, 0x000000FF, 0xFFFFFFFF, 0x000000FF,
    }; // 4x4
    const nn::image::Dimension dim = {4, 4};
    const nn::image::Dimension dimX0 = {0, 4};
    const nn::image::Dimension dimY0 = {4, 0};

    nn::image::JpegEncoder encoder;
    size_t jpegSize;
    void *workBuf = malloc(10 * 1024 * 1024);
    void *jpegData = malloc(10 * 1024 * 1024);

    // Invalid data specification
    NN_LOG("  - Test ID: %s\n", "Initialization");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetPixelData(nullptr, dim, 1);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetPixelData(pxData, dimX0, 1);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetPixelData(pxData, dimY0, 1);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetPixelData(pxData, dim, 0);}, ".*");

    // Unregistered
    NN_LOG("  - Test ID: %s\n", "Registration check");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Analyze();}, ".*");

    // Bad parameter
    NN_LOG("  - Test ID: %s\n", "Encoding params");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetQuality(0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetQuality(101);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetSamplingRatio((nn::image::JpegSamplingRatio) - 1);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.SetSamplingRatio((nn::image::JpegSamplingRatio)3);}, ".*");

    encoder.SetPixelData(pxData, dim, 1);

    // Unanalyzed
    NN_LOG("  - Test ID: %s\n", "Analyzation check");
    size_t exifWorkBufSize = nn::image::ExifBuilder::GetWorkBufferSize();
    void *exifWorkBuf = malloc(exifWorkBufSize);
    nn::image::ExifBuilder exifBuilder(exifWorkBuf, exifWorkBufSize);
    NN_ASSERT(exifBuilder.Analyze() == nn::image::JpegStatus_Ok);
    EXPECT_DEATH_IF_SUPPORTED({encoder.GetAnalyzedWorkBufferSize();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, 10 * 1024 * 1024);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, 10 * 1024 * 1024, &exifBuilder);}, ".*");

    EXPECT_EQ(nn::image::JpegStatus_Ok, encoder.Analyze());

    encoder.SetPixelData(pxData, dim, 1);

    // Unanalyzed (again)
    EXPECT_DEATH_IF_SUPPORTED({encoder.GetAnalyzedWorkBufferSize();}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, 10 * 1024 * 1024);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, 10 * 1024 * 1024, &exifBuilder);}, ".*");

    EXPECT_EQ(nn::image::JpegStatus_Ok, encoder.Analyze());

    size_t workBufSize = encoder.GetAnalyzedWorkBufferSize();
    NN_ASSERT(workBufSize <= 10 * 1024 * 1024);

    // Bad argument for Encode method.
    NN_LOG("  - Test ID: %s\n", "Arguments for 'Decode' method");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(nullptr, jpegData, 10 * 1024 * 1024, workBuf, workBufSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, nullptr, 10 * 1024 * 1024, workBuf, workBufSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 0, workBuf, workBufSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, nullptr, workBufSize);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, workBufSize - 1);}, ".*");

    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(nullptr, jpegData, 10 * 1024 * 1024, workBuf, workBufSize, nullptr);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, nullptr, 10 * 1024 * 1024, workBuf, workBufSize, nullptr);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 0, workBuf, workBufSize, nullptr);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, nullptr, workBufSize, &exifBuilder);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, workBufSize - 1, &exifBuilder);}, ".*");
    exifBuilder.SetOrientation(nn::image::ExifOrientation_Normal);
    EXPECT_DEATH_IF_SUPPORTED({encoder.Encode(&jpegSize, jpegData, 10 * 1024 * 1024, workBuf, workBufSize, &exifBuilder);}, ".*");

    free(workBuf);
    free(jpegData);
} // NOLINT(readability/fn_size)

#endif // NNT_IMAGE_ENABLE_JPEG_ENCODER_ASSERTION

#if defined(NNT_IMAGE_ENABLE_JPEG_ENCODER_ENCODE)

TEST(ImageJpegEncoder, EncodeTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG encoder\n");

    // ピクセルデータの初期化
    t::io::Buffer pxDataUE01(sizeof(uint32_t) * 640 * 480);
    t::io::Buffer pxDataUE02(sizeof(uint32_t) * 320 * 240);
    t::io::Buffer pxDataUE03(sizeof(uint32_t) * 320 * 240);
    t::io::Buffer pxDataUE04(sizeof(uint32_t) * 640 * 480);
    for (int i = 0; i < 640 * 480; i++)
    {
        pxDataUE01.GetPointerAs<uint32_t>()[i] = 0xFFFFFFFF;
        pxDataUE04.GetPointerAs<uint32_t>()[i] = 0xFF0000FF;
        if (i < 320 * 240)
        {
            pxDataUE02.GetPointerAs<uint32_t>()[i] = 0x00FF00FF;
            pxDataUE03.GetPointerAs<uint32_t>()[i] = 0x0000FFFF;
        }
    }

    // U-E-02 用 ExifBuilder の設定
    t::io::Buffer exifWorkBufUE02(nn::image::ExifBuilder::GetWorkBufferSize());
    nn::image::ExifBuilder exifBuilderUE02(exifWorkBufUE02.GetPointer(), exifWorkBufUE02.GetSize());
    ASSERT_EQ(nn::image::JpegStatus_Ok, exifBuilderUE02.Analyze());
    size_t exifSizeUE02 = exifBuilderUE02.GetAnalyzedOutputSize();

    // U-E-03 用 ExifBuilder の設定
    t::io::Buffer exifWorkBufUE03(nn::image::ExifBuilder::GetWorkBufferSize());
    nn::image::ExifBuilder exifBuilderUE03(exifWorkBufUE03.GetPointer(), exifWorkBufUE03.GetSize());
    t::io::Buffer makerNote(0xFFF7 - 12 - exifSizeUE02);
    std::memset(makerNote.GetPointer(), 0x3A, makerNote.GetSize());
    exifBuilderUE03.SetMakerNote(makerNote.GetPointer(), makerNote.GetSize());
    ASSERT_EQ(nn::image::JpegStatus_Ok, exifBuilderUE03.Analyze());
    size_t exifSizeUE03 = exifBuilderUE03.GetAnalyzedOutputSize();
    ASSERT_EQ(0xFFF7, exifSizeUE03);

    // テストケースを定義
    TestCaseJpegEncoder normalCases[] = {
        {"U-E-01", &pxDataUE01, {640, 480}, -1, -1, nullptr, 1},
        {"U-E-02", &pxDataUE02, {320, 240}, -1, nn::image::JpegSamplingRatio_444, &exifBuilderUE02, 3},
        {"U-E-03", &pxDataUE03, {320, 240}, 10, -1, &exifBuilderUE03, 8},
        {"U-E-04", &pxDataUE04, {640, 480}, 100, nn::image::JpegSamplingRatio_422, nullptr, 13},
    };

    for (int i = 0; i < static_cast<int>(sizeof(normalCases) / sizeof(TestCaseJpegEncoder)); i++)
    {
        TestCaseJpegEncoder &test = normalCases[i];

        NN_LOG("  - Test ID: %s\n", test.label);

        /** -----------------------------------------------------------------------------------
            正解データの作成
         */
        // ExifBuilder は正しいと仮定して、Exif 情報の正解データを生成
        // (ExifBuilder は testImageJpeg_ExifParser プロジェクトでテスト済み。)
        t::io::Buffer jpegBufTruth;
        t::io::Buffer exifBufTruth;

        size_t actualSizeTruth;
        GenerateTruth(&actualSizeTruth, jpegBufTruth, exifBufTruth, test);
        NN_LOG("    - Truth: %d bytes\n", actualSizeTruth);

        /** -----------------------------------------------------------------------------------
            評価対象のデータの作成
         */

        // JpegEncoder 用ピクセルデータ (アライン済み) を生成
        int alignedWidth = test.dim.width + (test.dim.width % test.alignment == 0? 0: - (test.dim.width % test.alignment) + test.alignment);
        t::io::Buffer rawPixelData(sizeof(uint32_t) * alignedWidth * test.dim.height);
        // Encoder に与えるデータは エンディアンに関係なく R,G,B,A の順でなければならない。
        SerializeColorBuffer(rawPixelData, *test.pxData, test.dim);

        nn::image::JpegEncoder encoder;
        encoder.SetPixelData(rawPixelData.GetPointer(), test.dim, test.alignment);
        if (test.quality >= 0)
        {
            encoder.SetQuality(test.quality);
        }
        if (test.samplingRatio >= 0)
        {
            encoder.SetSamplingRatio((nn::image::JpegSamplingRatio)test.samplingRatio);
        }
        ASSERT_EQ(nn::image::JpegStatus_Ok, encoder.Analyze());

        t::io::Buffer encodedJpeg(10 * 1024 * 1024);
        t::io::Buffer workBuf(10 * 1024 * 1024);
        size_t actualSizeTest;
        if (test.pExifBuilder != nullptr)
        {
            // Exif 情報あり版
            EXPECT_EQ(
                nn::image::JpegStatus_Ok,
                encoder.Encode(
                    &actualSizeTest,
                    encodedJpeg.GetPointer(), encodedJpeg.GetSize(),
                    workBuf.GetPointer(), workBuf.GetSize(),
                    test.pExifBuilder));
            NN_LOG("    - Test(+Exif): %d bytes\n", actualSizeTest);
            EXPECT_EQ(actualSizeTruth - 18 + 10 + exifBufTruth.GetSize(), actualSizeTest);

            EXPECT_TRUE(CompareJpegWithExif(jpegBufTruth, actualSizeTruth, exifBufTruth, encodedJpeg));
        }
        else
        {
            // Exif 情報なし版
            EXPECT_EQ(
                nn::image::JpegStatus_Ok,
                encoder.Encode(
                    &actualSizeTest,
                    encodedJpeg.GetPointer(), encodedJpeg.GetSize(),
                    workBuf.GetPointer(), workBuf.GetSize()));
            NN_LOG("    - Test: %d bytes\n", actualSizeTest);
            EXPECT_EQ(actualSizeTruth, actualSizeTest);

            NN_LOG("    - Comparing JPEG...\n");
            EXPECT_EQ(0, memcmp(jpegBufTruth.GetPointer(), encodedJpeg.GetPointer(), actualSizeTruth));
        }
    }
} // NOLINT(readability/fn_size)

#endif // NNT_IMAGE_ENABLE_JPEG_ENCODER_ENCODE

#if defined(NNT_IMAGE_ENABLE_JPEG_ENCODER_RGB24)

TEST(ImageJpegEncoder, EncodeTestRgb24)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG encoder\n");

    // ピクセルデータの初期化
    t::io::Buffer pxDataUE01(sizeof(uint32_t) * 640 * 480);
    t::io::Buffer pxDataUE02(sizeof(uint32_t) * 320 * 240);
    t::io::Buffer pxDataUE03(sizeof(uint32_t) * 320 * 240);
    t::io::Buffer pxDataUE04(sizeof(uint32_t) * 640 * 480);
    for (int i = 0; i < 640 * 480; i++)
    {
        pxDataUE01.GetPointerAs<uint32_t>()[i] = 0xFFFFFFFF;
        pxDataUE04.GetPointerAs<uint32_t>()[i] = 0xFF0000FF;
        if (i < 320 * 240)
        {
            pxDataUE02.GetPointerAs<uint32_t>()[i] = 0x00FF00FF;
            pxDataUE03.GetPointerAs<uint32_t>()[i] = 0x0000FFFF;
        }
    }

    // U-E-02 用 ExifBuilder の設定
    t::io::Buffer exifWorkBufUE02(nn::image::ExifBuilder::GetWorkBufferSize());
    nn::image::ExifBuilder exifBuilderUE02(exifWorkBufUE02.GetPointer(), exifWorkBufUE02.GetSize());
    ASSERT_EQ(nn::image::JpegStatus_Ok, exifBuilderUE02.Analyze());
    size_t exifSizeUE02 = exifBuilderUE02.GetAnalyzedOutputSize();

    // U-E-03 用 ExifBuilder の設定
    t::io::Buffer exifWorkBufUE03(nn::image::ExifBuilder::GetWorkBufferSize());
    nn::image::ExifBuilder exifBuilderUE03(exifWorkBufUE03.GetPointer(), exifWorkBufUE03.GetSize());
    t::io::Buffer makerNote(0xFFF7 - 12 - exifSizeUE02);
    std::memset(makerNote.GetPointer(), 0x3A, makerNote.GetSize());
    exifBuilderUE03.SetMakerNote(makerNote.GetPointer(), makerNote.GetSize());
    ASSERT_EQ(nn::image::JpegStatus_Ok, exifBuilderUE03.Analyze());
    size_t exifSizeUE03 = exifBuilderUE03.GetAnalyzedOutputSize();
    ASSERT_EQ(0xFFF7, exifSizeUE03);

    // テストケースを定義
    TestCaseJpegEncoder normalCases[] = {
        {"U-E-01", &pxDataUE01, {640, 480}, -1, -1, nullptr, 1},
        {"U-E-02", &pxDataUE02, {320, 240}, -1, nn::image::JpegSamplingRatio_444, &exifBuilderUE02, 3},
        {"U-E-03", &pxDataUE03, {320, 240}, 10, -1, &exifBuilderUE03, 8},
        {"U-E-04", &pxDataUE04, {640, 480}, 100, nn::image::JpegSamplingRatio_422, nullptr, 13},
    };

    for (int i = 0; i < static_cast<int>(sizeof(normalCases) / sizeof(TestCaseJpegEncoder)); i++)
    {
        TestCaseJpegEncoder &test = normalCases[i];

        NN_LOG("  - Test ID: %s\n", test.label);

        /** -----------------------------------------------------------------------------------
            正解データの作成
         */
        // ExifBuilder は正しいと仮定して、Exif 情報の正解データを生成
        // (ExifBuilder は testImageJpeg_ExifParser プロジェクトでテスト済み。)
        t::io::Buffer jpegBufTruth;
        t::io::Buffer exifBufTruth;

        size_t actualSizeTruth;
        GenerateTruth(&actualSizeTruth, jpegBufTruth, exifBufTruth, test);
        NN_LOG("    - Truth: %d bytes\n", actualSizeTruth);

        /** -----------------------------------------------------------------------------------
            評価対象のデータの作成
         */

        // JpegEncoder 用ピクセルデータ (アライン済み) を生成
        int alignedWidth = test.dim.width + (test.dim.width % test.alignment == 0? 0: - (test.dim.width % test.alignment) + test.alignment);
        t::io::Buffer rawPixelData(sizeof(uint8_t) * 3 * alignedWidth * test.dim.height);
        // Encoder に与えるデータは エンディアンに関係なく R,G,B,A の順でなければならない。
        SerializeColorBufferRgb24(rawPixelData, *test.pxData, test.dim);

        nn::image::JpegEncoder encoder;
        encoder.SetPixelData(rawPixelData.GetPointer(), nn::image::PixelFormat_Rgb24, test.dim, test.alignment);
        if (test.quality >= 0)
        {
            encoder.SetQuality(test.quality);
        }
        if (test.samplingRatio >= 0)
        {
            encoder.SetSamplingRatio((nn::image::JpegSamplingRatio)test.samplingRatio);
        }
        ASSERT_EQ(nn::image::JpegStatus_Ok, encoder.Analyze());

        t::io::Buffer encodedJpeg(10 * 1024 * 1024);
        t::io::Buffer workBuf(10 * 1024 * 1024);
        size_t actualSizeTest;
        if (test.pExifBuilder != nullptr)
        {
            // Exif 情報あり版
            EXPECT_EQ(
                nn::image::JpegStatus_Ok,
                encoder.Encode(
                    &actualSizeTest,
                    encodedJpeg.GetPointer(), encodedJpeg.GetSize(),
                    workBuf.GetPointer(), workBuf.GetSize(),
                    test.pExifBuilder));
            NN_LOG("    - Test(+Exif): %d bytes\n", actualSizeTest);
            EXPECT_EQ(actualSizeTruth - 18 + 10 + exifBufTruth.GetSize(), actualSizeTest);

            EXPECT_TRUE(CompareJpegWithExif(jpegBufTruth, actualSizeTruth, exifBufTruth, encodedJpeg));
        }
        else
        {
            // Exif 情報なし版
            EXPECT_EQ(
                nn::image::JpegStatus_Ok,
                encoder.Encode(
                    &actualSizeTest,
                    encodedJpeg.GetPointer(), encodedJpeg.GetSize(),
                    workBuf.GetPointer(), workBuf.GetSize()));
            NN_LOG("    - Test: %d bytes\n", actualSizeTest);
            EXPECT_EQ(actualSizeTruth, actualSizeTest);

            NN_LOG("    - Comparing JPEG...\n");
            EXPECT_EQ(0, memcmp(jpegBufTruth.GetPointer(), encodedJpeg.GetPointer(), actualSizeTruth));
        }
    }
} // NOLINT(readability/fn_size)

#endif // NNT_IMAGE_ENABLE_JPEG_ENCODER_RGB24
