﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net.Sockets;
using System.Threading;
using System.IO;

using Nintendo.Htcs;

namespace SleepTestTool
{
    internal class SleepTest
    {
        private readonly string m_KeyString;
        private readonly HtcsCommunicator m_HtcsCommunicator;
        private readonly string m_IpAddress;
        private readonly string m_SerialNumber;
        private readonly ManualResetEvent m_Event;

        const int DataSize = 256 * 1024 * 1024;

        public SleepTest(string keyString, HtcsCommunicator htcsCommunicator, string ipAddress, string serialNumber)
        {
            m_KeyString = keyString;
            m_HtcsCommunicator = htcsCommunicator;
            m_IpAddress = ipAddress;
            m_SerialNumber = serialNumber;
            m_Event = new ManualResetEvent(false);
        }

        public void SleepTrigger()
        {
            m_Event.WaitOne();
            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);
        }

        public void Recv(TcpClient client, byte[] recvData)
        {
            using (var reader = new BinaryReader(client.GetStream(), Encoding.ASCII, true))
            {
                int snipSize = 256 * 1024;
                int recievedSize = 0;
                int typeSize = System.Runtime.InteropServices.Marshal.SizeOf(recvData.GetType().GetElementType());
                while (recievedSize < DataSize)
                {
                    byte[] data = reader.ReadBytes(snipSize);
                    if (data.Length != snipSize)
                    {
                        Console.WriteLine("Error: {0} bytes expected, but {1} bytes received.", DataSize, data.Length);
                    }
                    Buffer.BlockCopy(data, 0, recvData, recievedSize * typeSize, data.Length * typeSize);
                    recievedSize += data.Length;
                    Console.WriteLine("Recv data {0} bytes ({1}/{2})", data.Length, recievedSize, DataSize);
                }
            }
        }

        public void Send(TcpClient client, byte[] sendData)
        {
            using (var writer = new BinaryWriter(client.GetStream(), Encoding.ASCII, true))
            {
                // Bulk Send
                writer.Write(sendData);
                writer.Flush();
            }
        }

        public void SleepAfterCreateSocket(string name)
        {
            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            Send(client, recvData);

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepAfterConnected(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            Send(client, recvData);

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepBeforeTransmission(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            byte[] recvData = new byte[DataSize];
            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);
            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);
            Send(client, recvData);

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepBeforeTransmission_HostContinue(string name)
        {
            var thread = new Thread(SleepTrigger);
            m_Event.Reset();
            thread.Start();

            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            byte[] recvData = new byte[DataSize];

            m_Event.Set();
            System.Threading.Thread.Sleep(5000);

            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            thread.Join();
            thread = new Thread(SleepTrigger);
            thread.Start();

            Send(client, recvData);

            thread.Join();
            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepWhileTransmission(string name)
        {
            var thread = new Thread(SleepTrigger);
            m_Event.Reset();
            thread.Start();

            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            byte[] recvData = new byte[DataSize];

            using (var reader = new BinaryReader(client.GetStream(), Encoding.ASCII, true))
            {
                int snipSize = 256 * 1024;
                int recievedSize = 0;
                int typeSize = System.Runtime.InteropServices.Marshal.SizeOf(recvData.GetType().GetElementType());
                int count = 0;
                while (recievedSize < DataSize)
                {
                    byte[] data = reader.ReadBytes(snipSize);
                    if (data.Length != snipSize)
                    {
                        Console.WriteLine("Error: {0} bytes expected, but {1} bytes received.", DataSize, data.Length);
                    }
                    Buffer.BlockCopy(data, 0, recvData, recievedSize * typeSize, data.Length * typeSize);
                    recievedSize += data.Length;
                    Console.WriteLine("Recv data {0} bytes ({1}/{2})", data.Length, recievedSize, DataSize);

                    if (count == 5)
                    {
                        m_Event.Set();
                    }
                    count++;
                }
            }

            byte[] ack = { 0 };
            Send(client, ack);

            thread = new Thread(SleepTrigger);
            m_Event.Reset();
            thread.Start();

            m_Event.Set();
            Send(client, recvData);

            thread.Join();

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepAfterPortOpened(string name)
        {
            string portName = name + m_KeyString;
            TcpClient client = new TcpClient();
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == portName);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }
            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            client.Connect(portMapItem.EndPoint);

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            Send(client, recvData);

            client.Close();
        }

        public void SleepAfterPortOpened_HostContinue(string name)
        {
            var thread = new Thread(SleepTrigger);
            m_Event.Reset();
            thread.Start();

            string portName = name + m_KeyString;
            TcpClient client = new TcpClient();
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == portName);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }

            m_Event.Set();
            Thread.Sleep(5000);

            client.Connect(portMapItem.EndPoint);

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            Send(client, recvData);

            thread.Join();

            client.Close();
        }

        public void ShutdownRd(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);

            Thread.Sleep(10000);

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void ShutdownWr(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            byte[] ack = { 0 };
            Send(client, ack);

            Thread.Sleep(10000);

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void ShutdownRw(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            Thread.Sleep(10000);

            client.Close();
            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepAndCloseAfterTargetClientConnected(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            ControlTargetUtil.Sleep(m_IpAddress);

            Thread.Sleep(15 * 1000);
            client.Close();

            ControlTargetUtil.Wake(m_IpAddress, m_SerialNumber);

            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void SleepAndCloseAfterTargetServerConnected(string name)
        {
            string portName = name + m_KeyString;
            TcpClient client = new TcpClient();
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == portName);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }
            client.Connect(portMapItem.EndPoint);

            ControlTargetUtil.Sleep(m_IpAddress);

            Thread.Sleep(15 * 1000);
            client.Close();

            ControlTargetUtil.Wake(m_IpAddress, m_SerialNumber);

            Thread.Sleep(10000);

            client.Close();
        }

        public void SleepWhileRecv_HostClose(string name)
        {
            var listener = new TcpListener(System.Net.IPAddress.Loopback, 0);
            listener.Start();
            string portName = name + m_KeyString;
            var portDesc = new HtcsPortDescriptor(HtcsPeerName.Any, portName);
            var portMapping = new PortMapItem(portDesc, ((System.Net.IPEndPoint)listener.LocalEndpoint));
            while (m_HtcsCommunicator.RegisterPort(portMapping) != 0)
            {
                Console.WriteLine("Registration error: {0}.", portName);
                Thread.Sleep(1000);
            }

            TcpClient client;
            client = listener.AcceptTcpClient();

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);

            byte[] ack = { 0 };
            Send(client, ack);

            ControlTargetUtil.Sleep(m_IpAddress);

            Thread.Sleep(15 * 1000);
            client.Close();

            ControlTargetUtil.Wake(m_IpAddress, m_SerialNumber);

            listener.Stop();
            m_HtcsCommunicator.UnregisterPort(portDesc);
        }

        public void NonBlocking(string name)
        {
            string portName = name + m_KeyString;
            TcpClient client = new TcpClient();
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == portName);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }
            ControlTargetUtil.SleepAndWake(m_IpAddress, m_SerialNumber);

            Thread.Sleep(10000);

            client.Connect(portMapItem.EndPoint);

            byte[] recvData = new byte[DataSize];
            Recv(client, recvData);
            Send(client, recvData);
            client.Close();
        }

        public void SleepAfterConnected_HostReconnect(string name)
        {
            string portName = name + m_KeyString;
            TcpClient client = new TcpClient();
            PortMapItem portMapItem;
            while (true)
            {
                try
                {
                    portMapItem = m_HtcsCommunicator.PortMap.First(x => x.HtcsPortDescriptor.HtcsPortName == portName);
                    break;
                }
                catch (InvalidOperationException ioe)
                {
                    Console.WriteLine("Invalid operation occurred({0:X8})", ioe.HResult);
                }

                // 例外発生時、しばらく待ってやり直す
                Thread.Sleep(1000);
            }
            client.Connect(portMapItem.EndPoint);

            ControlTargetUtil.Sleep(m_IpAddress);

            Thread.Sleep(15 * 1000);
            client.Close();
            client = new TcpClient();
            client.Connect(portMapItem.EndPoint);

            ControlTargetUtil.Wake(m_IpAddress, m_SerialNumber);

            Thread.Sleep(10000);

            client.Close();
        }
    }
}
