﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <map>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt.h>

/*
*   様々な通信を同時に行うストレステストです。
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    const size_t MaxThreadCount = 16;
    const size_t ThreadStackSize = 16 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStacks[MaxThreadCount][ThreadStackSize];

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    enum TestPattern
    {
        TestPattern_RecvClient = 0,
        TestPattern_SendClient,
        TestPattern_EchoClient,
        TestPattern_RecvServer,
        TestPattern_SendServer,
        TestPattern_EchoServer,
        TestPattern_NonBlockingAccept,
        TestPattern_RepeatConnect,
        TestPattern_HostFs,

        TestPattern_CountMax
    };

    std::map<TestPattern, const char*> testNameMap =
    {
        { TestPattern_RecvClient, "RecvClient" },
        { TestPattern_SendClient, "SendClient" },
        { TestPattern_EchoClient, "EchoClient" },
        { TestPattern_RecvServer, "RecvServer" },
        { TestPattern_SendServer, "SendServer" },
        { TestPattern_EchoServer, "EchoServer" },
        { TestPattern_NonBlockingAccept, "NonBlockingAccept" },
        { TestPattern_RepeatConnect, "RepeatConnect" },
        { TestPattern_HostFs, "HostFs" },
    };

    struct TestParams
    {
        int threadNum;
        int socket;
    };

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    uint32_t CreateRandomValue()
    {
        static uint64_t x = static_cast<uint64_t>(nn::os::GetSystemTick().GetInt64Value());
        x = x ^ (x << 13);
        x = x ^ (x >> 7);
        x = x ^ (x << 17);
        return static_cast<uint32_t>(x);
    }

    class Stress : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = CreateRandomValue() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    void NonBlockingAcceptThread()
    {
        int listenSocket = nn::htcs::Socket();
        ASSERT_LE(0, listenSocket);

        nn::htcs::SockAddrHtcs addr;
        addr.family = nn::htcs::HTCS_AF_HTCS;
        addr.peerName = nn::htcs::GetPeerNameAny();
        std::strcpy(addr.portName.name, "StressAccept");

        EXPECT_EQ(0, nn::htcs::Fcntl(listenSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));

        EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &addr));
        EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

        for (int i = 0; i < 100; ++i)
        {
            EXPECT_GT(0, nn::htcs::Accept(listenSocket, nullptr));
            int error = nn::htcs::GetLastError();
            EXPECT_TRUE(error == nn::htcs::HTCS_EWOULDBLOCK || error == nn::htcs::HTCS_EINTR);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
        nn::htcs::Close(listenSocket);
    }

    void RepeatConnect(nn::htcs::SockAddrHtcs* pAddr)
    {
        int socket = nn::htcs::Socket();
        for (int i = 0; i < 1000; ++i)
        {
            EXPECT_GT(0, nn::htcs::Connect(socket, pAddr));
            EXPECT_EQ(nn::htcs::HTCS_EADDRNOTAVAIL, nn::htcs::GetLastError());
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(17));
        }
        nn::htcs::Close(socket);
    }

    void HostFsThread(const char* portName)
    {
        nn::Result result;

        for (int count = 0; count < 10; ++count)
        {
            static const size_t FileSize = 512;

            size_t bufferSize = std::strlen(portName) + 16;
            char* rootDirPath = reinterpret_cast<char*>(std::malloc(bufferSize));
            char* dirPath = reinterpret_cast<char*>(std::malloc(bufferSize));
            char* filePath = reinterpret_cast<char*>(std::malloc(bufferSize));
            ASSERT_NE(nullptr, rootDirPath);
            ASSERT_NE(nullptr, dirPath);
            ASSERT_NE(nullptr, filePath);
            rootDirPath[0] = '\0';
            dirPath[0] = '\0';
            filePath[0] = '\0';

            std::strcat(rootDirPath, "host:/");
            std::strcat(rootDirPath, portName);

            std::strcat(dirPath, rootDirPath);
            std::strcat(dirPath, "/dir1");

            std::strcat(filePath, rootDirPath);
            std::strcat(filePath, "/file1");

            // ディレクトリ・ファイルを作成します。
            {
                NN_LOG("Create\t%s\n", rootDirPath);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(rootDirPath));

                NN_LOG("Create\t%s\n", dirPath);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(dirPath));

                NN_LOG("Create\t%s\n", filePath);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(filePath, FileSize));
            }

            // ディレクトリを操作します。
            {
                NN_LOG("Show entries in %s\n", rootDirPath);
                nn::fs::DirectoryHandle directoryHandle;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, rootDirPath, nn::fs::OpenDirectoryMode_All));

                nn::fs::DirectoryEntry directoryEntry[2];
                int64_t readNum = 0;

                (void)nn::fs::ReadDirectory(&readNum, directoryEntry, directoryHandle, 2);

                for (int i = 0; i < readNum; i++)
                {
                    NN_LOG("%10s <type = %4s, size = %lld>\n", directoryEntry[i].name,
                        directoryEntry[i].directoryEntryType == nn::fs::DirectoryEntryType_Directory ? "Dir" : "File",
                        directoryEntry[i].fileSize);
                }

                nn::fs::CloseDirectory(directoryHandle);
            }

            // ファイルを操作します。
            {
                nn::fs::FileHandle fileHandle;
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, filePath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

                static const size_t BufferSize = 1024;
                static char writeBuffer[BufferSize];
                for (int i = 0; i < BufferSize; i++)
                {
                    writeBuffer[i] = static_cast<char>(i);
                }
                NN_LOG("Write\t%s\n", filePath);
                result = nn::fs::WriteFile(fileHandle, 0, writeBuffer, BufferSize, nn::fs::WriteOption());
                if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    NN_ASSERT("Usable space not enough.\n");
                    return;
                }

                NN_LOG("Flush\t%s\n", filePath);

                (void)nn::fs::FlushFile(fileHandle);

                int64_t fileSize = 0;
                NN_LOG("Read\t%s\n", filePath);

                (void)nn::fs::GetFileSize(&fileSize, fileHandle);

                NN_ASSERT_EQUAL(fileSize, 1024);

                static char readBuffer[BufferSize];

                (void)nn::fs::ReadFile(fileHandle, 0, readBuffer, static_cast<size_t>(fileSize));

                NN_ASSERT(std::memcmp(writeBuffer, readBuffer, BufferSize) == 0);

                nn::fs::CloseFile(fileHandle);
            }

            // ディレクトリ・ファイルを削除します。
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(rootDirPath));

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));

            std::free(rootDirPath);
            std::free(dirPath);
            std::free(filePath);
        }

    }

    void DoRandomTest(void* pArgs)
    {
        TestParams* params = reinterpret_cast<TestParams*>(pArgs);
        for (int i = 0; i < 50; ++i)
        {
            nnt::htcs::util::ThreadArgs args;
            // ランダムにテストを選択
            TestPattern seed = static_cast<TestPattern>(CreateRandomValue() % TestPattern_CountMax);

            NN_LOG("Thread[%d]: Start %s\n", params->threadNum, testNameMap[seed]);

            // テスト設定
            nn::util::SNPrintf(args.address.portName.name, sizeof(args.address.portName), "Stress%d_%d_%s", params->threadNum, i, g_KeyString);
            args.pBarrier = nullptr;
            args.size = CreateRandomValue() % (1024 * 1024) + 1;
            args.count = CreateRandomValue() % 100 + 1;
            args.verbose = false;

            const char* testName = testNameMap[seed];
            size_t bufferSize = 128;
            char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
            memset(buffer, 0, bufferSize);

            nn::util::SNPrintf(buffer, bufferSize, "%d,%d,%s,%d,%d\n", params->threadNum, i, testName, (int)args.size, (int)args.count);

            size_t sendSize = strlen(buffer);
            ASSERT_TRUE(nnt::htcs::util::Send(params->socket, sendSize, reinterpret_cast<uint8_t*>(buffer)));
            std::free(buffer);

            switch (seed)
            {
            case TestPattern_RecvClient:
                args.mode = nnt::htcs::util::ThreadMode_Recv;
                nnt::htcs::util::ClientThread(&args);
                break;
            case TestPattern_SendClient:
                args.mode = nnt::htcs::util::ThreadMode_Send;
                nnt::htcs::util::ClientThread(&args);
                break;
            case TestPattern_EchoClient:
                args.mode = nnt::htcs::util::ThreadMode_Echo;
                nnt::htcs::util::ClientThread(&args);
                break;
            case TestPattern_RecvServer:
                args.mode = nnt::htcs::util::ThreadMode_Recv;
                nnt::htcs::util::ServerThread(&args);
                break;
            case TestPattern_SendServer:
                args.mode = nnt::htcs::util::ThreadMode_Send;
                nnt::htcs::util::ServerThread(&args);
                break;
            case TestPattern_EchoServer:
                args.mode = nnt::htcs::util::ThreadMode_Echo;
                nnt::htcs::util::ServerThread(&args);
                break;
            case TestPattern_NonBlockingAccept:
                NonBlockingAcceptThread();
                break;
            case TestPattern_RepeatConnect:
                RepeatConnect(&args.address);
                break;
            case TestPattern_HostFs:
                HostFsThread(args.address.portName.name);
                break;
            default:
                break;
            }

            NN_LOG("Thread[%d]: End %s\n", params->threadNum, testNameMap[seed]);

            // ポートの削除などが完了してない可能性があるので、少し待機
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        }
    }
}

TEST_F(Stress, Start)
{
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "Stress";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

    nn::htcs::Close(socket);

    std::free(buffer);
}

TEST_F(Stress, TestRandamParallel)
{
    NN_LOG("Mount C:/Windows/Temp to host\n");
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHost("host", "C:/Windows/Temp"));

    const size_t ThreadCount = 16;
    ASSERT_GE(MaxThreadCount, ThreadCount);

    nn::os::ThreadType threads[ThreadCount];
    TestParams params[ThreadCount];

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);
    nn::htcs::SockAddrHtcs addr;
    addr.family = nn::htcs::HTCS_AF_HTCS;
    addr.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(addr.portName.name, "StressTestManager");
    std::strcat(addr.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &addr);

    for (int i = 0; i < ThreadCount; ++i)
    {
        params[i].socket = socket;
        params[i].threadNum = i;
        nn::os::CreateThread(&threads[i], DoRandomTest, &params[i], g_ThreadStacks[i], ThreadStackSize, nn::os::DefaultThreadPriority);
        nn::os::StartThread(&threads[i]);
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }

    {
        const char* testName = "0,0,End,0,0\n";
        size_t testNameSize = strlen(testName);

        ASSERT_EQ(nn::htcs::Send(socket, testName, testNameSize, 0), testNameSize);
    }

    nn::htcs::Close(socket);

    NN_LOG("Unmount host\n\n");
    nn::fs::Unmount("host");
}
