﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nnt.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   スリープ復帰時の Htcs のテスト
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    // テストデータのサイズ
    const size_t DataSize = 256 * 1024 * 1024;
    static uint8_t* s_TestData;
    static uint8_t* s_RecvBuffer;

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    void CreateTestData()
    {
        s_TestData = reinterpret_cast<uint8_t*>(std::malloc(sizeof(uint8_t) * DataSize));
        ASSERT_NE(nullptr, s_TestData);
        nnt::htcs::util::FillBuffer(s_TestData, DataSize);

        s_RecvBuffer = reinterpret_cast<uint8_t*>(std::malloc(sizeof(uint8_t) * DataSize));
        ASSERT_NE(nullptr, s_RecvBuffer);
        memset(s_RecvBuffer, 0, sizeof(uint8_t) * DataSize);
    }

    void DestroyTestData()
    {
        std::free(s_TestData);
        std::free(s_RecvBuffer);
    }

    void CheckData(uint8_t* buffer, size_t bufferSize)
    {
        EXPECT_EQ(bufferSize, DataSize);
        if (bufferSize != DataSize)
        {
            return;
        }

        for (int i = 0; i < DataSize; i++)
        {
            EXPECT_EQ(buffer[i], s_TestData[i]);
        }
    }

    class Sleep : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';

            CreateTestData();
        }
        static void TearDownTestCase()
        {
            DestroyTestData();
        }
        virtual void SetUp()
        {
            memset(s_RecvBuffer, 0, sizeof(uint8_t) * DataSize);
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };

    void SendTestName(const char* name)
    {
        int socket = nn::htcs::Socket();
        ASSERT_LE(0, socket);

        nn::htcs::HtcsPortName portName;
        std::strcpy(portName.name, "HtcsSleepTestManager");
        nnt::htcs::util::ConnectToAnyHost(socket, &portName);

        size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(name) + 1;
        char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
        buffer[0] = '\0';

        std::strcat(buffer, g_KeyString);
        std::strcat(buffer, ",");
        std::strcat(buffer, name);

        ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

        std::free(buffer);
        EXPECT_EQ(0, nn::htcs::Close(socket));
    }

    void NonBlockingSend(int socket, nn::htcs::ssize_t size, uint8_t* buffer)
    {
        nn::htcs::ssize_t sentByteCount = 0;
        do
        {
            auto n = nn::htcs::Send(socket, &buffer[sentByteCount], size - sentByteCount, 0);
            if (n < 0)
            {
                EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
            sentByteCount += n;
        } while (sentByteCount < size);
        EXPECT_EQ(size, sentByteCount);
    }

    void NonBlockingRecv(int socket, nn::htcs::ssize_t size, uint8_t* buffer)
    {
        nn::htcs::ssize_t recievedByteCount = 0;
        do
        {
            auto n = nn::htcs::Recv(socket, &buffer[recievedByteCount], size - recievedByteCount, 0);
            if (n < 0)
            {
                EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                continue;
            }
            recievedByteCount += n;
        } while (recievedByteCount < size);
        EXPECT_EQ(size, recievedByteCount);
    }
}

TEST_F(Sleep, Socket)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Socket");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);
    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Socket");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);
    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, Connected)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Connected");

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Connected");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);
    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, BeforeTransmission)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("BeforeTrans1");
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "BeforeTrans1");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    uint8_t ack;

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");
    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");
    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);

    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, BeforeTransmissionHostContinue)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("BeforeTrans2");
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "BeforeTrans2");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    uint8_t ack;

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");
    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");
    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);

    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, Transmission)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Transmission");
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Transmission");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer, nn::htcs::HTCS_MSG_WAITALL);

    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, TransmissionGrad)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("TransmissionGrad");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "TransmissionGrad");
    std::strcat(address.portName.name, g_KeyString);

    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);
    nnt::htcs::util::Connect(socket, &address);

    uint8_t ack;

    size_t snipSize = 256 * 1024;
    nn::htcs::ssize_t sentByteCount = 0;
    do
    {
        if ((DataSize - sentByteCount) < snipSize)
        {
            snipSize = DataSize - sentByteCount;
        }
        auto n = nn::htcs::Send(socket, &s_TestData[sentByteCount], snipSize, 0);
        EXPECT_GT(n, 0);
        if (n < 0)
        {
            NN_LOG("LastError = %d\n", nn::htcs::GetLastError());
        }
        sentByteCount += n;
        NN_LOG("Send %d bytes (%d/%d)\n", n, (int)sentByteCount, DataSize);
    } while (sentByteCount < DataSize);
    EXPECT_EQ(DataSize, sentByteCount);

    nnt::htcs::util::Recv(socket, 1, &ack);

    nn::htcs::ssize_t recievedByteCount = 0;
    do
    {
        auto n = nn::htcs::Recv(socket, &s_RecvBuffer[recievedByteCount], DataSize - recievedByteCount, 0);
        EXPECT_GT(n, 0);
        if (n < 0)
        {
            NN_LOG("LastError = %d\n", nn::htcs::GetLastError());
        }
        recievedByteCount += n;
        NN_LOG("Recv %d bytes (%d/%d)\n", n, (int)recievedByteCount, DataSize);
    } while (recievedByteCount < DataSize);
    EXPECT_EQ(DataSize, recievedByteCount);

    CheckData(s_RecvBuffer, recievedByteCount);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, Listen)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Listen1");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Listen1");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);
    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(Sleep, ListenHostContinue)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Listen2");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Listen2");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);
    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(Sleep, Accept)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Accept1");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Accept1");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Accept\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);
    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(Sleep, AcceptHostContinue)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("Accept2");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "Accept2");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    NN_LOG("Accept\n");
    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);

    nnt::htcs::util::Recv(socket, DataSize, s_RecvBuffer);
    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(Sleep, ShutdownRd)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("ShutdownRd");
    int socket = nn::htcs::Socket();

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ShutdownRd");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    EXPECT_EQ(0, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_RD));

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    EXPECT_EQ(-1, nn::htcs::Recv(socket, s_RecvBuffer, DataSize, 0));
    EXPECT_EQ(nn::htcs::HTCS_EBUSY, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, ShutdownWr)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("ShutdownWr");
    int socket = nn::htcs::Socket();

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ShutdownWr");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    EXPECT_EQ(0, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_WR));

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    EXPECT_EQ(-1, nn::htcs::Send(socket, s_TestData, DataSize, 0));
    EXPECT_EQ(nn::htcs::HTCS_EBUSY, nn::htcs::GetLastError());

    uint8_t ack;
    nnt::htcs::util::Recv(socket, 1, &ack);

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, ShutdownRw)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("ShutdownRw");
    int socket = nn::htcs::Socket();

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ShutdownRw");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    EXPECT_EQ(0, nn::htcs::Shutdown(socket, nn::htcs::HTCS_SHUT_RDWR));

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    EXPECT_EQ(-1, nn::htcs::Send(socket, s_TestData, DataSize, 0));
    EXPECT_EQ(nn::htcs::HTCS_EBUSY, nn::htcs::GetLastError());
    EXPECT_EQ(-1, nn::htcs::Recv(socket, s_RecvBuffer, DataSize, 0));
    EXPECT_EQ(nn::htcs::HTCS_EBUSY, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, ConnectedHostClose)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("ConnectedFail");
    int socket = nn::htcs::Socket();

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "ConnectedFail");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    // 切断直後の送信は成功するので読み捨てる
    int lastError;
    nn::htcs::Send(socket, s_TestData, DataSize, 0);
    EXPECT_EQ(-1, nn::htcs::Send(socket, s_TestData, DataSize, 0));
    lastError = nn::htcs::GetLastError();
    EXPECT_TRUE((lastError == nn::htcs::HTCS_ECONNABORTED) || (lastError == nn::htcs::HTCS_ECONNRESET));

    EXPECT_EQ(-1, nn::htcs::Recv(socket, s_RecvBuffer, DataSize, 0));
    lastError = nn::htcs::GetLastError();
    EXPECT_TRUE((lastError == nn::htcs::HTCS_ECONNABORTED) || (lastError == nn::htcs::HTCS_ECONNRESET));

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, HostServerCloseWhileSend)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("HostServerClose");
    int socket = nn::htcs::Socket();

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "HostServerClose");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    // 切断直後の送信は成功するので読み捨てる
    nn::htcs::Send(socket, s_TestData, DataSize, 0);
    EXPECT_EQ(-1, nn::htcs::Send(socket, s_TestData, DataSize, 0));
    EXPECT_EQ(nn::htcs::HTCS_ECONNRESET, nn::htcs::GetLastError());

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, HostClientCloseWhileSend)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("HostClientClose");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "HostClientClose");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    // 切断直後の送信は成功するので読み捨てる
    nn::htcs::Send(socket, s_TestData, DataSize, 0);
    EXPECT_EQ(-1, nn::htcs::Send(socket, s_TestData, DataSize, 0));
    int lastError = nn::htcs::GetLastError();
    EXPECT_TRUE((lastError == nn::htcs::HTCS_ECONNABORTED) || (lastError == nn::htcs::HTCS_ECONNRESET));

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(Sleep, RecvHostClose)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("RecvHostClose");
    int socket = nn::htcs::Socket();

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "RecvHostClose");
    std::strcat(address.portName.name, g_KeyString);

    nnt::htcs::util::Connect(socket, &address);

    uint8_t ack;

    nnt::htcs::util::Send(socket, DataSize, s_TestData);
    nnt::htcs::util::Recv(socket, 1, &ack);
    EXPECT_EQ(0, nn::htcs::Recv(socket, s_RecvBuffer, DataSize, 0));

    EXPECT_EQ(0, nn::htcs::Close(socket));
}

TEST_F(Sleep, NonBlocking)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("NonBlocking");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "NonBlocking");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    EXPECT_EQ(0, nn::htcs::Fcntl(listenSocket, nn::htcs::HTCS_F_SETFL, nn::htcs::HTCS_O_NONBLOCK));

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    EXPECT_EQ(nn::htcs::HTCS_O_NONBLOCK, nn::htcs::Fcntl(listenSocket, nn::htcs::HTCS_F_GETFL, 0));

    NN_LOG("Waiting for connection from host.\n");
    int socket;
    do
    {
        socket = nn::htcs::Accept(listenSocket, nullptr);
        if (socket < 0)
        {
            EXPECT_EQ(nn::htcs::HTCS_EWOULDBLOCK, nn::htcs::GetLastError());
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    } while (socket < 0);
    NN_LOG("Connected.\n");

    NonBlockingSend(socket, DataSize, s_TestData);
    NonBlockingRecv(socket, DataSize, s_RecvBuffer);

    CheckData(s_RecvBuffer, DataSize);

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}

TEST_F(Sleep, ConnectedHostReconnect)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    SendTestName("HostRecconect");

    nn::htcs::SockAddrHtcs address;
    address.family = nn::htcs::HTCS_AF_HTCS;
    address.peerName = nn::htcs::GetPeerNameAny();
    std::strcpy(address.portName.name, "HostRecconect");
    std::strcat(address.portName.name, g_KeyString);

    int listenSocket = nn::htcs::Socket();
    ASSERT_LE(0, listenSocket);
    EXPECT_EQ(0, nn::htcs::Bind(listenSocket, &address));
    EXPECT_EQ(0, nn::htcs::Listen(listenSocket, 1));

    int socket = nn::htcs::Accept(listenSocket, nullptr);
    EXPECT_LE(0, socket);

    NN_LOG("Waiting for sleep\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
    NN_LOG("Resume\n");

    int lastError;
    nn::htcs::Send(socket, s_TestData, DataSize, 0);
    EXPECT_EQ(-1, nn::htcs::Send(socket, s_TestData, DataSize, 0));
    lastError = nn::htcs::GetLastError();
    EXPECT_TRUE((lastError == nn::htcs::HTCS_ECONNABORTED) || (lastError == nn::htcs::HTCS_ECONNRESET));
    EXPECT_EQ(-1, nn::htcs::Recv(socket, s_RecvBuffer, DataSize, 0));
    lastError = nn::htcs::GetLastError();
    EXPECT_TRUE((lastError == nn::htcs::HTCS_ECONNABORTED) || (lastError == nn::htcs::HTCS_ECONNRESET));

    EXPECT_EQ(0, nn::htcs::Close(socket));
    EXPECT_EQ(0, nn::htcs::Close(listenSocket));
}
