﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_InternalTypes.h>
#include <nn/htclow/detail/htclow_DebugApi.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../testHtclow_Util.h"

/*
 * @file ホスト (エコーサーバ) と通信を行うテスト
 */

namespace nnt { namespace htclow {

namespace {
    const nn::htclow::ModuleId TestModuleId = static_cast<nn::htclow::ModuleId>(0);
    const int TestChannelIdStart = 1;
    const int TestChannelCount = 8;
}

struct EchoTestSetting
{
    int32_t dataSize;
};

const EchoTestSetting g_EchoTestSettings[] =
{
    { 1 },
    { GetMaxBodySize() },
    { GetMaxBodySize() + 1 },
    { 64 * 1024 * 1024 }, // htclow サーバライブラリのヒープサイズより大
};

class EchoTest : public ::testing::TestWithParam<EchoTestSetting>
{
    virtual void SetUp() NN_OVERRIDE
    {
        nn::htclow::detail::OpenDriver(nn::htclow::detail::DriverType::Socket);
    }

    virtual void TearDown() NN_OVERRIDE
    {
        nn::htclow::detail::CloseDriver();
    }
};

INSTANTIATE_TEST_CASE_P(Default, EchoTest, ::testing::ValuesIn(g_EchoTestSettings));

TEST_P(EchoTest, SingleChannel)
{
    size_t size;

    // データの準備
    int32_t dataSize = GetParam().dataSize;

    const auto data = MakeRandomArray(dataSize, 1);
    auto buffer = MakeZeroArray(dataSize);

    // 通信の準備
    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelIdStart);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Connect());

    // データの送信
    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &dataSize, sizeof(dataSize)));
    ASSERT_EQ(sizeof(dataSize), size);

    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, data.get(), dataSize));
    ASSERT_EQ(dataSize, size);

    // データの受信
    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer.get(), dataSize));
    ASSERT_EQ(dataSize, size);

    // 比較
    ASSERT_EQ(0, std::memcmp(data.get(), buffer.get(), dataSize));

    // 通信の終了
    int32_t zero = 0;
    NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, &zero, sizeof(zero)));
    channel.Shutdown();
}

TEST_F(EchoTest, MultiChannel)
{
    size_t size;

    // 通信の準備
    nn::htclow::Module module(TestModuleId);

    nn::htclow::Channel channel[TestChannelCount];
    for (int i = 0; i < TestChannelCount; i++)
    {
        channel[i].Open(&module, static_cast<nn::htclow::ChannelId>(TestChannelIdStart + i));
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel[i].Connect());
    }

    for (auto& setting : g_EchoTestSettings)
    {
        for (int i = 0; i < TestChannelCount; i++)
        {
            // データの準備
            int32_t dataSize = setting.dataSize;

            const auto data = MakeRandomArray(dataSize, 1);
            auto buffer = MakeZeroArray(dataSize);

            // データの送信
            NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel[i].Send(&size, &dataSize, sizeof(dataSize)));
            ASSERT_EQ(sizeof(dataSize), size);

            NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel[i].Send(&size, data.get(), dataSize));
            ASSERT_EQ(dataSize, size);

            // データの受信
            NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel[i].Receive(&size, buffer.get(), dataSize));
            ASSERT_EQ(dataSize, size);

            // 比較
            ASSERT_EQ(0, std::memcmp(data.get(), buffer.get(), dataSize));
        }
    }

    // 通信の終了
    int32_t zero = 0;
    for (int i = 0; i < TestChannelCount; i++)
    {
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel[i].Send(&size, &zero, sizeof(zero)));
        channel[i].Shutdown();
    }
}

}}
