﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <thread>

#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nn/htclow.h>
#include <nn/htclow/detail/htclow_InternalTypes.h>
#include <nn/htclow/detail/htclow_DebugApi.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

#include "../../../../../Programs/Eris/Sources/Libraries/htclow/server/htclow_Packet.h"

#include "../testHtclow_Util.h"

#include "testHtclow_HandshakeUtil.h"
#include "testHtclow_PacketUtil.h"

namespace nnt { namespace htclow {

namespace {
    nn::htclow::ModuleId TestModuleId = static_cast<nn::htclow::ModuleId>(0);
    nn::htclow::ChannelId TestChannelId = static_cast<nn::htclow::ChannelId>(0);

}

class DurabilityTest : public ::testing::Test
{
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::htclow::detail::OpenDriver(nn::htclow::detail::DriverType::Debug);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::htclow::detail::CloseDriver();
    }
};

// Receive を繰り返してメモリリークが発生しないかテストする
TEST_F(DurabilityTest, ReceiveManyTimes)
{
    const int LoopCount = 10; // DebugDriver のバッファサイズ制限で、1023 が上限
    const int dataSize = GetMaxBodySize();

    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    nn::htclow::detail::PrintDefaultAllocatorUsage();
    const auto memoryUsageBefore = nn::htclow::detail::GetDefaultAllocatorUsage();

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));
    int64_t sequenceId = 1;

    DoHandshake(&channel);

    for (int i = 0; i < LoopCount; i++)
    {
        auto data = MakeRandomArray(dataSize, i);
        auto buffer = MakeZeroArray(dataSize);

        // データ送信
        auto packet = MakeDataPacket(channelInternal, sequenceId, data.get(), dataSize);
        SendPacketForTest(*packet);

        // データ受信
        size_t size;
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Receive(&size, buffer.get(), dataSize));
        ASSERT_EQ(dataSize, size);
        ASSERT_EQ(0, std::memcmp(data.get(), buffer.get(), dataSize));

        // ack 受信
        auto ack = ReceivePacketForTest();
        AssertDataAckPacket(*ack, channelInternal, sequenceId);

        sequenceId++;
    }

    const auto memoryUsageAfter = nn::htclow::detail::GetDefaultAllocatorUsage();
    EXPECT_EQ(memoryUsageBefore, memoryUsageAfter);
}

// Send を繰り返してメモリリークが発生しないかテストする
TEST_F(DurabilityTest, SendManyTimes)
{
    const int LoopCount = 1000; // DebugDriver のバッファサイズ制限で、1023 が上限
    const int dataSize = GetMaxBodySize();

    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    nn::htclow::detail::PrintDefaultAllocatorUsage();

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));
    int64_t sequenceId = 1;

    DoHandshake(&channel);

    for (int i = 0; i < LoopCount; i++)
    {
        auto data = MakeRandomArray(dataSize, i);
        auto buffer = MakeZeroArray(dataSize);

        // データ送信
        size_t size;
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, data.get(), dataSize));
        ASSERT_EQ(dataSize, size);

        // データ受信
        auto dataPacket = ReceivePacketForTest();
        AssertDataPacket(*dataPacket, channelInternal, sequenceId, data.get(), dataSize);

        // ack 送信
        auto ackPacket = MakeDataAckPacket(channelInternal, sequenceId);
        SendPacketForTest(*ackPacket);

        sequenceId++;
    }

    // CAUTION:
    //  ack 受信による送信バッファの削除と同期を取ることができないので、
    //  送信バッファの削除が間に合わずにメモリ使用量が多く表示される場合がある。
    //  メモリリークの検査自動化ができないので、定期的にマニュアルチェックする。
    nn::htclow::detail::PrintDefaultAllocatorUsage();
}

// パケット送信中にチャンネルをクローズしてメモリリークが発生しないかテストする
TEST_F(DurabilityTest, CloseDuringSending)
{
    const int LoopCount = 32;
    const int dataSize = GetMaxBodySize();

    const auto memoryUsageBefore = nn::htclow::detail::GetDefaultAllocatorUsage();

    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));

    DoHandshake(&channel);

    for (int i = 0; i < LoopCount; i++)
    {
        auto data = MakeRandomArray(dataSize, i);

        // データ送信 (ack は受け取らない)
        size_t size;
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(channel.Send(&size, data.get(), dataSize));
        ASSERT_EQ(dataSize, size);
    }

    // チャンネルのリソースをすべて解放
    channel.Close();

    const auto memoryUsageAfter = nn::htclow::detail::GetDefaultAllocatorUsage();
    EXPECT_EQ(memoryUsageBefore, memoryUsageAfter);
}

// パケット受信中にチャンネルをクローズしてメモリリークが発生しないかテストする
TEST_F(DurabilityTest, CloseDuringReceiving)
{
    const int LoopCount = 32;
    const int dataSize = GetMaxBodySize();

    nn::htclow::detail::PrintDefaultAllocatorUsage();

    nn::htclow::Module module(TestModuleId);
    nn::htclow::Channel channel(&module, TestChannelId);

    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(channel.GetBase()));
    int64_t sequenceId = 1;

    DoHandshake(&channel);

    for (int i = 0; i < LoopCount; i++)
    {
        auto data = MakeRandomArray(dataSize, i);

        // テスト側からデータ送信 (nn::htclow::Receive() は呼ばない)
        auto packet = MakeDataPacket(channelInternal, sequenceId, data.get(), dataSize);
        SendPacketForTest(*packet);

        sequenceId++;
    }

    // チャンネルのリソースをすべて解放
    channel.Close();

    // CAUTION:
    //  パケット受信と同期を取ることができないので、
    //  受信スレッドがパケットを確保した分、メモリ使用量が多く表示される場合がある。
    //  メモリリークの検査自動化ができないので、定期的にマニュアルチェックする。
    nn::htclow::detail::PrintDefaultAllocatorUsage();
}

}}
