﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LocalCommunication.h"

/* アプレットは全てメインスレッドとは別のスレッドで動作させてください */
namespace frm {

    /* Singleton パターン */
    LocalCommunication& LocalCommunication::GetInstance() NN_NOEXCEPT
    {
        static LocalCommunication Instance;
        return Instance;
    }

    const char * LocalCommunication::ToString(nn::Result result)
    {
        if (result.IsSuccess())
        {
            return "Success";
        }
        else if (nn::ldn::ResultDeviceNotAvailable::Includes(result))
        {
            return "Device Not Available";
        }
        else if (nn::ldn::ResultDeviceOccupied::Includes(result))
        {
            return "Device Occupied";
        }
        else if (nn::ldn::ResultDeviceDisabled::Includes(result))
        {
            return "Device Disabled";
        }
        else if (nn::ldn::ResultWifiOff::Includes(result))
        {
            return "Wifi Off";
        }
        else if (nn::ldn::ResultSleep::Includes(result))
        {
            return "Sleep";
        }
        else if (nn::ldn::ResultInvalidState::Includes(result))
        {
            return "Invalid State";
        }
        else if (nn::ldn::ResultNodeNotFound::Includes(result))
        {
            return "Node Not Found";
        }
        else if (nn::ldn::ResultConnectionFailed::Includes(result))
        {
            return "Connection Failed";
        }
        else if (nn::ldn::ResultNetworkNotFound::Includes(result))
        {
            return "Network Not Found";
        }
        else if (nn::ldn::ResultConnectionTimeout::Includes(result))
        {
            return "Connection Timeout";
        }
        else if (nn::ldn::ResultConnectionRejected::Includes(result))
        {
            return "Connection Rejected";
        }
        else if (nn::ldn::ResultInvalidNetwork::Includes(result))
        {
            return "Invalid Network";
        }
        else if (nn::ldn::ResultCancelled::Includes(result))
        {
            return "Cancelled";
        }
        else if (nn::ldn::ResultIncompatibleVersion::Includes(result))
        {
            return "Incompatible Version";
        }
        else if (nn::ldn::ResultLowerVersion::Includes(result))
        {
            return "Lower Version";
        }
        else if (nn::ldn::ResultHigherVersion::Includes(result))
        {
            return "Higher Version";
        }
        else if (nn::ldn::ResultNodeCountLimitation::Includes(result))
        {
            return "Node Count Limitation";
        }
        else
        {
            return "Unexpected Result";
        }
    }

    const char * LocalCommunication::ToString(nn::ldn::State state)
    {
        if (state == nn::ldn::State_None)
        {
            return "None";
        }
        else if (state == nn::ldn::State_Initialized)
        {
            return "Initialized";
        }
        else if (state == nn::ldn::State_AccessPoint)
        {
            return "AccessPoint";
        }
        else if (state == nn::ldn::State_AccessPointCreated)
        {
            return "AccessPointCreated";
        }
        else if (state == nn::ldn::State_Station)
        {
            return "Station";
        }
        else if (state == nn::ldn::State_StationConnected)
        {
            return "StationConnected";
        }
        else if (state == nn::ldn::State_Error)
        {
            return "Error";
        }
        else
        {
            return "Unexpected Result";
        }
    }

    /* ------------------------------------------------------------ */
    // PRIVATE関数
    /* ------------------------------------------------------------ */
    void LocalCommunication::UpdateTwinkleCounter()
    {
        if (m_TwinkleSwitchFlag == false)
        {
            m_TwinkleCounter = m_TwinkleCounter + 5;
            if (m_TwinkleCounter >= 250)
            {
                m_TwinkleSwitchFlag = true;
            }
        }
        else
        {
            m_TwinkleCounter = m_TwinkleCounter - 5;
            if (m_TwinkleCounter <= 50)
            {
                m_TwinkleSwitchFlag = false;
            }
        }
    }

    void LocalCommunication::UpdateSwipeCounter(int functionMenu)
    {
        m_SwipeCounterState = functionMenu == 3 ? 1 : 2;

        if (m_SwipeCounterState == 1)
        {
            if (m_SwipeCounter < 10)
            {
                m_SwipeCounter++;
            }
            // 0→Widthに向かう感じ
            m_DeltaX = m_DrawPotition.Width - m_DrawPotition.Width / (static_cast<float>(m_SwipeCounter) * 30.f);
        }
        else if (m_SwipeCounterState == 2)
        {
            if (m_SwipeCounter > 0)
            {
                m_SwipeCounter--;
            }
            else
            {
                m_SwipeCounterState = 0;
                m_DeltaX = 0.f;
            }
            // Width→0に向かう感じ
            m_DeltaX = pow(static_cast<double>(m_SwipeCounter), 6.0) * (m_DrawPotition.Width / pow(10.0, 6.0));
        }
    }

    void LocalCommunication::StartNetwork()
    {
        int count = 0;

        auto result = nn::ldn::Initialize();
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(300));

        while (!result.IsSuccess())
        {
            count++;
            result = nn::ldn::Initialize();
            nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(300));
            if (count >= 30)
            {
                NN_LOG("[LDN] >> Initialize Failed\n");
                break;
            }
        }
        NN_LOG("[LDN] >> Initialize Result : %s\n", ToString(result));
    }

    void LocalCommunication::Configuration()
    {
        m_Config.network.intentId = nn::ldn::MakeIntentId(nn::ldn::DefaultLocalCommunicationId, 0);
        m_Config.network.channel = 11;
        m_Config.network.nodeCountMax = m_connectCount;
        m_Config.network.localCommunicationVersion = nn::ldn::LocalCommunicationVersionMin;

        m_Config.security.securityMode = nn::ldn::SecurityMode_Debug;
        const char* passphrase = "PalmaTestToolForMct";
        m_Config.security.passphraseSize = std::strlen(passphrase);
        std::memcpy(m_Config.security.passphrase, passphrase, m_Config.security.passphraseSize);

        const char* userName = "MctTester";
        std::strncpy(m_Config.user.userName, userName, nn::ldn::UserNameBytesMax);

        m_Config.filter.flag = nn::ldn::ScanFilterFlag_LocalCommunicationId;
        m_Config.filter.networkId.intentId.localCommunicationId = nn::ldn::DefaultLocalCommunicationId;
    }

    void LocalCommunication::Scaning()
    {
        auto result = nn::ldn::Scan(m_Info.info, &m_Info.count, nn::ldn::ScanResultCountMax, m_Config.filter, 11);
        nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(200));

        int count = 0;
        while (!result.IsSuccess())
        {
            nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(200));
            result = nn::ldn::Scan(m_Info.info, &m_Info.count, nn::ldn::ScanResultCountMax, m_Config.filter, 11);
            count++;
            NN_LOG("[LDN] >> Scan Retry : %d\n", count);
            if (count >= 5)
            {
                nn::ldn::Finalize();
                break;
            }
        }
        NN_LOG("[LDN] >> Scan Result : %s\n", ToString(result));
        NN_LOG("[LDN] >> Scan Count : %d\n", m_Info.count);
    }

    void LocalCommunication::Connection()
    {
        if (m_Info.count == 0)
        {
            NN_LOG("[LDN] >> Connect Ignored\n");
        }
        else
        {
            nn::Result result;
            for (int i = 0; i < m_Info.count; ++i)
            {
                result = nn::ldn::Connect(m_Info.info[i], m_Config.security, m_Config.user, m_Config.network.localCommunicationVersion, nn::ldn::ConnectOption_None);
                nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(200));

                int count = 0;
                while (!result.IsSuccess())
                {
                    nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(200));
                    result = nn::ldn::Connect(m_Info.info[i], m_Config.security, m_Config.user, m_Config.network.localCommunicationVersion, nn::ldn::ConnectOption_None);
                    count++;
                    NN_LOG("[LDN] >> Connect Retry : %d\n", count);
                    if (count >= 5)
                    {
                        nn::ldn::Finalize();
                        break;
                    }
                }
                NN_LOG("[LDN] >> Connect Result[%d] : %s\n", i, ToString(result));
            }
        }
    }

    void LocalCommunication::GetInfo()
    {
        if (nn::ldn::GetState() == nn::ldn::State_AccessPointCreated || nn::ldn::GetState() == nn::ldn::State_StationConnected)
        {
            for (int i = 0; i < m_Info.count; ++i)
            {
                auto result = nn::ldn::GetNetworkInfo(m_Info.info);
                NN_LOG("[LDN] >> GetNetworkInfo Result[%d] : %s\n", i, ToString(result));
            }
        }
    }

    void LocalCommunication::CreateNetwork()
    {
        m_Info.info = new nn::ldn::NetworkInfo[nn::ldn::ScanResultCountMax];

        if (nn::ldn::GetState() == nn::ldn::State_None)
        {
            StartNetwork();             // ライブラリの初期化
        }

        if (nn::ldn::GetState() == nn::ldn::State_Initialized)
        {
            nn::ldn::OpenAccessPoint(); // アクセスポイントの表明
        }

        if (nn::ldn::GetState() == nn::ldn::State_AccessPoint)
        {
            Configuration();            // 各種Configの設定
            // ネットワークの構築
            int count = 0;
            nn::ldn::State state = nn::ldn::GetState();

            NN_LOG("[LDN] >> state : %d\n", state);

            auto result = nn::ldn::CreateNetwork(m_Config.network, m_Config.security, m_Config.user);
            while (!result.IsSuccess())
            {
                nn::os::SleepThread(nn::TimeSpanType::FromMilliSeconds(100));
                result = nn::ldn::CreateNetwork(m_Config.network, m_Config.security, m_Config.user);
                count++;
                NN_LOG("[LDN] >> CreateNetwork Retry : %d\n", count);
                if (count >= 10)
                {
                    nn::ldn::Finalize();
                    break;
                }
            }
            NN_LOG("[LDN] >> CreateNetwork Result : %s\n", ToString(result));
        }
    }

    void LocalCommunication::DestroyNetwork()
    {
        delete m_Info.info;

        auto result = nn::ldn::DestroyNetwork();
        NN_LOG("[LDN] >> DestroyNetwork Result : %s\n", ToString(result));

        result = nn::ldn::CloseAccessPoint();
        NN_LOG("[LDN] >> CloseAccessPoint Result : %s\n", ToString(result));

        nn::ldn::Finalize();
    }

    void LocalCommunication::OpenStation()
    {
        m_Info.info = new nn::ldn::NetworkInfo[nn::ldn::ScanResultCountMax];

        if (nn::ldn::GetState() == nn::ldn::State_None)
        {
            StartNetwork();             // ライブラリの初期化
        }

        if (nn::ldn::GetState() == nn::ldn::State_Initialized)
        {
            nn::ldn::OpenStation();     // ステーションとして機能する
        }

        if (nn::ldn::GetState() == nn::ldn::State_Station)
        {
            Configuration();            // 各種Configの設定
            Scaning();                  // MasterをScanする
            Connection();               // Masterと接続する
        }
    }

    void LocalCommunication::CloseStation()
    {
        delete m_Info.info;

        auto result = nn::ldn::Disconnect();
        NN_LOG("[LDN] >> Disconnect Result : %s\n", ToString(result));

        result = nn::ldn::CloseStation();
        NN_LOG("[LDN] >> CloseStation Result : %s\n", ToString(result));

        nn::ldn::Finalize();
    }

    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    void LocalCommunication::Control(frm::NpadState* npadState)
    {
        m_cursol.up = false;
        m_cursol.down = false;
        m_cursol.left = false;
        m_cursol.right = false;
        m_cursol.fix = false;
        m_cursol.cancel = false;

        // 接続中のコントローラのボタンの押下げ状況を確認します
        for (auto index = 0; index < frm::NpadIdNum; index++)
        {
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLDown>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickRDown>()
                ) {
                m_cursol.down = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLUp>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickRUp>()
                ) {
                m_cursol.up = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLLeft>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickRLeft>()
                ) {
                m_cursol.left = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLRight>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickRRight>()
                ) {
                m_cursol.right = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickL>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickR>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::L>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::R>()
                ) {
                m_cursol.fix = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadPalmaButton::Palma>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::ZL>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::ZR>()
                ) {
                m_cursol.cancel = true;
            }
        }

        if (m_cursol.up || m_cursol.down)
        {
            m_topMenu = m_topMenu == 0 ? 1 : 0;
        }
        if (m_cursol.left)
        {
            switch (m_topMenu)
            {
                case 0:         // 接続台数の選択
                    m_connectCount = m_connectCount == 1 ? 8 : m_connectCount - 1;
                    break;
                case 1:         // 機能の選択
                    m_function = m_function == 0 ? LdnFunctionTextSize - 1 : m_function - 1;
                    break;
                default:
                    break;
            }

        }
        if (m_cursol.right)
        {
            switch (m_topMenu)
            {
                case 0:         // 接続台数の選択
                    m_connectCount = m_connectCount == 8 ? 1 : m_connectCount + 1;
                    break;
                case 1:         // 機能の選択
                    m_function = m_function == LdnFunctionTextSize - 1 ? 0 : m_function + 1;
                    break;
                default:
                    break;
            }
        }
        if (m_cursol.fix)
        {
            if (m_topMenu == 1)      // 機能の選択
            {
                switch (m_function)
                {
                    case 0:         // Create Network
                        CreateNetwork();
                        break;
                    case 1:         // Delete Network
                        DestroyNetwork();
                        break;
                    case 2:         // Open Station
                        OpenStation();
                        break;
                    case 3:         // Close Station
                        CloseStation();
                        break;
                    default:
                        break;
                }
            }
        }

        if (m_getStateCount == 0)
        {
            GetInfo();
            m_getStateCount++;
        }
        else if(m_getStateCount >= 30)
        {
            m_getStateCount = 0;
        }
        else
        {
            m_getStateCount++;
        }

    } // NOLINT(impl/function_size)

    void LocalCommunication::Draw(GraphicsSystem* pGraphicsSystem,
                             nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             int functionMenu)
    {
        UpdateTwinkleCounter();
        UpdateSwipeCounter(functionMenu);

        m_DrawPotition.X = 1280.f - m_DeltaX - 5.f;

        nn::util::Color4u8 color = { 150, 50, 10, 255 };

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // 小枠
        pPrimitiveRenderer->SetColor(Color::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), m_DrawPotition.X, m_DrawPotition.Y, m_DrawPotition.Width, m_DrawPotition.Height);
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(Color::Black);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), m_DrawPotition.X, m_DrawPotition.Y, m_DrawPotition.Width, m_DrawPotition.Height);
        pPrimitiveRenderer->SetColor(color);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), m_DrawPotition.X + 3, m_DrawPotition.Y + 3, m_DrawPotition.Width - 6, m_DrawPotition.Height - 6);

        pTextWriter->SetScale(1.0, 1.0);

        int DeltaX = 30;
        int DeltaY = 10;

        // TOP項目の作成
        pTextWriter->SetTextColor(Color::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + 10, m_DrawPotition.Y + DeltaY - 5);
        pTextWriter->Print("[Local Network]");

        DeltaY += 20;

        nn::util::Color4u8 t_color = m_topMenu == 1 ? color : Color::Black;
        pTextWriter->SetTextColor(t_color);
        pTextWriter->SetCursor(m_DrawPotition.X + 25, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("Function");
        pTextWriter->SetTextColor(ExClearColor::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + 25 + 2, m_DrawPotition.Y + DeltaY + 2);
        pTextWriter->Print("Function", LdnFunctionText[m_function]);

        pTextWriter->SetTextColor(t_color);
        pTextWriter->SetCursor(m_DrawPotition.X + 135, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print(": %s", LdnFunctionText[m_function]);
        pTextWriter->SetTextColor(ExClearColor::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + 135 + 2, m_DrawPotition.Y + DeltaY + 2);
        pTextWriter->Print(": %s", LdnFunctionText[m_function]);

        DeltaY += 25;

        t_color = m_topMenu == 0 ? color : Color::Black;
        pTextWriter->SetTextColor(t_color);
        pTextWriter->SetCursor(m_DrawPotition.X + 25, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("ConnectMax");
        pTextWriter->SetTextColor(ExClearColor::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + 25 + 2, m_DrawPotition.Y + DeltaY + 2);
        pTextWriter->Print("ConnectMax");

        pTextWriter->SetTextColor(t_color);
        pTextWriter->SetCursor(m_DrawPotition.X + 135, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print(": %d", m_connectCount);
        pTextWriter->SetTextColor(ExClearColor::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + 135 + 2, m_DrawPotition.Y + DeltaY + 2);
        pTextWriter->Print(": %d", m_connectCount);

        DeltaY += 30;

        pTextWriter->SetScale(0.8, 0.8);
        pTextWriter->SetTextColor(QuorterColor::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + 25 + DeltaX, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("[State] : %s", ToString(nn::ldn::GetState()));

        if (nn::ldn::GetState() == nn::ldn::State_AccessPointCreated || nn::ldn::GetState() == nn::ldn::State_StationConnected)
        {
            DeltaY += 20;

            pTextWriter->SetScale(0.8, 0.8);
            pTextWriter->SetTextColor(QuorterColor::Black);
            pTextWriter->SetCursor(m_DrawPotition.X + 25 + DeltaX, m_DrawPotition.Y + DeltaY);
            pTextWriter->Print("[Channel] : %d", m_Info.info[0].common.channel);

            DeltaY += 20;

            pTextWriter->SetScale(0.8, 0.8);
            pTextWriter->SetTextColor(QuorterColor::Black);
            pTextWriter->SetCursor(m_DrawPotition.X + 25 + DeltaX, m_DrawPotition.Y + DeltaY);
            pTextWriter->Print("[LinkLevel] : %d", m_Info.info[0].common.linkLevel);
        }
    }

    /* ------------------------------------------------------------ */
    // クラス外関数
    /* ------------------------------------------------------------ */

}
