﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "FileSystem.h"

namespace frm{
    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    Fs& Fs::GetInstance()
    {
        static Fs Instance;
        return Instance;
    }

    const char* Fs::ToString(nn::Result result)
    {
        if (result.IsSuccess())
        {
            return "Success";
        }
        else if (nn::fs::ResultTargetNotFound::Includes(result))
        {
            return "ResultTarget Not Found";
        }
        else if (nn::fs::ResultMountNameAlreadyExists::Includes(result))
        {
            return "MountName AlreadyExists";
        }
        else
        {
            return "Unexpected Result";
        }
    }

    void Fs::Initialize()
    {
        // とりあえず初期状態はHOST接続
        MountHost("D://NCL/Resources");
    }

    void Fs::MountHost(std::string rootpath)
    {
        m_mount.rootPath = rootpath;
        m_mount.path = rootpath.substr(3, rootpath.length() - 3);
        m_mount.type = "PC";

        // ドライブ名を検索
        m_mount.divChar = ":/";
        auto drivePos = rootpath.find(m_mount.divChar, 0);
        auto isFindDriveLetter = (drivePos != std::string::npos) && (rootpath.find(m_mount.divChar, drivePos + 2) == std::string::npos);
        if (!isFindDriveLetter)
        {
            // 「:/」が見つからない場合「:\」を検索
            m_mount.divChar = ":\\";
            drivePos = rootpath.find(m_mount.divChar, 0);
            isFindDriveLetter = (drivePos != std::string::npos) && (rootpath.find(m_mount.divChar, drivePos + 2) == std::string::npos);
        }
        if (!isFindDriveLetter)
        {
            // ドライブ名が見つからない、もしくは区切り文字が複数個発見された場合失敗
            NN_LOG("[FS] >> DriveLetter is Unexpected\n");
            return;
        }

        // マウント名に不正な文字が含まれていないか検索
        if (
            (rootpath.substr(0, drivePos).find("@", 0) == 0) ||                       // マウント名が@から始まる
            (rootpath.substr(0, drivePos).find(":", 0) != std::string::npos) ||       // マウント名に:が含まれる
            (rootpath.substr(0, drivePos).find("/", 0) != std::string::npos)          // マウント名に/が含まれる
            )
        {
            // マウント名として不正な文字が含まれていた場合は失敗
            NN_LOG("[FS] >> MountName is Unexpected\n");
            return;
        }

        m_mount.driveName = rootpath.substr(0, drivePos + 2);
        auto result = nn::fs::MountHost(m_mount.type.c_str(), m_mount.driveName.c_str());
        NN_LOG("[FS] >> MountHost(%s) : %s\n", m_mount.driveName.c_str(), gFs.ToString(result));
        m_mount.isMounted = result.IsSuccess();
        return;
    }

    void Fs::MountSd(std::string rootpath)
    {
        m_mount.rootPath = rootpath;
        m_mount.type = "SD";

        auto result = nn::fs::MountSdCardForDebug(m_mount.type.c_str());
        NN_LOG("[FS] >> MountSd : %s\n", gFs.ToString(result));
        auto pos = rootpath.find_first_of(":");
        if (pos != std::string::npos)
        {
            // マウント名として不正な文字が含まれていた場合は失敗
            NN_LOG("[FS] >> MountName is Unexpected\n");
            return;
        }
        m_mount.path = rootpath;
        return;
    }

    void Fs::LoadFileRootPath()
    {
        const size_t ApplicationHeapSize = 128 * 1024;

        ApplicationHeap applicationHeap;
        applicationHeap.Initialize(ApplicationHeapSize);

        nn::swkbd::ShowKeyboardArg showKeyboardArg;
        nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

        showKeyboardArg.keyboardConfig.isUseUtf8 = true;

        // 文字列の設定
        const char* guide_string = u8"Select Resource Path";
        nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig, guide_string);

        // 共有メモリ用バッファの割り当て
        size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize(false);
        void* swkbd_work_buffer = applicationHeap.Allocate(in_heap_size, nn::os::MemoryPageSize);
        showKeyboardArg.workBuf = swkbd_work_buffer;
        showKeyboardArg.workBufSize = in_heap_size;

        const char* initial_string = m_mount.rootPath.c_str();
        nn::swkbd::SetInitialTextUtf8(&showKeyboardArg, initial_string);

        // 終了パラメータの設定
        size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
        nn::swkbd::String output_string;
        output_string.ptr = reinterpret_cast<char16_t*>(applicationHeap.Allocate(out_heap_size, nn::os::MemoryPageSize));
        output_string.bufSize = out_heap_size;

        nn::swkbd::ShowKeyboard(&output_string, showKeyboardArg);

        char* str;
        memcpy(&str, &output_string.ptr, 64);

        m_mount.rootPath = str;

        if (m_mount.isMounted)
        {
            Unmount();
        }
        MountHost(m_mount.rootPath);

        applicationHeap.Deallocate(output_string.ptr);
        applicationHeap.Deallocate(swkbd_work_buffer);

        applicationHeap.Finalize();
    }

    void Fs::LoadFile(std::string* outputStr, int* outputFileCount, std::string pathStr)
    {
        std::string path;

        if(!m_mount.isMounted)
        {
            MountHost(m_mount.rootPath);
        }

        if (m_mount.isMounted)
        {
            int64_t fileCount = 0;
            nn::fs::DirectoryEntry entry[MaxEntryCount];
            nn::fs::DirectoryHandle hDirectory;

            path = m_mount.type + m_mount.divChar + m_mount.path + pathStr;
            auto result = nn::fs::OpenDirectory(&hDirectory, path.c_str(), nn::fs::OpenDirectoryMode_File);
            if (result.IsSuccess())
            {
                result = nn::fs::ReadDirectory(&fileCount, entry, hDirectory, MaxEntryCount);

                for (auto count = 0; count < fileCount; count++)
                {
                    outputStr[count] = entry[count].name;
                    NN_LOG("[FS] >> File : %s\n", &outputStr[count][0]);
                }
                *outputFileCount = fileCount;
                nn::fs::CloseDirectory(hDirectory);
            }
            else
            {
                NN_LOG("[FS] >> Directory(%s) is not found \n", path.c_str());
            }
        }
    }

    void Fs::Unmount()
    {
        // ファイルシステムをアンマウントする
        nn::fs::Unmount(m_mount.type.c_str());
    }
}
