﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <string>
#include <nn/util/util_MatrixApi.h>

#include "VectorChart.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

namespace {

s2d::Point2D Point2DFromVector(const IVectorChartDataSetAccessor::VectorType& input,
                             const s2d::Point2D& center,
                             const float& radius,
                             const float& max) NN_NOEXCEPT
{
    auto output = center;
    if (input.length > max)
    {
        output.x += input.x / input.length * radius;
        output.y -= input.y / input.length * radius;
    }
    else
    {
        output.x += input.x / max * radius;
        output.y -= input.y / max * radius;
    }
    return output;
}

void DrawTextAuto(const s2d::Point2D& position, float value, bool isFloat) NN_NOEXCEPT
{
    if (isFloat)
    {
        s2d::DrawText(position, "%3.2f", value);
    }
    else
    {
        s2d::DrawText(position, "%d", static_cast<int>(value));
    }
}

}

VectorChart::VectorChart(const VectorChartUiParameterConfigList& config, bool yInFloat) NN_NOEXCEPT :
    m_UiParameterConfig(config),
    m_YInFloat(yInFloat)
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_UiParameter); ++i)
    {
        m_UiParameter[i] = m_UiParameterConfig.v[i].init;
    }
    UpdateUiParameter();
}

void VectorChart::Increment(UiParameterType param) NN_NOEXCEPT
{
    m_UiParameter[param] = std::min(m_UiParameterConfig.v[param].max,
                                    m_UiParameter[param] * m_UiParameterConfig.v[param].step);
    UpdateUiParameter();
}

void VectorChart::Decrement(UiParameterType param) NN_NOEXCEPT
{
    m_UiParameter[param] = std::max(m_UiParameterConfig.v[param].min,
                                    m_UiParameter[param] / m_UiParameterConfig.v[param].step);
    UpdateUiParameter();
}

void VectorChart::Draw(s2d::Point2D position,
                       float    size,
                       std::string  xStr,
                       std::string  yStr,
                       IVectorChartDataSetAccessor* pDataSets[],
                       int count) NN_NOEXCEPT
{
    auto graphPosition = position;
    graphPosition.x += XSizeLineChartLabel;
    DrawFrame(graphPosition, size, xStr, yStr);
    DrawSamples(graphPosition, size, pDataSets, count);
    DrawValue(graphPosition, size, pDataSets, count);
}

void VectorChart::DrawFrame(s2d::Point2D position,
                            float        size,
                            std::string  xStr,
                            std::string  yStr) NN_NOEXCEPT
{
    s2d::Point2D center = { {position.x + size / 2, position.y + size / 2 } };
    s2d::Point2D textPosition;
    float radius = size * 0.45;

    s2d::SetTextScale(TextScaleSmall);

    // Grid の描画
    s2d::DrawLine({ {position.x, center.y } },
                  { {position.x + size, center.y} },
                  ColorSceneBorderBold,
                  WidthBorderDefault );
    s2d::DrawLine({ {center.x, position.y} },
                  { {center.x, position.y + size} },
                  ColorSceneBorderBold,
                  WidthBorderDefault );

    // 円の描画
    const int circleCount = 5;
    for (int i = 0; i < circleCount; ++i)
    {
        s2d::DrawCircle(center, radius * (i + 1) / (circleCount), ColorSceneBorderLight, WidthBorderDefault);
    }

    // 文字の描画
    textPosition.x = center.x - TextScaleSmall.x * TextSize;
    textPosition.y = position.y;
    s2d::DrawText(textPosition, "%s", yStr.c_str());
    textPosition.y = center.y - TextScaleSmall.y * TextSize;
    textPosition.x = position.x + size - TextScaleSmall.x * TextSize / 2;
    s2d::DrawText(textPosition, "%s", yStr.c_str());

    textPosition.x = center.x + TextScaleSmall.x * TextSize / 2;
    textPosition.y = position.y;
    DrawTextAuto(textPosition, m_Max, m_YInFloat);
    textPosition.y = position.y + size - TextScaleSmall.y * TextSize / 2;
    DrawTextAuto(textPosition, m_Max, m_YInFloat);
    textPosition.y = center.y + TextScaleSmall.y * TextSize / 2;
    textPosition.x = position.x;
    DrawTextAuto(textPosition, m_Max, m_YInFloat);
    textPosition.x = position.x + size - TextScaleSmall.x * TextSize / 2;
    DrawTextAuto(textPosition, m_Max, m_YInFloat);
    s2d::SetTextScale(TextScaleDefault);

}

void VectorChart::DrawSamples(s2d::Point2D position,
                              float        size,
                              IVectorChartDataSetAccessor* pDataSets[],
                              int count) NN_NOEXCEPT
{
    s2d::Point2D center = { {position.x + size / 2, position.y + size / 2 } };
    float radius = size * 0.45;

    for (int dataSetIndex = 0; dataSetIndex < count; ++dataSetIndex)
    {
        auto& dataSet = *(pDataSets[dataSetIndex]);
        int sampleCount = std::min(static_cast<int>(dataSet.size()), m_SampleCountMax);
        if (sampleCount == 0)
        {
            continue;
        }

        int startingIndex = (dataSet.size() > sampleCount) ?
                             dataSet.size() - sampleCount :
                             0;
        for (int sampleIndex = 0; sampleIndex < (sampleCount - 1); ++sampleIndex)
        {
            s2d::Point2D lineStart = Point2DFromVector(
                dataSet.at(startingIndex + sampleIndex),
                center,
                radius,
                m_Max
            );
            s2d::Point2D lineEnd = Point2DFromVector(
                dataSet.at(startingIndex + sampleIndex + 1),
                center,
                radius,
                m_Max
            );
            auto color = dataSet.color();
            color.a = 64 - ((64 / m_SampleCountMax) * (sampleCount - sampleIndex));
            s2d::DrawLine(lineStart, lineEnd, color, WidthBorderDefaultBold);
        }

        auto vector = dataSet.back();
        auto endPoint = Point2DFromVector(vector, center, radius, m_Max);
        if (vector.x != 0 || vector.y != 0)
        {
            DrawArrow(center,
                endPoint,
                dataSet.color(),
                WidthBorderDefaultBold,
                radius * 0.10f);
        }
    }
}

void VectorChart::DrawValue(s2d::Point2D position,
                            float        size,
                            IVectorChartDataSetAccessor* pDataSets[],
                            int count) NN_NOEXCEPT
{
    auto valueXPosition = position.x + size + XSizePadding * 5;

    position.x = valueXPosition + XSizeLineChartLabel;

    s2d::SetTextScale(TextScaleSmall);
    s2d::DrawText(position, "x");
    position.x += XSizeLineChartLabel;
    s2d::DrawText(position, "y");
    position.x += XSizeLineChartLabel;
    s2d::DrawText(position, "angle");
    position.x += XSizeLineChartLabel;
    position.y += YSizeLineSmall;
    position.x = valueXPosition + XSizeLineChartLabel;
    s2d::DrawText(position, "length");
    position.x += XSizeLineChartLabel;
    s2d::DrawText(position, "max");
    position.x += XSizeLineChartLabel;
    s2d::DrawText(position, "min");
    position.x += XSizeLineChartLabel;
    position.y += YSizeLineSmall * 2;

    for (int dataSetIndex = 0; dataSetIndex < count; ++dataSetIndex)
    {
        auto& dataSet = *(pDataSets[dataSetIndex]);
        if (dataSet.size() == 0)
        {
            continue;
        }

        position.x = valueXPosition;
        s2d::SetTextColor(dataSet.color());
        s2d::DrawText(position, "■");
        s2d::SetTextColor(ColorSceneTextDefault);
        s2d::DrawText({ { position.x + XSizeLineChartColorIndex, position.y} }, "%s", dataSet.label());
        position.x += XSizeLineChartLabel;

        char outBuffer[256];
        s2d::DrawText(position, "%s", dataSet.xStr(outBuffer));
        position.x += XSizeLineChartLabel;
        s2d::DrawText(position, "%s", dataSet.yStr(outBuffer));
        position.x += XSizeLineChartLabel;
        s2d::DrawText(position, "%s", dataSet.dirStr(outBuffer));
        position.x += XSizeLineChartLabel;
        position.y += YSizeLineSmall;
        position.x = valueXPosition + XSizeLineChartLabel;
        s2d::DrawText(position, "%s", dataSet.lenStr(outBuffer));
        position.x += XSizeLineChartLabel;
        s2d::DrawText(position, "%s", dataSet.lenMaxStr(outBuffer));
        position.x += XSizeLineChartLabel;
        s2d::DrawText(position, "%s", dataSet.lenMinStr(outBuffer));
        position.x += XSizeLineChartLabel;
        position.y += YSizeLineSmall;
    }
    s2d::SetTextScale(TextScaleDefault);
}

void VectorChart::UpdateUiParameter() NN_NOEXCEPT
{
    // Y軸方向の表示レンジ
    m_Max = m_UiParameter[UiParameterType_Scale];

    // X軸方向の表示サンプル数
    m_SampleCountMax = static_cast<int>(m_UiParameter[UiParameterType_SampleCount]);
}
