﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/hid/debug/hid_UniquePad.h>

#include "UniquePadController.h"

namespace
{
    nn::hid::system::UniquePadId g_UniquePadIds[nn::hid::system::UniquePadIdCountMax];
    int g_UniquePadIdCount;

    bool IsUniquePadDisconnected(UniquePadAccessor* pController) NN_NOEXCEPT
    {
        for (int i = 0; i < g_UniquePadIdCount; ++i)
        {
            if (pController->GetId()._storage == g_UniquePadIds[i]._storage)
            {
                return false;
            }
        }
        return true;
    }

}

UniquePadAccessor::UniquePadAccessor(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT :
    m_Id(id),
    m_LastSamplingNumber(std::numeric_limits<int64_t>::min())
{
    nn::hid::detail::PadDriverState center;
    nn::hid::debug::GetUniquePadDriverState(&center, m_Id);
    m_LeftAnalogStickFifo.Initialize(ConvertAnalogStickState(center.analogStickL));
    m_RightAnalogStickFifo.Initialize(ConvertAnalogStickState(center.analogStickR));
    m_RxPacketHistoryFifo.Initialize();


    switch (nn::hid::system::GetUniquePadType(m_Id))
    {
    case nn::hid::system::UniquePadType_LeftController:
        m_DeviceType = nn::hid::system::DeviceType::JoyConLeft::Mask;
        break;
    case nn::hid::system::UniquePadType_RightController:
        m_DeviceType = nn::hid::system::DeviceType::JoyConRight::Mask;
        break;
    case nn::hid::system::UniquePadType_FullKeyController:
        m_DeviceType = nn::hid::system::DeviceType::SwitchProController::Mask;
        break;
    default:
        m_DeviceType = nn::hid::system::DeviceType::Unknown::Mask;
    }

    if (nn::hid::system::GetUniquePadControllerNumber(&m_PlayerNumber, m_Id).IsFailure())
    {
        m_PlayerNumber = 0;
    }
}

void UniquePadAccessor::Update() NN_NOEXCEPT
{
    UpdateState();
    UpdateSixAxisSensorState();

    // パケット受信履歴を更新
    nn::hid::detail::RxPacketHistory rxHistory;
    nn::hid::debug::GetRxPacketHistory(&rxHistory, m_Id);
    m_RxPacketHistoryFifo.AddSample(rxHistory);
}

nn::hid::system::UniquePadId UniquePadAccessor::GetId() NN_NOEXCEPT
{
    return m_Id;
}

nn::hid::detail::PadDriverState UniquePadAccessor::GetPadDriverState() NN_NOEXCEPT
{
    return m_PadDriverState;
}

AnalogStickStateExFifo* UniquePadAccessor::GetAnalogStickFifo(DeviceIndex index) NN_NOEXCEPT
{
    if (index == DeviceIndex_Left)
    {
        return &m_LeftAnalogStickFifo;
    }
    return &m_RightAnalogStickFifo;
}

SixAxisSensorDriverStateFifo* UniquePadAccessor::GetSixAxisSensorFifo() NN_NOEXCEPT
{
    return &m_SixAxisSensorFifo;
}

RxPacketHistoryFifo* UniquePadAccessor::GetRxPacketHistoryFifo() NN_NOEXCEPT
{
    return &m_RxPacketHistoryFifo;
}

nn::hid::system::DeviceTypeSet UniquePadAccessor::GetDeviceType() NN_NOEXCEPT
{
    return m_DeviceType;
}

int UniquePadAccessor::GetPlayerNumber() NN_NOEXCEPT
{
    return m_PlayerNumber;
}

void UniquePadAccessor::ResetFifo() NN_NOEXCEPT
{
    m_LeftAnalogStickFifo.Clear();
    m_RightAnalogStickFifo.Clear();
    m_SixAxisSensorFifo.Clear();
    m_RxPacketHistoryFifo.Clear();
}

void UniquePadAccessor::UpdateState() NN_NOEXCEPT
{
    // 最新のステートを取得
    nn::hid::debug::GetUniquePadDriverState(&m_PadDriverState, m_Id);
    if (m_PadDriverState.sampleNumber > m_LastSamplingNumber)
    {
        // AnalogStick の FIFO に新しいデータを追加
        m_LeftAnalogStickFifo.AddSample(ConvertAnalogStickState(m_PadDriverState.analogStickL));
        m_RightAnalogStickFifo.AddSample(ConvertAnalogStickState(m_PadDriverState.analogStickR));
    }
}

void UniquePadAccessor::UpdateSixAxisSensorState() NN_NOEXCEPT
{
    // 最新のステートを取得
    nn::hid::detail::SixAxisSensorDriverState state[nn::hid::debug::UniquePadStateCountMax];
    auto count = nn::hid::debug::GetSixAxisSensorDriverStates(state, NN_ARRAY_SIZE(state), m_Id);
    for (int i = count - 1; i >= 0; --i)
    {
        m_SixAxisSensorFifo.AddSample(state[i]);
    }
}

void UniquePadList::Initialize() NN_NOEXCEPT
{
    nn::hid::system::BindUniquePadConnectionEvent(&m_UpdateEvent, nn::os::EventClearMode_AutoClear);
}

void UniquePadList::Update() NN_NOEXCEPT
{
    if (nn::os::TryWaitSystemEvent(&m_UpdateEvent) == true)
    {
        g_UniquePadIdCount = nn::hid::system::ListUniquePads(g_UniquePadIds, NN_ARRAY_SIZE(g_UniquePadIds));
        // 各コントローラーの内部状態の更新
        m_List.erase(
            std::remove_if(m_List.begin(), m_List.end(), IsUniquePadDisconnected),
            m_List.end()
        );

        for (int i = 0; i < g_UniquePadIdCount; ++i)
        {
            bool isNewDeviceConnected = true;
            for (auto& controller : m_List)
            {
                if (controller->GetId()._storage == g_UniquePadIds[i]._storage)
                {
                    isNewDeviceConnected = false;
                    break;
                }
            }
            if (isNewDeviceConnected)
            {
                m_List.push_back(new UniquePadAccessor(g_UniquePadIds[i]));
            }
        }
    }
    // 各コントローラーの内部状態の更新
    for (auto& controller : m_List)
    {
        controller->Update();
    }

    // 選択済みコントローラーの更新
    for (auto& controller : m_List)
    {
        auto state = controller->GetPadDriverState();
        if (state.buttons.Test<nn::hid::detail::PadButton::StickL>())
        {
            UpdateSelectedDevice(controller, DeviceIndex_Left);
        }
        if (state.buttons.Test<nn::hid::detail::PadButton::StickR>())
        {
            UpdateSelectedDevice(controller, DeviceIndex_Right);
        }
    }
}

bool UniquePadList::GetSelectedController(UniquePadAccessor** pOutValue) NN_NOEXCEPT
{
    if (m_Selected == true)
    {
        *pOutValue = m_pSelectedController;
        return true;
    }

    return false;
}

bool UniquePadList::GetSelectedDeviceIndex(DeviceIndex* pOutValue) NN_NOEXCEPT
{
    if (m_Selected == true)
    {
        *pOutValue = m_PositionIndex;
        return true;
    }

    return false;
}

void UniquePadList::ClearSelectedController() NN_NOEXCEPT
{
    m_Selected = false;
}

void UniquePadList::ResetSelectedControllerFifo() NN_NOEXCEPT
{
    if (m_Selected == true && m_pSelectedController != nullptr)
    {
        m_pSelectedController->ResetFifo();
    }
}

UniquePadList::Array::iterator UniquePadList::begin() NN_NOEXCEPT
{
    return m_List.begin();
}

UniquePadList::Array::iterator UniquePadList::end() NN_NOEXCEPT
{
    return m_List.end();
}

void UniquePadList::UpdateSelectedDevice(UniquePadAccessor* pController,
                                           DeviceIndex positionIndex) NN_NOEXCEPT
{
    if (m_pSelectedController != pController ||
        m_PositionIndex != positionIndex ||
        m_Selected == false)
    {
        m_pSelectedController = pController;
        m_PositionIndex = positionIndex;
        m_Selected = true;
    }

}
