﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <vector>

#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/os.h>

#include "MenuItem.h"
#include "ScreenStatics.h"
#include "s2d/Simple2D.h"

MenuItem::MenuItem(const std::string& name, const bool& isFunctional) NN_NOEXCEPT :
    IsFunctional(isFunctional),
    Name(name),
    m_Value(0),
    m_IsSelected(false)
{
    // 何もしない
}

void MenuItem::Print(s2d::Point2D & position, int depth) NN_NOEXCEPT
{
    s2d::DrawText(position, "%s", Name.c_str());
    if (m_Selectables.size() > 0)
    {
        if (m_IsSelected == true)
        {
            s2d::SetTextColor(ColorMenuTextStrong);
        }
        auto itemPosition  = position;
        itemPosition .x += (XSizeMenu - XSizeMenuItemIndent * depth);
        s2d::DrawText(itemPosition , "%s", m_Selectables.at(m_Value).c_str());
        s2d::SetTextColor(ColorMenuTextDefault);
    }
    position.y += YSizeLine;
}

void MenuItem::NextValue(bool recursive) NN_NOEXCEPT
{
    if (m_Selectables.size() == 0 || recursive == true)
    {
        return;
    }

    m_IsSelected = true;
    ++m_Value;
    if (m_Value >= m_Selectables.size())
    {
        m_Value = 0;
    }
}

void MenuItem::PrevValue(bool recursive) NN_NOEXCEPT
{
    if (m_Selectables.size() == 0 || recursive == true)
    {
        return;
    }

    m_IsSelected = true;
    --m_Value;
    if (m_Value < 0)
    {
        m_Value = m_Selectables.size() - 1;
    }
}

void MenuItem::AddSelectable(std::string selectable) NN_NOEXCEPT
{
    m_Selectables.push_back(selectable);
}

MenuItemHolder::MenuItemHolder(const std::string& name, const bool& isFunctional) NN_NOEXCEPT :
    MenuItemHolder(name, isFunctional, false)
{
    // 何もしない
}

MenuItemHolder::MenuItemHolder(const std::string& name, const bool& isFunctional, const bool& expandInNewMenu) NN_NOEXCEPT :
    MenuItemHolder(name, isFunctional, expandInNewMenu, 0)
{
    // 何もしない
}

MenuItemHolder::MenuItemHolder(const std::string& name, const bool& isFunctional, const bool& expandInNewMenu, const int& selectionCountMax) NN_NOEXCEPT :
    MenuItem(name, isFunctional),
    ExpandInNewMenu(expandInNewMenu),
    SelectionCountMax(selectionCountMax),
    m_ItemIndex(0),
    m_FunctionalItemCount(0),
    m_SelectedCount(0),
    m_Focus(Focus_Out)
{
    for (auto& value : m_ValuesSelected)
    {
        value = 0;
    }
}

void MenuItemHolder::Print(s2d::Point2D & position, int depth) NN_NOEXCEPT
{
    if (m_Focus == Focus_Child)
    {
        if (m_pMenuItems.at(m_ItemIndex)->IsChildMenuOpened())
        {
            m_pMenuItems.at(m_ItemIndex)->Print(position, depth);
            return;
        }
    }

    if (Name.size() > 0)
    {
        s2d::DrawText(position, "%s", Name.c_str());
        position.y += YSizeLine;
    }
    if (m_Focus != Focus_Out || ExpandInNewMenu == false)
    {
        position.x += XSizeMenuItemIndent;
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
            it != m_pMenuItems.end();
            ++it)
        {
            if (this->IsValueSelected(std::distance(m_pMenuItems.begin(), it)))
            {
                s2d::SetTextColor(ColorMenuTextStrong);
            }
            (*it)->Print(position, depth + 1);
            s2d::SetTextColor(ColorMenuTextDefault);
        }
        position.x -= XSizeMenuItemIndent;
    }
}

bool MenuItemHolder::IsChildMenuOpened() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_Focus, Focus_Out);

    if (m_Focus == Focus_Child)
    {
        return m_pMenuItems.at(m_ItemIndex)->IsChildMenuOpened() | ExpandInNewMenu;
    }

    return ExpandInNewMenu;
}

void MenuItemHolder::NextValue(bool recursive) NN_NOEXCEPT
{
    if (m_Focus == Focus_Out || recursive == false)
    {
        return;
    }

    m_pMenuItems.at(m_ItemIndex)->NextValue(m_Focus == Focus_Child);
}

void MenuItemHolder::PrevValue(bool recursive) NN_NOEXCEPT
{
    if (m_Focus == Focus_Out || recursive == false)
    {
        return;
    }

    m_pMenuItems.at(m_ItemIndex)->PrevValue(m_Focus == Focus_Child);
}

void MenuItemHolder::NextItem() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_Focus, Focus_Out);

    if (m_Focus == Focus_Child)
    {
        m_pMenuItems.at(m_ItemIndex)->NextItem();
        return;
    }

    if (m_FunctionalItemCount == 0)
    {
        return;
    }

    while (true)
    {
        ++m_ItemIndex;
        if (m_ItemIndex >= m_pMenuItems.size())
        {
            m_ItemIndex = 0;
        }
        if (m_pMenuItems.at(m_ItemIndex)->IsFunctional == true && this->IsValueSelected(m_ItemIndex) == false)
        {
            return;
        }
    }
}

void MenuItemHolder::PrevItem() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(m_Focus, Focus_Out);

    if (m_Focus == Focus_Child)
    {
        m_pMenuItems.at(m_ItemIndex)->PrevItem();
        return;
    }

    if (m_FunctionalItemCount == 0)
    {
        return;
    }

    while (true)
    {
        --m_ItemIndex;
        if (m_ItemIndex < 0)
        {
            m_ItemIndex = m_pMenuItems.size() - 1;
        }
        if (m_pMenuItems.at(m_ItemIndex)->IsFunctional == true && this->IsValueSelected(m_ItemIndex) == false)
        {
            return;
        }
    }
}

bool MenuItemHolder::Select() NN_NOEXCEPT
{
    if (m_Focus == Focus_Out)
    {
        if (m_FunctionalItemCount == 0)
        {
            // 選択可能なアイテムが1つもない場合は自身を実行して戻る
            this->Proceed();
            return true;
        }
        else
        {
            m_ItemIndex = 0;
            while (true)
            {
                if (m_ItemIndex >= m_pMenuItems.size())
                {
                    m_ItemIndex = 0;
                }
                if (m_pMenuItems.at(m_ItemIndex)->IsFunctional == true)
                {
                    break;
                }
                ++m_ItemIndex;
            }
        }

        m_Focus = Focus_On;
        return false;
    }

    auto result = m_pMenuItems.at(m_ItemIndex)->Select();

    for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
         it != m_pMenuItems.end();
         ++it)
    {
        (*it)->ClearSelected();
    }

    if (result == true && m_Focus == Focus_On)
    {
        if (m_SelectedCount < SelectionCountMax)
        {
            m_ValuesSelected[m_SelectedCount] = m_ItemIndex;
            ++m_SelectedCount;
            result = false;
        }

        if (m_SelectedCount == SelectionCountMax)
        {
            this->Proceed();
            m_SelectedCount = 0;
            result = true;
            m_Focus = Focus_Out;
        }
    }
    else
    {
        m_Focus = result ? Focus_Out : Focus_Child;
    }
    return result;
}

bool MenuItemHolder::Cancel() NN_NOEXCEPT
{
    switch (m_Focus)
    {
    case Focus_On:
        if (m_SelectedCount > 0)
        {
            --m_SelectedCount;
        }
        else if (m_SelectedCount == 0)
        {
            m_Focus = Focus_Out;
            for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
                it != m_pMenuItems.end();
                ++it)
            {
                (*it)->ClearSelected();
            }
            return true;
        }
        return false;

    case Focus_Child:
        m_Focus = m_pMenuItems.at(m_ItemIndex)->Cancel() ? Focus_On : Focus_Child;
        return false;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void MenuItemHolder::GetItemIndex(int* pOutIndex, int* pOutDepth) NN_NOEXCEPT
{
    *pOutIndex = 0;
    *pOutDepth = 0;

    if (m_Focus == Focus_Child)
    {
        if (m_pMenuItems.at(m_ItemIndex)->IsChildMenuOpened())
        {
            m_pMenuItems.at(m_ItemIndex)->GetItemIndex(pOutIndex, pOutDepth);
            return;
        }
    }

    if (Name.size() > 0)
    {
        ++(*pOutIndex);
    }

    for (int index = 0; index < m_ItemIndex; ++index)
    {
        *pOutIndex += m_pMenuItems.at(index)->GetSize();
    }

    if (m_Focus == Focus_Child)
    {
        int index = 0;
        int depth = 0;
        m_pMenuItems.at(m_ItemIndex)->GetItemIndex(&index, &depth);
        *pOutIndex += index;
        *pOutDepth += depth;
        ++(*pOutDepth);
    }
}

int MenuItemHolder::GetSize() NN_NOEXCEPT
{
    int value = 0;
    if (ExpandInNewMenu == false)
    {
        for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
            it != m_pMenuItems.end();
            ++it)
        {
            value += (*it)->GetSize();
        }
    }

    if (Name.size() > 0)
    {
        ++value;
    }
    return value;
}

void MenuItemHolder::Update() NN_NOEXCEPT
{
    for (std::vector<MenuItem*>::iterator it = m_pMenuItems.begin();
         it != m_pMenuItems.end();
         ++it)
    {
        (*it)->Update();
    }
}


void MenuItemHolder::AddMenuItem(MenuItem* pMenuItem) NN_NOEXCEPT
{
    if(pMenuItem->IsFunctional)
    {
        ++m_FunctionalItemCount;
    }
    m_pMenuItems.push_back(pMenuItem);
}

bool MenuItemHolder::IsValueSelected(int value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(m_SelectedCount, SelectionCountMax);

    for (int i = 0; i < m_SelectedCount; ++i)
    {
        if (m_ValuesSelected[i] == value)
        {
            return true;
        }
    }

    return false;
}

MenuItemBoolean::MenuItemBoolean(const std::string& name) NN_NOEXCEPT:
    MenuItem(name, true)
{
    this->AddSelectable("\uE14B");
    this->AddSelectable("\uE14C");
}

bool MenuItemBoolean::IsEnabled() NN_NOEXCEPT
{
    return (m_Value == 0);
}

void MenuItemBoolean::UpdateValue(bool enabled) NN_NOEXCEPT
{
    m_Value = enabled ? 0 : 1;
}

MenuItemNpadIdsSelectable::MenuItemNpadIdsSelectable(const std::string& name, const bool& isHandheldSupported, const int& selectionCountMax) NN_NOEXCEPT:
    MenuItemHolder(name, true, true, selectionCountMax),
    IsHandheldSupported(isHandheldSupported)
{
    AddMenuItem(new MenuItem("NpadId::No1", true));
    AddMenuItem(new MenuItem("NpadId::No2", true));
    AddMenuItem(new MenuItem("NpadId::No3", true));
    AddMenuItem(new MenuItem("NpadId::No4", true));
    AddMenuItem(new MenuItem("NpadId::No5", true));
    AddMenuItem(new MenuItem("NpadId::No6", true));
    AddMenuItem(new MenuItem("NpadId::No7", true));
    AddMenuItem(new MenuItem("NpadId::No8", true));
    if (IsHandheldSupported)
    {
        AddMenuItem(new MenuItem("NpadId::Handheld", true));
    }
}

int MenuItemNpadIdsSelectable::GetValue(nn::hid::NpadIdType* pOutIds, int size)
{
    NN_SDK_REQUIRES_GREATER_EQUAL(size, m_SelectedCount);

    for (int i = 0; i < m_SelectedCount; ++i)
    {
        pOutIds[i] = this->GetNpadIdTypeFromValue(m_ValuesSelected[i]);
    }
    return m_SelectedCount;
}

nn::hid::NpadIdType MenuItemNpadIdsSelectable::GetNpadIdTypeFromValue(int value) NN_NOEXCEPT
{
    switch (value)
    {
    case 0:
        return nn::hid::NpadId::No1;
    case 1:
        return nn::hid::NpadId::No2;
    case 2:
        return nn::hid::NpadId::No3;
    case 3:
        return nn::hid::NpadId::No4;
    case 4:
        return nn::hid::NpadId::No5;
    case 5:
        return nn::hid::NpadId::No6;
    case 6:
        return nn::hid::NpadId::No7;
    case 7:
        return nn::hid::NpadId::No8;
    case 8:
        return nn::hid::NpadId::Handheld;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}
