﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <string>

#include "LineChart.h"
#include "ScreenStatics.h"
#include "s2d/Simple2D.h"

namespace {

template <typename T>
inline T Clamp(const T& value, const T& maxRange, const T& minRange) NN_NOEXCEPT
{
    if (value > maxRange)
    {
        return maxRange;
    }
    if (value < minRange)
    {
        return minRange;
    }
    return value;
}

} // namespace

LineChart::LineChart(const LineChartUiParameterConfigList& config, bool yInFloat) NN_NOEXCEPT :
    m_UiParameterConfig(config),
    m_YInFloat(yInFloat)
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_UiParameter); ++i)
    {
        m_UiParameter[i] = m_UiParameterConfig.v[i].init;
    }
    UpdateUiParameter();
}

void LineChart::Increment(UiParameterType param) NN_NOEXCEPT
{
    switch (param)
    {
    case UiParameterType_XSampleCount:
    case UiParameterType_YScale:
        m_UiParameter[param] = std::min(m_UiParameterConfig.v[param].max,
                                        m_UiParameter[param] * m_UiParameterConfig.v[param].step);
        break;
    case UiParameterType_YShift:
        m_UiParameter[param] = m_UiParameter[param] + (m_Range) / 4;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    UpdateUiParameter();
}

void LineChart::Decrement(UiParameterType param) NN_NOEXCEPT
{
    switch (param)
    {
    case UiParameterType_XSampleCount:
    case UiParameterType_YScale:
        m_UiParameter[param] = std::max(m_UiParameterConfig.v[param].min,
                                        m_UiParameter[param] / m_UiParameterConfig.v[param].step);
        break;
    case UiParameterType_YShift:
        m_UiParameter[param] = m_UiParameter[param] - (m_Range) / 4;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    UpdateUiParameter();
}

void LineChart::Draw(s2d::Point2D position,
                 s2d::Size    size,
                 bool plot,
                 ILineChartDataSetAccessor* pDataSets[],
                 int count) NN_NOEXCEPT
{
    auto graphPosition = position;
    graphPosition.x += XSizeLineChartLabel;
    DrawFrame(graphPosition, size);
    DrawSamples(graphPosition, size, plot, pDataSets, count);
    DrawValue(graphPosition, size, pDataSets, count);
}

void LineChart::DrawFrame(s2d::Point2D position,
                      s2d::Size    size) NN_NOEXCEPT
{
    s2d::Rectangle rectangle;
    rectangle.position = position;
    rectangle.size = size;
    s2d::DrawRectangle(rectangle, ColorSceneBorderBold, WidthBorderDefault);

    s2d::Point2D lineStart = { { position.x, position.y } };
    s2d::Point2D lineEnd   = { { position.x + size.width, position.y } };

    s2d::SetTextScale(TextScaleSmall);
    s2d::Point2D textPosition = { { position.x - XSizeLineChartLabel, position.y } };

    auto maxRange = m_UiParameter[UiParameterType_YShift] + m_Range;
    auto minRange = m_UiParameter[UiParameterType_YShift] - m_Range;

    // 単位目盛の補助線の描画
    float startingPosition;
    startingPosition = static_cast<int>(minRange / m_YGridUnit) * m_YGridUnit;

    for (auto unit = startingPosition; unit < maxRange; unit += m_YGridUnit)
    {
        if (unit <= minRange)
        {
            continue;
        }
        float yPosition = position.y + size.height * ((maxRange - unit) / (maxRange - minRange));
        lineStart.y = yPosition;
        lineEnd.y = yPosition;
        s2d::DrawLine(lineStart, lineEnd, ColorSceneBorderLight, WidthBorderDefault);
        textPosition.y = yPosition - TextScaleSmall.y / 2 * TextSize;
        if (m_YInFloat)
        {
            s2d::DrawText(textPosition, "%3.2f", unit);
        }
        else
        {
            s2d::DrawText(textPosition, "%d", static_cast<int>(unit));
        }
    }

    textPosition.x = position.x;
    textPosition.y = position.y + size.height + TextScaleSmall.y * TextSize;
    s2d::DrawText(textPosition, "0");
    textPosition.x += size.width / 2;
    s2d::DrawText(textPosition, "%d", m_SampleCountMax / 2);
    textPosition.x += size.width / 2;
    s2d::DrawText(textPosition, "%d", m_SampleCountMax);

    s2d::SetTextScale(TextScaleDefault);
}

void LineChart::DrawSamples(s2d::Point2D position,
                        s2d::Size    size,
                        bool plot,
                        ILineChartDataSetAccessor* pDataSets[],
                        int count) NN_NOEXCEPT
{
    auto maxRange = m_UiParameter[UiParameterType_YShift] + m_Range;
    auto minRange = m_UiParameter[UiParameterType_YShift] - m_Range;

    for (int dataSetIndex = 0; dataSetIndex < count; ++dataSetIndex)
    {
        auto& dataSet = *(pDataSets[dataSetIndex]);
        int sampleCount = std::min(static_cast<int>(dataSet.size()), m_SampleCountMax);
        float xPosition = position.x + size.width * (static_cast<float>(m_SampleCountMax - sampleCount) / (m_SampleCountMax - 1));
        if (sampleCount > 1)
        {
            int startingIndex = (dataSet.size() > sampleCount) ?
                                 dataSet.size() - sampleCount :
                                 0;
            for (int sampleIndex = 0; sampleIndex < (sampleCount - 1); ++sampleIndex)
            {
                auto first = Clamp(dataSet.at(startingIndex + sampleIndex), maxRange, minRange);
                auto second = Clamp(dataSet.at(startingIndex + sampleIndex + 1), maxRange, minRange);
                s2d::Point2D lineStart = { { xPosition,
                                             position.y + size.height * ((maxRange - first) / (maxRange - minRange))
                                         } };
                if (plot)
                {
                    s2d::FillCircle(lineStart, SizeLineChartPlot, dataSet.color());
                }
                xPosition += (size.width / (m_SampleCountMax - 1));
                s2d::Point2D lineEnd = { { xPosition,
                                           position.y + size.height * ((maxRange - second) / (maxRange - minRange))
                                         } };
                s2d::DrawLine(lineStart, lineEnd, dataSet.color(), WidthBorderDefaultBold);
            }
        }
    }
}

void LineChart::DrawValue(s2d::Point2D position,
                      s2d::Size    size,
                      ILineChartDataSetAccessor* pDataSets[],
                      int count) NN_NOEXCEPT
{
    auto valueYPosition = position.y + size.height + YSizeLineSmall;

    position.x = position.x - XSizeLineChartLabel;
    position.y = valueYPosition + YSizeLineSmall * 2;

    s2d::SetTextScale(TextScaleSmall);
    s2d::DrawText(position, "value");
    position.y += YSizeLineSmall;
    s2d::DrawText(position, "max");
    position.y += YSizeLineSmall;
    s2d::DrawText(position, "min");

    position.x += XSizeLineChartLabel;

    for (int dataSetIndex = 0; dataSetIndex < count; ++dataSetIndex)
    {
        auto& dataSet = *(pDataSets[dataSetIndex]);
        if (dataSet.size() > 0)
        {
            position.y = valueYPosition + YSizeLineSmall;
            s2d::SetTextColor(dataSet.color());
            s2d::DrawText(position, "■");
            s2d::SetTextColor(ColorSceneTextDefault);
            s2d::DrawText({ { position.x + XSizeLineChartColorIndex, position.y} }, "%s", dataSet.label());
            position.y += YSizeLineSmall;
            char outBuffer[256];
            s2d::DrawText(position, "%s", dataSet.backStr(outBuffer));
            position.y += YSizeLineSmall;
            s2d::DrawText(position, "%s", dataSet.maxStr(outBuffer));
            position.y += YSizeLineSmall;
            s2d::DrawText(position, "%s", dataSet.minStr(outBuffer));
            position.x += XSizeLineChartValue;
        }
    }

    s2d::SetTextScale(TextScaleDefault);
}

void LineChart::UpdateUiParameter() NN_NOEXCEPT
{
    // Y軸方向の表示レンジ
    m_Range = m_UiParameter[UiParameterType_YScale] * 1.25f;

    // X軸方向の表示サンプル数
    m_SampleCountMax = static_cast<int>(m_UiParameter[UiParameterType_XSampleCount]);

    // 縦方向のクランプ
    m_UiParameter[UiParameterType_YShift] = std::min(m_UiParameterConfig.v[UiParameterType_YShift].max - m_UiParameter[UiParameterType_YScale],
                                                     m_UiParameter[UiParameterType_YShift]);
    m_UiParameter[UiParameterType_YShift] = std::max(m_UiParameterConfig.v[UiParameterType_YShift].min - m_UiParameter[UiParameterType_YScale],
                                                     m_UiParameter[UiParameterType_YShift]);

    // Y軸の補助線の表示単位
    if (m_UiParameter[UiParameterType_YScale] == m_UiParameterConfig.v[UiParameterType_YScale].max ||
        m_UiParameter[UiParameterType_YScale] == m_UiParameterConfig.v[UiParameterType_YScale].min ||
        m_UiParameter[UiParameterType_YScale] == m_UiParameterConfig.v[UiParameterType_YScale].init)
    {
        m_YGridUnit = m_UiParameter[UiParameterType_YScale] / 4;
        return;
    }

    for (int i = 8; i >= -8; --i)
    {
        for (int j = 5; j > 0; --j)
        {
            auto unit = std::pow(10, i) * 2 * j;
            if (m_UiParameter[UiParameterType_YScale] > unit)
            {
                m_YGridUnit = unit / 2;
                return;
            }
        }
    }
}
