﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/hid/hid_Vibration.h>

namespace {

//!< メモリ上に展開済みの振動ファイルデータ (ループ情報なし)
const uint8_t BnvibData1[] = {
    0x04, 0x00, 0x00, 0x00, 0x03, 0x00, 0xC8, 0x00, 0x40, 0x00, 0x00, 0x00, 0xFF, 0x80, 0x00, 0xA0,
    0xEE, 0x80, 0x00, 0xA0, 0xDD, 0x80, 0x00, 0xA0, 0xCC, 0x80, 0x00, 0xA0, 0xFF, 0x70, 0x00, 0xA0,
    0xFF, 0x78, 0x00, 0xA0, 0xFF, 0x88, 0x00, 0xA0, 0xFF, 0x90, 0x00, 0xA0, 0x00, 0x80, 0xFF, 0xA0,
    0x00, 0x80, 0xEE, 0xA0, 0x00, 0x80, 0xDD, 0xA0, 0x00, 0x80, 0xCC, 0xA0, 0x00, 0x80, 0xFF, 0x90,
    0x00, 0x80, 0xFF, 0x98, 0x00, 0x80, 0xFF, 0xA8, 0x00, 0x80, 0xFF, 0xB0};

//!< メモリ上に展開済みの振動ファイルデータ (ループ情報あり、ループインターバル情報なし)
const uint8_t BnvibData2[] = {
    0x0C, 0x00, 0x00, 0x00, 0x03, 0x00, 0xC8, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0D, 0x00, 0x00, 0x00,
    0x40, 0x00, 0x00, 0x00, 0xFF, 0x80, 0x00, 0xA0, 0xEE, 0x80, 0x00, 0xA0, 0xDD, 0x80, 0x00, 0xA0,
    0xCC, 0x80, 0x00, 0xA0, 0xFF, 0x70, 0x00, 0xA0, 0xFF, 0x78, 0x00, 0xA0, 0xFF, 0x88, 0x00, 0xA0,
    0xFF, 0x90, 0x00, 0xA0, 0x00, 0x80, 0xFF, 0xA0, 0x00, 0x80, 0xEE, 0xA0, 0x00, 0x80, 0xDD, 0xA0,
    0x00, 0x80, 0xCC, 0xA0, 0x00, 0x80, 0xFF, 0x90, 0x00, 0x80, 0xFF, 0x98, 0x00, 0x80, 0xFF, 0xA8,
    0x00, 0x80, 0xFF, 0xB0};

//!< メモリ上に展開済みの振動ファイルデータ (ループ情報あり、ループインターバル情報あり)
const uint8_t BnvibData3[] = {
    0x10, 0x00, 0x00, 0x00, 0x03, 0x00, 0xC8, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0D, 0x00, 0x00, 0x00,
    0x64, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0xFF, 0x80, 0x00, 0xA0, 0xEE, 0x80, 0x00, 0xA0,
    0xDD, 0x80, 0x00, 0xA0, 0xCC, 0x80, 0x00, 0xA0, 0xFF, 0x70, 0x00, 0xA0, 0xFF, 0x78, 0x00, 0xA0,
    0xFF, 0x88, 0x00, 0xA0, 0xFF, 0x90, 0x00, 0xA0, 0x00, 0x80, 0xFF, 0xA0, 0x00, 0x80, 0xEE, 0xA0,
    0x00, 0x80, 0xDD, 0xA0, 0x00, 0x80, 0xCC, 0xA0, 0x00, 0x80, 0xFF, 0x90, 0x00, 0x80, 0xFF, 0x98,
    0x00, 0x80, 0xFF, 0xA8, 0x00, 0x80, 0xFF, 0xB0};

//!< BnvibData に記録されていると期待される振動値
const ::nn::hid::VibrationValue ExpectedVibrationValueArray[] = {
    ::nn::hid::VibrationValue::Make(1.0000f, 160.00f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.9333f, 160.00f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.8667f, 160.00f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.8000f, 160.00f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(1.0000f, 113.14f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(1.0000f, 134.54f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(1.0000f, 190.27f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(1.0000f, 226.27f, 0.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.9333f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.8667f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 0.8000f, 320.00f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 226.27f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 269.09f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 380.55f),
    ::nn::hid::VibrationValue::Make(0.0000f, 160.00f, 1.0000f, 452.55f)
};

//!< テスト対象となるデータのセット
struct DataSet
{
    const void* address;                //!< 振動ファイルのバイナリが展開されているアドレス
    size_t fileSize;                    //!< 振動ファイルのファイルサイズ
    int32_t expectedSampleLength;       //!< sampleLength の期待値
    bool expectedIsLoop;                //!< isLoop の期待値
    uint32_t expectedLoopStartPosition; //!< loopStartPosition の期待値
    uint32_t expectedLoopEndPosition;   //!< loopEndPosition の期待値
    uint32_t expectedLoopInterval;      //!< loopInterval の期待値

    //!< 引数に指定された VibrationPlayer を使って振動データをロードし、期待値通りの結果になるかテストします
    void DoLoadTest(::nn::hid::VibrationPlayer* pPlayer) const NN_NOEXCEPT
    {
        NNT_EXPECT_RESULT_SUCCESS(pPlayer->Load(address, fileSize));
        EXPECT_TRUE(pPlayer->IsLoaded());
        ::nn::hid::VibrationFileInfo fileInfo = pPlayer->GetFileInfo();
        EXPECT_EQ(expectedSampleLength, fileInfo.sampleLength);
    }
};

const DataSet DataSet1 = { BnvibData1, sizeof(BnvibData1), 16, false, 0, 16, 0 };
const DataSet DataSet2 = { BnvibData2, sizeof(BnvibData2), 16, true, 3, 13, 0 };
const DataSet DataSet3 = { BnvibData3, sizeof(BnvibData3), 16, true, 3, 13, 100 };

struct LoopRange
{
    int start;
    int end;
};

//!< 振動値同士の比較テストをします
void TestVibrationValue(const ::nn::hid::VibrationValue& expected, const ::nn::hid::VibrationValue& actual) NN_NOEXCEPT
{
    const float AmplitudeError = 0.001f;
    const float FrequencyError = 0.1f;

    EXPECT_NEAR(expected.amplitudeLow, actual.amplitudeLow, AmplitudeError);
    EXPECT_NEAR(expected.frequencyLow, actual.frequencyLow, FrequencyError);
    EXPECT_NEAR(expected.amplitudeHigh, actual.amplitudeHigh, AmplitudeError);
    EXPECT_NEAR(expected.frequencyHigh, actual.frequencyHigh, FrequencyError);
}

//!< ロード後のVibrationPlayer の状態をテストします
void TestVibrationPlayerStatus(
    const ::nn::hid::VibrationPlayer* pPlayer,
    bool expectedIsPlaying,
    int expectedPosition,
    const ::nn::hid::VibrationValue& expectedVibration) NN_NOEXCEPT
{
    EXPECT_TRUE(pPlayer->IsLoaded());
    EXPECT_EQ(expectedIsPlaying, pPlayer->IsPlaying());
    EXPECT_EQ(expectedPosition, pPlayer->GetCurrentPosition());
    TestVibrationValue(expectedVibration, pPlayer->GetCurrentVibration());
}

//!< ロード後のVibrationPlayer の状態をテストします (期待される振動値は ExpectedVibrationValueArray からとります)
void TestVibrationPlayerStatus(
    const ::nn::hid::VibrationPlayer* pPlayer,
    bool expectedIsPlaying,
    int expectedPosition) NN_NOEXCEPT
{
    TestVibrationPlayerStatus(
        pPlayer,
        expectedIsPlaying,
        expectedPosition,
        ExpectedVibrationValueArray[expectedPosition]);
}

}

//! 振動ファイルのロード前、ロード後、アンロード後のステータスが想定通りか
TEST(VibrationPlayerSuite, VibrationPlayerLoadAndUnload)
{
    const int UpdateLoopCount = 10;
    const int LargeLoopCount = 2;

    // 初期化直後、ロード前
    ::nn::hid::VibrationPlayer player;
    EXPECT_FALSE(player.IsLoaded());
    for(int i = 0; i < UpdateLoopCount; i++)
    {
        // ロード前は Update しても状態が不変であることを確認
        ::nn::hid::VibrationNode::Update();
        EXPECT_FALSE(player.IsLoaded());
    }

    for(int largeLoop = 0; largeLoop < LargeLoopCount; largeLoop++)
    {
        // ロード後
        NNT_EXPECT_RESULT_SUCCESS(player.Load(BnvibData1, sizeof(BnvibData1)));
        TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < UpdateLoopCount; i++)
        {
            // プレイしていなければ Update しても状態が不変であることを確認
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        }

        // アンロード後
        player.Unload();
        EXPECT_FALSE(player.IsLoaded());
        for(int i = 0; i < UpdateLoopCount; i++)
        {
            // アンロード後は Update しても状態が不変であることを確認
            ::nn::hid::VibrationNode::Update();
            EXPECT_FALSE(player.IsLoaded());
        }
    }
}

//!< VibrationPlayer が振動ファイルを正しく読んで再生できるか (ループ情報なし)
TEST(VibrationPlayerSuite, VibrationPlayerPlayBasic1)
{
    const int UpdateLoopCount = 10;
    const int LargeLoopCount = 2;
    const int SampleLength = DataSet1.expectedSampleLength;

    ::nn::hid::VibrationPlayer player;

    // ファイル情報が正しく読めているか
    DataSet1.DoLoadTest(&player);

    // プレイ前の状態を確認
    TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());

    for(int largeLoop = 0; largeLoop < LargeLoopCount; largeLoop++)
    {
        // プレイ中の状態を確認
        player.Play();
        TestVibrationPlayerStatus(&player, true, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < SampleLength; i++)
        {
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, true, i);
        }

        // プレイ後の状態を確認 (自動的に停止状態になっているはず)
        ::nn::hid::VibrationNode::Update();
        TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < UpdateLoopCount; i++)
        {
            // プレイしていなければ Update しても状態が不変であることを確認
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        }
    }
}

//!< VibrationPlayer が振動ファイルを正しく読んで再生できるか (ループ情報あり、ループインターバル情報なし)
TEST(VibrationPlayerSuite, VibrationPlayerPlayBasic2)
{
    const int UpdateLoopCount = 10;
    const int SmallLoopCount = 5;
    const int LargeLoopCount = 2;

    ::nn::hid::VibrationPlayer player;

    // ファイル情報が正しく読めているか
    DataSet2.DoLoadTest(&player);

    // プレイ前の状態を確認
    TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());

    for(int largeLoop = 0; largeLoop < LargeLoopCount; largeLoop++)
    {
        // プレイ中の状態を確認
        player.Play();
        TestVibrationPlayerStatus(&player, true, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < player.GetLoopEndPosition(); i++)
        {
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, true, i);
        }

        // ループ中の状態
        for(int smallLoop = 0; smallLoop < SmallLoopCount; smallLoop++)
        {
            int loopLength = player.GetLoopEndPosition() - player.GetLoopStartPosition();
            for(int i = 0; i < loopLength; i++)
            {
                ::nn::hid::VibrationNode::Update();
                TestVibrationPlayerStatus(&player, true, player.GetLoopStartPosition() + i);
            }
        }

        // ストップして先頭に戻る
        player.Stop();
        player.SetCurrentPosition(0);
        ::nn::hid::VibrationNode::Update();
        TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < UpdateLoopCount; i++)
        {
            // プレイしていなければ Update しても状態が不変であることを確認
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        }
    }
}

//!< VibrationPlayer が振動ファイルを正しく読んで再生できるか (ループ情報あり、ループインターバル情報あり)
TEST(VibrationPlayerSuite, VibrationPlayerPlayBasic3)
{
    const int UpdateLoopCount = 10;
    const int SmallLoopCount = 5;
    const int LargeLoopCount = 2;

    ::nn::hid::VibrationPlayer player;

    // ファイル情報が正しく読めているか
    DataSet3.DoLoadTest(&player);

    // プレイ前の状態を確認
    TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());

    for(int largeLoop = 0; largeLoop < LargeLoopCount; largeLoop++)
    {
        // プレイ中の状態を確認
        player.Play();
        TestVibrationPlayerStatus(&player, true, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < player.GetLoopEndPosition(); i++)
        {
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, true, i);
        }

        // ループ中の状態
        for(int smallLoop = 0; smallLoop < SmallLoopCount; smallLoop++)
        {
            int loopInterval = player.GetLoopInterval();
            for(int i = 0; i < loopInterval; i++)
            {
                ::nn::hid::VibrationNode::Update();
                TestVibrationPlayerStatus(&player, true, -loopInterval + i, ::nn::hid::VibrationValue::Make());
            }

            int loopLength = player.GetLoopEndPosition() - player.GetLoopStartPosition();
            for(int i = 0; i < loopLength; i++)
            {
                ::nn::hid::VibrationNode::Update();
                TestVibrationPlayerStatus(&player, true, player.GetLoopStartPosition() + i);
            }
        }

        // ストップして先頭に戻る
        player.Stop();
        player.SetCurrentPosition(0);
        ::nn::hid::VibrationNode::Update();
        TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < UpdateLoopCount; i++)
        {
            // プレイしていなければ Update しても状態が不変であることを確認
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
        }
    }
}

//!< VibrationPlayer が振動ファイル内で再生位置を正しく変更できるか (停止した状態での SetPosition)
TEST(VibrationPlayerSuite, VibrationPlayerSetPosition1)
{
    const int UpdateLoopCount = 10;
    const int SampleLength = DataSet1.expectedSampleLength;
    const int StopPosition = SampleLength - 1;
    const int PositionList[] = {
        0,
        SampleLength / 3,
        SampleLength * 2 / 3,
        SampleLength - 1};

    ::nn::hid::VibrationPlayer player;
    DataSet1.DoLoadTest(&player);

    // 一度途中までプレイ
    player.Play();
    TestVibrationPlayerStatus(&player, true, 0, ::nn::hid::VibrationValue::Make());
    for(int i = 0; i < StopPosition; i++)
    {
        ::nn::hid::VibrationNode::Update();
        TestVibrationPlayerStatus(&player, true, i);
    }

    for(auto& pos : PositionList)
    {
        // 一旦停止して SetPosition
        player.Stop();
        TestVibrationPlayerStatus(&player, false, StopPosition, ::nn::hid::VibrationValue::Make());
        player.SetCurrentPosition(pos);
        TestVibrationPlayerStatus(&player, false, pos, ::nn::hid::VibrationValue::Make());

        for(int i = 0; i < UpdateLoopCount; i++)
        {
            // プレイしていなければ Update しても状態が不変であることを確認
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, false, pos, ::nn::hid::VibrationValue::Make());
        }

        // 途中までプレイ
        player.Play();
        TestVibrationPlayerStatus(&player, true, pos, ::nn::hid::VibrationValue::Make());
        for(int i = 0; i < StopPosition - pos; i++)
        {
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, true, pos + i);
        }
    }

    // この時点でファイル末尾に到達しているはず
    ::nn::hid::VibrationNode::Update();
    TestVibrationPlayerStatus(&player, true, SampleLength - 1);
    ::nn::hid::VibrationNode::Update();
    TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
}

//!< VibrationPlayer が振動ファイル内で再生位置を正しく変更できるか (プレイ中の SetPosition)
TEST(VibrationPlayerSuite, VibrationPlayerSetPosition2)
{
    const int SampleLength = DataSet1.expectedSampleLength;
    const int StopPosition = SampleLength - 1;
    const int PositionList[] = {
        0,
        SampleLength / 3,
        SampleLength * 2 / 3,
        SampleLength - 1};

    ::nn::hid::VibrationPlayer player;
    DataSet1.DoLoadTest(&player);

    // 一度途中までプレイ
    player.Play();
    TestVibrationPlayerStatus(&player, true, 0, ::nn::hid::VibrationValue::Make());
    for(int i = 0; i < StopPosition; i++)
    {
        ::nn::hid::VibrationNode::Update();
        TestVibrationPlayerStatus(&player, true, i);
    }

    for(auto& pos : PositionList)
    {
        // 停止せずに SetPosition
        player.SetCurrentPosition(pos);
        TestVibrationPlayerStatus(&player, true, pos);

        // そのまま途中までプレイ
        for(int i = 0; i < StopPosition - pos; i++)
        {
            ::nn::hid::VibrationNode::Update();
            TestVibrationPlayerStatus(&player, true, pos + i);
        }
    }

    // この時点でファイル末尾に到達しているはず
    ::nn::hid::VibrationNode::Update();
    TestVibrationPlayerStatus(&player, true, SampleLength - 1);
    ::nn::hid::VibrationNode::Update();
    TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
}

//!< VibrationPlayer のループ関連機能が正常に動作するか (ループ解除はしない)
TEST(VibrationPlayerSuite, VibrationPlayerLoop1)
{
    const int LoopCount = 3;
    const int SampleLength = DataSet1.expectedSampleLength;

    const LoopRange LoopRangeList[] = {
        { 3, SampleLength - 3 },
        { 1, SampleLength - 1 },
        { 0, SampleLength - 3 },
        { 3, SampleLength },
        { 0, SampleLength },
    };
    const int LoopIntervalList[] = {
        0, 1, 100
    };

    ::nn::hid::VibrationPlayer player;
    DataSet1.DoLoadTest(&player);

    for(auto& loopRange : LoopRangeList)
    {
        for(auto& loopInterval : LoopIntervalList)
        {
            // ポジションを冒頭に移す
            player.Stop();
            player.SetCurrentPosition(0);

            // ループ設定
            player.SetLoop(true);
            EXPECT_TRUE(player.IsLoop());
            player.SetLoopStartPosition(loopRange.start);
            EXPECT_EQ(loopRange.start, player.GetLoopStartPosition());
            player.SetLoopEndPosition(loopRange.end);
            EXPECT_EQ(loopRange.end, player.GetLoopEndPosition());
            player.SetLoopInterval(loopInterval);
            EXPECT_EQ(loopInterval, player.GetLoopInterval());

            // プレイ
            player.Play();
            TestVibrationPlayerStatus(&player, true, 0, ::nn::hid::VibrationValue::Make());

            // loopStartPosition までを正しく再生できているか
            for(int i = 0; i < loopRange.start; i++)
            {
                ::nn::hid::VibrationNode::Update();
                TestVibrationPlayerStatus(&player, true, i);
            }

            // ループ区間内を正しく再生できているか
            for(int loopIdx = 0; loopIdx < LoopCount; loopIdx++)
            {
                for(int i = 0; i < loopRange.end - loopRange.start; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                    TestVibrationPlayerStatus(&player, true, loopRange.start + i);
                }

                // インターバル区間が無振動になっているか
                for(int i = 0; i < loopInterval; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                    TestVibrationPlayerStatus(&player, true, i - loopInterval, ::nn::hid::VibrationValue::Make());
                }
            }
        }
    }
}

//!< VibrationPlayer のループ関連機能が正常に動作するか (ループ区間内でループ解除)
TEST(VibrationPlayerSuite, VibrationPlayerLoop2)
{
    const int LoopCount = 3;
    const int SampleLength = DataSet1.expectedSampleLength;

    const LoopRange LoopRangeList[] = {
        { 3, SampleLength - 3 },
        { 1, SampleLength - 1 },
        { 0, SampleLength - 3 },
        { 3, SampleLength },
        { 0, SampleLength },
    };
    const int LoopIntervalList[] = {
        0, 1, 100
    };

    ::nn::hid::VibrationPlayer player;
    DataSet1.DoLoadTest(&player);

    for(auto& loopRange : LoopRangeList)
    {
        for(auto& loopInterval : LoopIntervalList)
        {
            const int ExtraPlayCountList[] = {
                1, 2, loopRange.end - loopRange.start
            };

            for(auto& extraPlayCount : ExtraPlayCountList)
            {
                // ポジションを冒頭に移す
                player.Stop();
                player.SetCurrentPosition(0);

                // ループ設定
                player.SetLoop(true);
                player.SetLoopStartPosition(loopRange.start);
                player.SetLoopEndPosition(loopRange.end);
                player.SetLoopInterval(loopInterval);

                // プレイ
                int updateCount =
                    loopRange.start
                    + (loopRange.end - loopRange.start + loopInterval) * LoopCount
                    + extraPlayCount;
                player.Play();
                for(int i = 0; i < updateCount; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                }

                // ループ区間内でループ解除
                int expectedPosition = loopRange.start + extraPlayCount - 1;
                EXPECT_TRUE(player.IsLoop());
                TestVibrationPlayerStatus(&player, true, expectedPosition);
                player.SetLoop(false);
                EXPECT_FALSE(player.IsLoop());
                TestVibrationPlayerStatus(&player, true, expectedPosition);

                // ファイル末尾まで再生できるか
                for(int i = 1; i < SampleLength - expectedPosition; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                    TestVibrationPlayerStatus(&player, true, expectedPosition + i);
                }

                // 自動的に停止するはず
                ::nn::hid::VibrationNode::Update();
                TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
            }
        }
    }
}

//!< VibrationPlayer のループ関連機能が正常に動作するか (ループインターバル中にループ解除)
TEST(VibrationPlayerSuite, VibrationPlayerLoop3)
{
    const int LoopCount = 3;
    const int SampleLength = DataSet1.expectedSampleLength;

    const LoopRange LoopRangeList[] = {
        { 3, SampleLength - 3 },
        { 1, SampleLength - 1 },
        { 0, SampleLength - 3 },
        { 3, SampleLength },
        { 0, SampleLength },
    };
    const int LoopIntervalList[] = {
        5, 100
    };

    ::nn::hid::VibrationPlayer player;
    DataSet1.DoLoadTest(&player);

    for(auto& loopRange : LoopRangeList)
    {
        for(auto& loopInterval : LoopIntervalList)
        {
            const int ExtraPlayCountList[] = {
                1, 2, loopInterval
            };

            for(auto& extraPlayCount : ExtraPlayCountList)
            {
                // ポジションを冒頭に移す
                player.Stop();
                player.SetCurrentPosition(0);

                // ループ設定
                player.SetLoop(true);
                player.SetLoopStartPosition(loopRange.start);
                player.SetLoopEndPosition(loopRange.end);
                player.SetLoopInterval(loopInterval);

                // プレイ
                int updateCount =
                    loopRange.start
                    + (loopRange.end - loopRange.start + loopInterval) * LoopCount
                    + (loopRange.end - loopRange.start)
                    + extraPlayCount;
                player.Play();
                for(int i = 0; i < updateCount; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                }

                // ループインターバル内でループ解除
                int expectedPosition = -loopInterval + extraPlayCount - 1;
                EXPECT_TRUE(player.IsLoop());
                TestVibrationPlayerStatus(&player, true, expectedPosition, ::nn::hid::VibrationValue::Make());
                player.SetLoop(false);
                EXPECT_FALSE(player.IsLoop());
                TestVibrationPlayerStatus(&player, true, expectedPosition, ::nn::hid::VibrationValue::Make());

                // 残りのループインターバルの再生
                for(int i = 1; i <= loopInterval - extraPlayCount; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                    TestVibrationPlayerStatus(&player, true, expectedPosition + i, ::nn::hid::VibrationValue::Make());
                }

                // ループ区間先頭に飛んでからファイル末尾までの再生
                for(int i = 0; i < SampleLength - loopRange.start; i++)
                {
                    ::nn::hid::VibrationNode::Update();
                    TestVibrationPlayerStatus(&player, true, loopRange.start + i);
                }

                // 自動的に停止するはず
                ::nn::hid::VibrationNode::Update();
                TestVibrationPlayerStatus(&player, false, 0, ::nn::hid::VibrationValue::Make());
            }
        }
    }
}
