﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_TouchScreen.h"

template<typename T>
class TouchScreenReadingStateSuite : public ::testing::Test
{
protected:
    static ::nn::hid::TouchScreenState<T::Value>
        s_States[::nn::hid::TouchScreenStateCountMax];

    //!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
    static void ThreadingTest(int priority) NN_NOEXCEPT;
};

// TouchScreenState::touches の数が以下の場合についてそれぞれテストを実行する。
typedef ::testing::Types<
    ::nnt::hid::TouchStateCountType<::nn::hid::TouchStateCountMax>
    > TouchStateCountTypes;

TYPED_TEST_CASE(TouchScreenReadingStateSuite, TouchStateCountTypes);

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TYPED_TEST(TouchScreenReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::TouchScreenState<
             TypeParam::Value>& state : TestFixture::s_States)
    {
        state.samplingNumber = 0xDEADBEEF;
        state.count = 0xCAFE;
    }

    ::nn::hid::InitializeTouchScreen();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax / 2 *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetTouchScreenStates(
        TestFixture::s_States,
        ::nn::hid::TouchScreenStateCountMax);

    EXPECT_GE(::nn::hid::TouchScreenStateCountMax, count);

    for (int i = count; i < ::nn::hid::TouchScreenStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, TestFixture::s_States[i].samplingNumber);
        EXPECT_EQ(0xCAFE, TestFixture::s_States[i].count);
    }

    ::nn::hid::debug::FinalizeTouchScreen();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TYPED_TEST(TouchScreenReadingStateSuite, ThreadingTest1)
{
    TouchScreenReadingStateSuite<
        TypeParam>::ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TYPED_TEST(TouchScreenReadingStateSuite, ThreadingTest2)
{
    TouchScreenReadingStateSuite<
        TypeParam>::ThreadingTest(::nn::os::LowestThreadPriority);
}

template<typename T>
::nn::hid::TouchScreenState<T::Value>
    TouchScreenReadingStateSuite<T>::s_States[
        ::nn::hid::TouchScreenStateCountMax];

template<typename T>
void TouchScreenReadingStateSuite<T>::ThreadingTest(int priority) NN_NOEXCEPT
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::InitializeTouchScreen();

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::GetTouchScreenStates(
            s_States,
            ::nn::hid::TouchScreenStateCountMax);
        EXPECT_LE(lastCount, count);
        EXPECT_LE(lastSamplingNumber, s_States[0].samplingNumber);
        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::TouchScreenState<T::Value>& lhs = s_States[j];
            const ::nn::hid::TouchScreenState<T::Value>& rhs = s_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = s_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::TouchScreenStateCountMax, lastCount);

    ::nn::hid::debug::FinalizeTouchScreen();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}
