﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Result.h>

#include <nnt/nntest.h>

#include "../../../Programs/Iris/Sources/Libraries/hid/detail/hid_SixAxisSensorCalibrator.h"

namespace {

const ::nn::xcd::SensorState HorizontalAccelerometer = { 350, 0, 4081 };
const int32_t HorizontalAccelerometerRange = 10;
const int32_t AngularVelocityCountRange = 20;
const int32_t ValidSampleCountMax = 200;
const ::nn::xcd::SensorCalibrationValue CalibrationValue = {{ 0, 0, 0 },            //!< 加速度センサーの原点値
                                                            { 4096, 4096, 4096 },   //!< 加速度センサーの感度
                                                            { 0, 0, 0 },            //!< ジャイロセンサーの原点値
                                                            { 1024, 1024, 1024 } }; //!< ジャイロセンサーの感度
const ::nn::util::Float3 TypicalAccelerometerCount = {{{ 0, 0, 4096 }}};

void SetCalibrator(::nn::hid::detail::SixAxisSensorCalibrator* pOutCalibrator) NN_NOEXCEPT
{
    pOutCalibrator->Reset();

    EXPECT_TRUE(pOutCalibrator->IsCalibrated() != true);

    pOutCalibrator->SetCalibrationParameters(HorizontalAccelerometer,
                                             HorizontalAccelerometerRange,
                                             AngularVelocityCountRange,
                                             ValidSampleCountMax,
                                             CalibrationValue,
                                             TypicalAccelerometerCount);
}

void DumpSensorCalibrationValue(const ::nn::xcd::SensorCalibrationValue& calibrationValue) NN_NOEXCEPT
{
    NN_LOG("acc origin : (%d, %d, %d)\n", calibrationValue.accelerometerOrigin.x,
                                          calibrationValue.accelerometerOrigin.y,
                                          calibrationValue.accelerometerOrigin.z);

    NN_LOG("acc sensitivity : (%d, %d, %d)\n", calibrationValue.accelerometerSensitivity.x,
                                               calibrationValue.accelerometerSensitivity.y,
                                               calibrationValue.accelerometerSensitivity.z);

    NN_LOG("gyro origin : (%d, %d, %d)\n", calibrationValue.gyroscopeOrigin.x,
                                           calibrationValue.gyroscopeOrigin.y,
                                           calibrationValue.gyroscopeOrigin.z);

    NN_LOG("gyro sensitivity : (%d, %d, %d)\n", calibrationValue.gyroscopeSensitivity.x,
                                                calibrationValue.gyroscopeSensitivity.y,
                                                calibrationValue.gyroscopeSensitivity.z);

}

//!< Mean Squared Error を返します
template<typename T>
float GetMeanSquaredError(const T& state1, const T& state2) NN_NOEXCEPT
{
    T deltaState = state1;
    deltaState.x -= state2.x;
    deltaState.y -= state2.y;
    deltaState.z -= state2.z;

    return (deltaState.x * deltaState.x +
            deltaState.y * deltaState.y +
            deltaState.z * deltaState.z) / 3.f;
}


} // namespace

//!< SixAxisSensorCalibrator の水平判定が有効か
TEST(SixAxisSensorCalibrator, IsHorizontalTest1)
{
    nn::hid::detail::SixAxisSensorCalibrator calibrator;
    SetCalibrator(&calibrator);

    ::nn::xcd::SixAxisSensorState state = ::nn::xcd::SixAxisSensorState();

    // 水平判定範囲内のときに成功する
    state.sampleNumber = 1;
    state.accelerometer = ::nn::util::Float3{{{
        static_cast<float>(HorizontalAccelerometer.x),
        static_cast<float>(HorizontalAccelerometer.y),
        static_cast<float>(HorizontalAccelerometer.z)
    }}};
    calibrator.Update(state);
    EXPECT_TRUE(calibrator.IsHorizontal());

    // 水平判定範囲外のときに失敗する
    state.sampleNumber++;
    state.accelerometer.x += (HorizontalAccelerometerRange + 1);
    calibrator.Update(state);
    EXPECT_TRUE(!calibrator.IsHorizontal());
}

//!< SixAxisSensorCalibrator のキャリブレーション処理に成功するか
TEST(SixAxisSensorCalibrator, SuccessTest1)
{
    nn::hid::detail::SixAxisSensorCalibrator calibrator;
    SetCalibrator(&calibrator);

    ::nn::xcd::SixAxisSensorState state = ::nn::xcd::SixAxisSensorState();
    state.accelerometer = ::nn::util::Float3{{{
        static_cast<float>(HorizontalAccelerometer.x),
        static_cast<float>(HorizontalAccelerometer.y),
        static_cast<float>(HorizontalAccelerometer.z)
    }}};

    state.gyroscope = nn::util::Float3();

    for(int i = 0; i < ValidSampleCountMax; i++)
    {
        state.sampleNumber = i;
        calibrator.Update(state);
    }

    EXPECT_TRUE(calibrator.IsCalibrated());

    ::nn::xcd::SensorCalibrationValue calibrationValue;
    auto result = calibrator.GetUserCalibrationValue(&calibrationValue);
    EXPECT_TRUE(result.IsSuccess());

    // 平均二乗誤差が閾値以内であるか
    const float MseThreshold = 1.f;
    EXPECT_LE(GetMeanSquaredError(calibrationValue.accelerometerOrigin,
                                  CalibrationValue.accelerometerOrigin),
              MseThreshold);

    EXPECT_LE(GetMeanSquaredError(calibrationValue.accelerometerSensitivity,
                                  CalibrationValue.accelerometerSensitivity),
              MseThreshold);

    EXPECT_LE(GetMeanSquaredError(calibrationValue.gyroscopeOrigin,
                                  CalibrationValue.gyroscopeOrigin),
              MseThreshold);

    EXPECT_LE(GetMeanSquaredError(calibrationValue.gyroscopeSensitivity,
                                  CalibrationValue.gyroscopeSensitivity),
              MseThreshold);

    // 値の確認
    DumpSensorCalibrationValue(calibrationValue);
}
