﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_ResultController.h>
#include <nnt/nntest.h>

#include "testHid_NpadCommon.h"
#include "../Common/testHid_Npad.h"

//!< Npad 未初期化時はジョイコンの割り当て読み出しに失敗する
TEST(NpadJoyAssignmentTest, NpadNoInitialized)
{
    for (auto& npadId : ::testhid::NpadIdsMinimalSet)
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::hid::GetNpadJoyAssignment(npadId), "");
    }
}

//!< ジョイコンの割り当てモードの初期値は 2本持ち
TEST(NpadJoyAssignmentTest, DefaultDual)
{
    ::testhid::EnableIdAll();

    for (auto& npadId : ::testhid::NpadIds)
    {
        auto assignment = nn::hid::GetNpadJoyAssignment(npadId);

        EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);
    }

    ::testhid::DisableIdAll();
}

//!< ジョイコンの割り当てモードを1本持ちに変更して 2本持ちに戻す
TEST(NpadJoyAssignmentTest, SwitchToSingle)
{
    ::testhid::EnableIdAll();
    ::testhid::EnableStyleAll();

    for (int i = 0; i < ::testhid::NpadIdCountMax; i++)
    {
        auto npadId = ::testhid::NpadIds[i];
        nn::hid::SetNpadJoyAssignmentModeSingle(npadId);

        for (int j = 0; j < ::testhid::NpadIdCountMax; j++)
        {
            auto assignment = nn::hid::GetNpadJoyAssignment(::testhid::NpadIds[j]);
            if (j > i)
            {
                // Style 未変更
                // 他の Npad に対して値が影響しないことをチェック
                EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);
            }
            else
            {
                // Style 変更済み
                EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);
            }
        }
    }

    // 設定を Dual に戻す
    for (auto& npadId : ::testhid::NpadIds)
    {
        nn::hid::SetNpadJoyAssignmentModeDual(npadId);

        auto assignment = nn::hid::GetNpadJoyAssignment(npadId);

        EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);
    }

    ::testhid::DisableStyleAll();
    ::testhid::DisableIdAll();
}

//!< 残すジョイコンを明示して 1本持ちに変更
TEST(NpadJoyAssignmentTest, SwitchToSingleSpecified)
{
    ::testhid::EnableIdAll();
    ::testhid::EnableStyleAll();

    auto npadId = nn::hid::NpadId::No1;

    auto assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);

    // 左を残してシングルに変更
    nn::hid::SetNpadJoyAssignmentModeSingle(npadId, nn::hid::NpadJoyDeviceType_Left);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    // 上書きしても問題ない
    nn::hid::SetNpadJoyAssignmentModeSingle(npadId, nn::hid::NpadJoyDeviceType_Right);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    // 上書きしても問題ない
    nn::hid::SetNpadJoyAssignmentModeSingle(npadId);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    // 2本持ちにいったん戻す
    nn::hid::SetNpadJoyAssignmentModeDual(npadId);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);

    // 右を残してシングルに変更
    nn::hid::SetNpadJoyAssignmentModeSingle(npadId, nn::hid::NpadJoyDeviceType_Right);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    // 上書きしても問題ない
    nn::hid::SetNpadJoyAssignmentModeSingle(npadId, nn::hid::NpadJoyDeviceType_Left);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    // 上書きしても問題ない
    nn::hid::SetNpadJoyAssignmentModeSingle(npadId);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    // 2本持ちに戻す
    nn::hid::SetNpadJoyAssignmentModeDual(npadId);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);

    ::testhid::DisableStyleAll();
    ::testhid::DisableIdAll();

}

//!< ジョイコンのマージのエラーチェック
TEST(NpadJoyAssignmentTest, MergeSingle)
{
    ::testhid::EnableIdAll();
    ::testhid::EnableStyleAll();

    auto npadId = nn::hid::NpadId::No1;
    auto mergedId = nn::hid::NpadId::No2;

    auto result = ::nn::hid::MergeSingleJoyAsDualJoy(npadId, mergedId);
    EXPECT_TRUE(nn::hid::ResultNpadDualConnected::Includes(result) == true);

    nn::hid::SetNpadJoyAssignmentModeSingle(npadId);
    auto assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    nn::hid::SetNpadJoyAssignmentModeSingle(mergedId);
    assignment = nn::hid::GetNpadJoyAssignment(mergedId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Single, assignment);

    result = ::nn::hid::MergeSingleJoyAsDualJoy(npadId,  mergedId);
    EXPECT_TRUE(nn::hid::ResultNpadDualConnected::Includes(result) == true);

    nn::hid::SetNpadJoyAssignmentModeDual(npadId);
    assignment = nn::hid::GetNpadJoyAssignment(npadId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);

    nn::hid::SetNpadJoyAssignmentModeDual(mergedId);
    assignment = nn::hid::GetNpadJoyAssignment(mergedId);
    EXPECT_EQ(nn::hid::NpadJoyAssignmentMode_Dual, assignment);

    ::testhid::DisableStyleAll();
    ::testhid::DisableIdAll();

}
