﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/debug/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/util/util_BitPack.h>
#include <nnt/nntest.h>

#include "testHid_NpadCommon.h"

namespace {

    template <typename TState>
    void ExpectNoInputState(const TState& state, bool styleAvailable)
    {
        EXPECT_LE(0, state.samplingNumber);
        if (styleAvailable == false)
        {
            EXPECT_FALSE(state.attributes.template Test<
                ::nn::hid::NpadAttribute::IsConnected>());
            EXPECT_FALSE(state.attributes.template Test<
                ::nn::hid::NpadAttribute::IsWired>());
            EXPECT_FALSE(state.attributes.template Test<
                ::nn::hid::NpadJoyAttribute::IsLeftConnected>());
            EXPECT_FALSE(state.attributes.template Test<
                ::nn::hid::NpadJoyAttribute::IsLeftWired>());
            EXPECT_FALSE(state.attributes.template Test<
                ::nn::hid::NpadJoyAttribute::IsRightConnected>());
            EXPECT_FALSE(state.attributes.template Test<
                ::nn::hid::NpadJoyAttribute::IsRightWired>());
        }
        EXPECT_TRUE(state.buttons.IsAllOff());
        EXPECT_EQ(0, state.analogStickR.x);
        EXPECT_EQ(0, state.analogStickR.y);
        EXPECT_EQ(0, state.analogStickL.x);
        EXPECT_EQ(0, state.analogStickL.y);
    }

    template <typename TState>
    void ExpectNoInputState(TState& state, nn::hid::NpadIdType id, bool styleAvailable)
    {
        nn::hid::GetNpadState(&state, id);
        ExpectNoInputState(state, styleAvailable);
    }

    template <typename TState>
    void ExpectNoInputSystemState(TState& state, nn::hid::NpadIdType id, bool styleAvailable)
    {
        nn::hid::system::GetNpadState(&state, id);
        ExpectNoInputState(state, styleAvailable);
    }

    template <typename TState>
    void ExpectNoInputStates(TState* pStates, int length, nn::hid::NpadIdType id, bool styleAvailable)
    {
        auto count = nn::hid::GetNpadStates(pStates, length, id);
        EXPECT_LE(0, count);
        EXPECT_GE(nn::hid::NpadStateCountMax, count);
        ExpectNoInputState(pStates[0], styleAvailable);
    }

    template <typename TState>
    void ExpectNoInputSystemStates(TState* pStates, int length, nn::hid::NpadIdType id, bool styleAvailable)
    {
        auto count = nn::hid::system::GetNpadStates(pStates, length, id);
        EXPECT_LE(0, count);
        EXPECT_GE(nn::hid::NpadStateCountMax, count);
        ExpectNoInputState(pStates[0], styleAvailable);
    }
} // namespace

namespace testhid {

    void EnableStyleAll() NN_NOEXCEPT
    {
        nn::hid::SetSupportedNpadStyleSet(GetSupportedNpadStyleMask());
    };

    void DisableStyleAll() NN_NOEXCEPT
    {
        auto nullStyle = nn::hid::NpadStyleSet();
        nullStyle.Reset();

        nn::hid::SetSupportedNpadStyleSet(nullStyle);
    }

    nn::hid::NpadStyleSet GetSupportedNpadStyleMask() NN_NOEXCEPT
    {
        auto supportedStyleSet = nn::hid::NpadStyleSet();
        for (int i = 0; i < NpadStyleCountMax; i++)
        {
            supportedStyleSet |= NpadStyles[i];
        }

        return supportedStyleSet;
    }


    void EnableIdAll() NN_NOEXCEPT
    {
        nn::hid::InitializeNpad();
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
    };

    void DisableIdAll() NN_NOEXCEPT
    {
        nn::hid::SetSupportedNpadIdType(nullptr, 0);
        nn::hid::debug::FinalizeNpad();
    };

    void ExpectDeathGetNpadState(nn::hid::NpadIdType id, nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        if (style == nn::hid::NpadStyleFullKey::Mask)
        {
            nn::hid::NpadFullKeyState state;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadState(&state, id), "");
        }
        else if (style == nn::hid::NpadStyleHandheld::Mask)
        {
            nn::hid::NpadHandheldState state;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadState(&state, id), "");
        }
        else if (style == nn::hid::NpadStyleJoyDual::Mask)
        {
            nn::hid::NpadJoyDualState state;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadState(&state, id), "");
        }
        else if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            nn::hid::NpadJoyLeftState state;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadState(&state, id), "");
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            nn::hid::NpadJoyRightState state;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadState(&state, id), "");
        }
        else if (style == nn::hid::NpadStyleGc::Mask)
        {
            nn::hid::NpadGcState state;
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadState(&state, id), "");
        }
    };

    void ExpectNoInputGetNpadState(nn::hid::NpadIdType id, nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        auto styleAvailable = (nn::hid::GetNpadStyleSet(id) & style).IsAnyOn();
        if (style == nn::hid::NpadStyleFullKey::Mask)
        {
            nn::hid::NpadFullKeyState state;
            ExpectNoInputState(state, id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleHandheld::Mask)
        {
            nn::hid::NpadHandheldState state;
            ExpectNoInputState(state, id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleJoyDual::Mask)
        {
            nn::hid::NpadJoyDualState state;
            ExpectNoInputState(state, id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            nn::hid::NpadJoyLeftState state;
            ExpectNoInputState(state, id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            nn::hid::NpadJoyRightState state;
            ExpectNoInputState(state, id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleGc::Mask)
        {
            nn::hid::NpadGcState state;
            ExpectNoInputState(state, id, styleAvailable);
        }
        else if (style == nn::hid::system::NpadStyleSystem::Mask)
        {
            nn::hid::system::NpadSystemState state;
            ExpectNoInputSystemState(state, id, styleAvailable);
        }
        else if (style == nn::hid::system::NpadStyleSystemExt::Mask)
        {
            nn::hid::system::NpadSystemExtState state;
            ExpectNoInputSystemState(state, id, styleAvailable);
        }
    };

    void ExpectDeathGetNpadStates(nn::hid::NpadIdType id, nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        if (style == nn::hid::NpadStyleFullKey::Mask)
        {
            nn::hid::NpadFullKeyState states[nn::hid::NpadStateCountMax];
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, id), "");
        }
        else if (style == nn::hid::NpadStyleHandheld::Mask)
        {
            nn::hid::NpadHandheldState states[nn::hid::NpadStateCountMax];
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, id), "");
        }
        else if (style == nn::hid::NpadStyleJoyDual::Mask)
        {
            nn::hid::NpadJoyDualState states[nn::hid::NpadStateCountMax];
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, id), "");
        }
        else if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            nn::hid::NpadJoyLeftState states[nn::hid::NpadStateCountMax];
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, id), "");
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            nn::hid::NpadJoyRightState states[nn::hid::NpadStateCountMax];
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, id), "");
        }
        else if (style == nn::hid::NpadStyleGc::Mask)
        {
            nn::hid::NpadGcState states[nn::hid::NpadStateCountMax];
            EXPECT_DEATH_IF_SUPPORTED(
                nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, id), "");
        }
    };

    void ExpectNoInputGetNpadStates(nn::hid::NpadIdType id, nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        auto styleAvailable = (nn::hid::GetNpadStyleSet(id) & style).IsAnyOn();
        if (style == nn::hid::NpadStyleFullKey::Mask)
        {
            nn::hid::NpadFullKeyState states[nn::hid::NpadStateCountMax];
            ExpectNoInputStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleHandheld::Mask)
        {
            nn::hid::NpadHandheldState states[nn::hid::NpadStateCountMax];
            ExpectNoInputStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleJoyDual::Mask)
        {
            nn::hid::NpadJoyDualState states[nn::hid::NpadStateCountMax];
            ExpectNoInputStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            nn::hid::NpadJoyLeftState states[nn::hid::NpadStateCountMax];
            ExpectNoInputStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            nn::hid::NpadJoyRightState states[nn::hid::NpadStateCountMax];
            ExpectNoInputStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::NpadStyleGc::Mask)
        {
            nn::hid::NpadGcState states[nn::hid::NpadStateCountMax];
            ExpectNoInputStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::system::NpadStyleSystem::Mask)
        {
            nn::hid::system::NpadSystemState states[nn::hid::NpadStateCountMax];
            ExpectNoInputSystemStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
        else if (style == nn::hid::system::NpadStyleSystemExt::Mask)
        {
            nn::hid::system::NpadSystemExtState states[nn::hid::NpadStateCountMax];
            ExpectNoInputSystemStates(states, sizeof(states) / sizeof(states[0]), id, styleAvailable);
        }
    };

    void TestGetNpadState(nn::hid::NpadIdType id, bool expectDeath) NN_NOEXCEPT
    {
        for (int styleIndex = 0; styleIndex < NpadStyleCountMax; styleIndex++)
        {
            if (expectDeath == true)
            {
                ExpectDeathGetNpadState(id, NpadStyles[styleIndex]);
            }
            else
            {
                ExpectNoInputGetNpadState(id, NpadStyles[styleIndex]);
            }
        }
    }

    void TestGetNpadState(bool expectDeath) NN_NOEXCEPT
    {
        for (int idIndex = 0; idIndex < NpadIdCountMax; idIndex++)
        {
            TestGetNpadState(NpadIds[idIndex], expectDeath);
        }
    }

    void TestGetNpadStates(nn::hid::NpadIdType id, bool expectDeath) NN_NOEXCEPT
    {
        for (int styleIndex = 0; styleIndex < NpadStyleCountMax; styleIndex++)
        {
            if(expectDeath == true)
            {
                ExpectDeathGetNpadStates(id, NpadStyles[styleIndex]);
            }
            else
            {
                ExpectNoInputGetNpadStates(id, NpadStyles[styleIndex]);
            }
        }
    }

    void TestGetNpadStates(bool expectDeath) NN_NOEXCEPT
    {
        for (int idIndex = 0; idIndex < NpadIdCountMax; idIndex++)
        {
            TestGetNpadStates(NpadIds[idIndex], expectDeath);
        }
    }

} // namespace testhid
