﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Mouse.h"

namespace
{

::nn::hid::MouseState g_States[::nn::hid::MouseStateCountMax];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority) NN_NOEXCEPT;

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(MouseReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::MouseState& state : g_States)
    {
        state = ::nn::hid::MouseState();
        state.samplingNumber = 0xDEADBEEF;
        state.x = 0x1111;
        state.y = 0x2222;
        state.deltaX = 0x3333;
        state.deltaY = 0x4444;
        state.wheelDelta = 0x5555;
        state.buttons.Set(6, true);
        state.attributes.Set(7, true);
    }

    ::nn::hid::InitializeMouse();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_GE(::nn::hid::MouseStateCountMax, count);

    for (int i = count; i < ::nn::hid::MouseStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
        EXPECT_EQ(0x1111, g_States[i].x);
        EXPECT_EQ(0x2222, g_States[i].y);
        EXPECT_EQ(0x3333, g_States[i].deltaX);
        EXPECT_EQ(0x4444, g_States[i].deltaY);
        EXPECT_EQ(0x5555, g_States[i].wheelDelta);
        auto buttons = ::nn::hid::MouseButtonSet();
        buttons.Set(6, true);
        EXPECT_EQ(buttons, g_States[i].buttons);
        auto attributes = ::nn::hid::MouseAttributeSet();
        attributes.Set(7, true);
        EXPECT_EQ(attributes, g_States[i].attributes);
    }

    ::nn::hid::debug::FinalizeMouse();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(MouseReadingStateSuite, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(MouseReadingStateSuite, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority) NN_NOEXCEPT
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::InitializeMouse();

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::GetMouseStates(g_States,
                                              ::nn::hid::MouseStateCountMax);
        EXPECT_LE(lastCount, count);
        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);
        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::MouseState& lhs = g_States[j];
            const ::nn::hid::MouseState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::MouseStateCountMax, lastCount);

    ::nn::hid::debug::FinalizeMouse();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
