﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/debug/hid_HomeButton.h>
#include <nn/hid/system/hid_HomeButton.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_HomeButton.h"

namespace
{
::nn::hid::system::HomeButtonState g_State;
} // namespace

//!< ホームボタンの初期化状態は GetHomeButtonStates() の呼び出しに反映されるか
TEST(HomeButtonInitializationSuite, NoInitializationTest1)
{
    // ホームボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetHomeButtonStates(&g_State, 1), "");

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンが終了しきっていないので成功する。
    ::nn::hid::system::GetHomeButtonStates(&g_State, 1);

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetHomeButtonStates(&g_State, 1), "");
}

//!< ホームボタンの初期化状態は SetHomeButtonAutoPilotState() の呼び出しに反映されるか
TEST(HomeButtonInitializationSuite, NoInitializationTest2)
{
    ::nn::hid::debug::HomeButtonAutoPilotState autoPilotState = {};

    // ホームボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState), "");

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンが終了しきっていないので成功する。
    ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState);

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetHomeButtonAutoPilotState(autoPilotState), "");
}

//!< ホームボタンの初期化状態は UnsetHomeButtonAutoPilotState() の呼び出しに反映されるか
TEST(HomeButtonInitializationSuite, NoInitializationTest3)
{
    // ホームボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetHomeButtonAutoPilotState(), "");

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンが終了しきっていないので成功する。
    ::nn::hid::debug::UnsetHomeButtonAutoPilotState();

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetHomeButtonAutoPilotState(), "");
}

//!< ホームボタンの初期化処理は最低一つの入力状態の取得を保証するか
TEST(HomeButtonInitializationSuite, StateReadingTest1)
{
    ::nn::hid::system::InitializeHomeButton();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::system::GetHomeButtonStates(&g_State, 1);
    EXPECT_LE(0, g_State.samplingNumber);
    EXPECT_TRUE(g_State.buttons.IsAllOff());

    ::nn::hid::debug::FinalizeHomeButton();
}

//!< ホームボタンの初期化処理はサンプリングを正しく有効化するか
TEST(HomeButtonInitializationSuite, StateReadingTest2)
{
    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::system::GetHomeButtonStates(&g_State, 1);

    int64_t samplingNumber = g_State.samplingNumber;

    ::nn::hid::debug::FinalizeHomeButton();

    // ホームボタンが終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::GetHomeButtonStates(&g_State, 1);

    EXPECT_LT(samplingNumber + ::nn::hid::system::HomeButtonStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeHomeButton();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_State.samplingNumber;

    // ホームボタンが終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::HomeButtonStateCountMax *
            ::nnt::hid::GetHomeButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::InitializeHomeButton();

    ::nn::hid::system::GetHomeButtonStates(&g_State, 1);

    EXPECT_GT(samplingNumber + ::nn::hid::system::HomeButtonStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeHomeButton();
#endif
}
