﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/hid.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/debug/hid_SevenSixAxisSensor.h>

#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_ConsoleSixAxisSensor.h"

namespace
{

const int StateCountMax = ::nn::hid::SevenSixAxisSensorStateCountMax;
::nn::hid::SevenSixAxisSensorState g_States[StateCountMax];

// 最大サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
static uint8_t s_WorkBuffer[::nn::hid::SevenSixAxisSensorWorkBufferSize];

} // namespace


//!< InitializeSevenSixAxisSensor の事前条件テスト
TEST(SevenSixAxisSensorInitializationSuite, DISABLED_PreconditionTest1)
{
    // SevenSixAxisSensor が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetSevenSixAxisSensorStates(g_States, 1), "");

    // バッファへのポインタが nullptr だとアボートする
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::InitializeSevenSixAxisSensor(nullptr,
                                                ::nn::hid::SevenSixAxisSensorWorkBufferSize),
        "");
    // アラインされていないとアボートする
    {
        int dummyArray[10] = {};
        EXPECT_DEATH_IF_SUPPORTED(
            ::nn::hid::InitializeSevenSixAxisSensor(dummyArray,
                                                    ::nn::hid::SevenSixAxisSensorWorkBufferSize),
            "");

    }

    // バッファサイズ指定が異なるとアボートする
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                                ::nn::hid::SevenSixAxisSensorWorkBufferSize - 1),
        "");

    // 初期化処理を行う
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    // 初期化済の場合はアボートする
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                                ::nn::hid::SevenSixAxisSensorWorkBufferSize),
        "");

    ::nn::hid::FinalizeSevenSixAxisSensor();

    // FinalizeSevenSixAxisSensor() した後は InitializeSevenSixAxisSensor() できる
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::FinalizeSevenSixAxisSensor();

    // SevenSixAxisSensor の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::GetSevenSixAxisSensorStates(&g_States[0], 1),
                              "");
}

//!< InitializeSevenSixAxisSensor / FinalizeSevenSixAxisSensor の呼び出しテスト
TEST(SevenSixAxisSensorInitializationSuite, InitializeFinalizeTestTest1)
{
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::FinalizeSevenSixAxisSensor();

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< SevenSixAxisSensor の初期化処理は最低一つの入力状態の取得を保証するか
TEST(SevenSixAxisSensorInitializationSuite, StateReadingTest1)
{
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    // Seven 向け 6 軸センサーの初期化が完了したので成功する。
    int count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
    EXPECT_LE(0, count);
    EXPECT_LE(0, g_States[0].samplingNumber);

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< SevenSixAxisSensor の初期化処理はサンプリングを正しく有効化するか
TEST(SevenSixAxisSensorInitializationSuite, StateReadingTest2)
{
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    // StartSevenSixAxisSensor() 前はサンプリングを開始しない
    ::nn::hid::GetSevenSixAxisSensorStates(&g_States[0], 1);
    int64_t samplingNumber = g_States[0].samplingNumber;

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::GetSevenSixAxisSensorStates(&g_States[0], 1);

    // サンプリングが停止しているためサンプル番号は増加しない。
    EXPECT_EQ(g_States[0].samplingNumber, samplingNumber);

    samplingNumber = g_States[0].samplingNumber;

    // StartSevenSixAxisSensor() 後はサンプリング番号が増加する
    EXPECT_TRUE(::nn::hid::StartSevenSixAxisSensor().IsSuccess());

    ::nnt::hid::SleepThreadOnSamplingStarted();

    ::nn::hid::GetSevenSixAxisSensorStates(&g_States[0], 1);

    // サンプリングが開始されているためサンプル番号が増加する。
    EXPECT_LE(samplingNumber, g_States[0].samplingNumber);

    // SevenSixAxisSensor が終了しているのでサンプリングは停止する。
    ::nn::hid::StopSevenSixAxisSensor();

    // サンプリングの停止は非同期に行われ、ラグがあるので少し待つ。
    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::GetSevenSixAxisSensorStates(&g_States[0], 1);
    samplingNumber = g_States[0].samplingNumber;

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::GetSevenSixAxisSensorStates(&g_States[0], 1);

    // サンプリングが停止しているためサンプル番号は増加しない。
    EXPECT_EQ(samplingNumber, g_States[0].samplingNumber);

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< SevenSixAxisSensor の状態取得数の確認
TEST(SevenSixAxisSensorInitializationSuite, StateReadingTest3)
{
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    EXPECT_TRUE(::nn::hid::StartSevenSixAxisSensor().IsSuccess());

    ::nnt::hid::SleepThreadOnSamplingStarted();

    auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
    EXPECT_EQ(count, StateCountMax);
    EXPECT_EQ(g_States[0].samplingNumber - g_States[StateCountMax - 1].samplingNumber,
              StateCountMax - 1);

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< SevenSixAxisSensor の状態変更を頻繁に行っても問題が無いことの確認
TEST(SevenSixAxisSensorInitializationSuite, StartStopTest1)
{
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    EXPECT_TRUE(::nn::hid::StartSevenSixAxisSensor().IsSuccess());

    ::nnt::hid::SleepThreadOnSamplingStarted();

    auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
    const auto SamplingNumber0 = g_States[0].samplingNumber;

    EXPECT_EQ(count, StateCountMax);
    EXPECT_EQ(SamplingNumber0 - g_States[StateCountMax - 1].samplingNumber,
              StateCountMax - 1);

    // Start/Stop を繰り返す
    for (int i = 0; i < 10; i++)
    {
        ::nn::hid::StopSevenSixAxisSensor();

        EXPECT_TRUE(::nn::hid::StartSevenSixAxisSensor().IsSuccess());
    }

    ::nnt::hid::SleepThreadOnSamplingStarted();

    count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);

    EXPECT_EQ(count, StateCountMax);
    EXPECT_LT(SamplingNumber0, g_States[0].samplingNumber);

    ::nn::hid::StopSevenSixAxisSensor();
    ::nn::hid::FinalizeSevenSixAxisSensor();
}
