﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/hid.h>
#include <nn/hid/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyCommon.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>

#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_ConsoleSixAxisSensor.h"

namespace
{

const int StateCountMax = ::nn::hid::ConsoleSixAxisSensorStateCountMax;
::nn::hid::tmp::SixAxisSensorCountState g_States[StateCountMax];

} // namespace


//!< ConsoleSixAxisSensor の初期化状態は GetConsoleSixAxisSensorState() の呼び出しに反映されるか
TEST(ConsoleSixAxisSensorInitializationSuite, NoInitializationTest1)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    // ConsoleSixAxisSensor が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States,
                                                                                   1,
                                                                                   handle),
                              "");

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();

    // ConsoleSixAxisSensor が終了しきっていないので成功する。
    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States, 1, handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();

    // ConsoleSixAxisSensor の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle),
                              "");
}

//!< ConsoleSixAxisSensor の初期化状態は GetConsoleSixAxisSensorStates() の呼び出しに反映されるか
TEST(ConsoleSixAxisSensorInitializationSuite, NoInitializationTest2)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    // ConsoleSixAxisSensor が未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0],
                                                                                   StateCountMax,
                                                                                   handle),
                              "");

    ::nn::hid::InitializeConsoleSixAxisSensor();
    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();

    // ConsoleSixAxisSensor が終了しきっていないので成功する。
    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], StateCountMax, handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();

    // ConsoleSixAxisSensor の終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0],
                                                                                   StateCountMax,
                                                                                   handle),
                              "");
}

//!< ConsoleSixAxisSensor の初期化処理は最低一つの入力状態の取得を保証するか
TEST(ConsoleSixAxisSensorInitializationSuite, StateReadingTest1)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    // 本体6軸センサーのサンプリングが開始されたので成功する。
    int count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States,
                                                                   StateCountMax,
                                                                   handle);
    EXPECT_LE(1, count);
    EXPECT_LE(0, g_States[0].samplingNumber);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< ConsoleSixAxisSensor の初期化処理はサンプリングを正しく有効化するか
TEST(ConsoleSixAxisSensorInitializationSuite, StateReadingTest2)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    // StartSixAxisSensor() 前はサンプリングを開始しない
    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle);
    int64_t samplingNumber = g_States[0].samplingNumber;

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle);

    // サンプリングが停止しているためサンプル番号は増加しない。
    EXPECT_EQ(g_States[0].samplingNumber, samplingNumber);

    samplingNumber = g_States[0].samplingNumber;

    // StartSixAxisSensor() 後はサンプリング番号が増加する
    ::nn::hid::StartSixAxisSensor(handle);

    ::nnt::hid::SleepThreadOnSamplingStarted();

    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle);

    // サンプリングが開始されているためサンプル番号が増加する。
    EXPECT_LE(samplingNumber, g_States[0].samplingNumber);

    // ConsoleSixAxisSensor が終了しているのでサンプリングは停止する。
    ::nn::hid::StopSixAxisSensor(handle);

    // サンプリングの停止は非同期に行われ、ラグがあるので少し待つ。
    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle);
    samplingNumber = g_States[0].samplingNumber;

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle);

    // サンプリングが停止しているためサンプル番号は増加しない。
    EXPECT_EQ(samplingNumber, g_States[0].samplingNumber);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< ConsoleSixAxisSensor の状態取得数の確認
TEST(ConsoleSixAxisSensorInitializationSuite, DISABLED_StateReadingTest3)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::StartSixAxisSensor(handle);

    ::nnt::hid::SleepThreadOnSamplingStarted();

    auto count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States,
                                                                    StateCountMax,
                                                                    handle);
    EXPECT_EQ(count, StateCountMax);
    EXPECT_EQ(g_States[0].samplingNumber - g_States[StateCountMax - 1].samplingNumber,
              StateCountMax - 1);

    ::nn::hid::StopSixAxisSensor(handle);

    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< ConsoleSixAxisSensor の状態変更を頻繁に行っても問題が無いことの確認
TEST(ConsoleSixAxisSensorInitializationSuite, DISABLED_StartStopTest1)
{
    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(&g_States[0], 1, handle);
    const auto SamplingNumber0 = g_States[0].samplingNumber;

    // Start/Stop を繰り返す
    for (int i = 0; i < 10; i++)
    {
        ::nn::hid::StartSixAxisSensor(handle);

        ::nn::hid::StopSixAxisSensor(handle);
    }

    ::nn::hid::StartSixAxisSensor(handle);

    ::nnt::hid::SleepThreadOnSamplingStarted();

    auto count = ::nn::hid::tmp::GetConsoleSixAxisSensorCountStates(g_States,
                                                                    StateCountMax,
                                                                    handle);
    EXPECT_EQ(count, StateCountMax);
    EXPECT_LT(SamplingNumber0, g_States[0].samplingNumber);

    ::nn::hid::StopSixAxisSensor(handle);
    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}

//!< ConsoleSixAxisSensor の Handheld 操作形態での状態取得ができることの確認
TEST(ConsoleSixAxisSensorInitializationSuite, HandheldConsoleSixAxisSensor)
{
    ::nn::hid::SixAxisSensorHandle handheldHandle;
    int count = ::nn::hid::GetSixAxisSensorHandles(&handheldHandle,
                                                   1,
                                                   ::nn::hid::NpadId::Handheld,
                                                   ::nn::hid::NpadStyleHandheld::Mask);
    EXPECT_EQ(1, count);

    ::nn::hid::ConsoleSixAxisSensorHandle handle;
    ::nn::hid::GetSixAxisSensorHandle(&handle);

    ::nn::hid::InitializeNpad();

    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleHandheld::Mask);
    const nn::hid::NpadIdType NpadId = nn::hid::NpadId::Handheld;

    nn::hid::SetSupportedNpadIdType(&NpadId, 1);

    ::nn::hid::InitializeConsoleSixAxisSensor();

    ::nn::hid::SetNpadHandheldActivationMode(::nn::hid::NpadHandheldActivationMode_None);

    ::nn::hid::StartSixAxisSensor(handheldHandle);
    ::nn::hid::StartSixAxisSensor(handle);

    ::nnt::hid::SleepThreadOnSamplingStarted();

    ::nn::hid::SixAxisSensorState state = {};

    const int ReadTestCount = 5; // 読み出し試行回数
    auto samplingNumber = state.samplingNumber;

    for (int i = 0; i < ReadTestCount; i++)
    {
        ::nn::hid::GetSixAxisSensorState(&state, handheldHandle);
        NN_LOG("[%lld] Acc : (%f, %f, %f), Gyro : (%f, %f, %f)\n",
            state.samplingNumber,
            state.acceleration.x,
            state.acceleration.y,
            state.acceleration.z,
            state.angularVelocity.x,
            state.angularVelocity.y,
            state.angularVelocity.z
        );
        EXPECT_LE(samplingNumber, state.samplingNumber);

        // 1 周期待つ
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetConsoleSixAxisSensorSamplingInterval().GetMilliSeconds()
            )
        );
    }

    ::nn::hid::StopSixAxisSensor(handle);
    ::nn::hid::StopSixAxisSensor(handheldHandle);
    ::nn::hid::debug::FinalizeConsoleSixAxisSensor();
}
