﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/vi.h>
#include <nn/applet/applet_Apis.h>

#include "testGrc_Macro.h"
#include "testGrc_MemoryManagement.h"
#include "testGrc_NvnUtility.h"
#include "testGrc_SceneUtility.h"
#include "testGrc_AudioUtil.h"

#include <nn/album/album_MovieMaker.h>
#include <nn/album/album_MovieMaker.internal.h>

#include <nn/grc/grc_Api.h>

#define PRESENT_TO_RECORDER
//#define EXIT_BY_ABORT


NNT_GRC_TEST_SERVERSTATE(MovieMaker30fps)
{
    static const size_t ControlMemorySize     = 4 * 4096;
    static const size_t CommandPoolMemorySize = 10 * 4096;
    static const size_t FramePoolMemorySize = 8 * 1024 * 1024;

    static const int FileCount = 1;
    static const int FrameCount = 150;

    // AM の代わりに自分で使う OffscreenLayer を作っておく
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
    uint64_t hOffscrn;
    NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&hOffscrn, nn::applet::GetAppletResourceUserId()));
    NN_UTIL_SCOPE_EXIT{ NNT_GRC_EXPECT_SUCCESS(pService->DestroyOffscreenLayer(hOffscrn)); };
    // MovieMaker を作成
    nn::sf::SharedPointer<nn::grcsrv::IMovieMaker> pMovieMaker;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateMovieMakerForApplication(&pMovieMaker, nn::applet::GetAppletResourceUserId(), {0x010000000000B1C1ull}));

    InitializeGraphics();

    // 通常のレイヤを作成
    nn::vi::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::vi::Finalize(); };

    nn::vi::Display* pDisplay;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));
    NN_UTIL_SCOPE_EXIT{ nn::vi::CloseDisplay(pDisplay); };

    nn::vi::Layer* pDisplayLayer;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::CreateLayer(&pDisplayLayer, pDisplay));

    nn::vi::NativeWindowHandle hDisplayNativeWindow;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetNativeWindow(&hDisplayNativeWindow, pDisplayLayer));

    EXPECT_FALSE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    // 録画用レイヤを作成
    //nn::album::Initialize();
    //NN_UTIL_SCOPE_EXIT{ nn::album::Finalize(); };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::InitializeMovieMakerInternal(std::move(pMovieMaker), hOffscrn));
    NN_UTIL_SCOPE_EXIT{ nn::album::FinalizeMovieMaker(); };

    EXPECT_TRUE(nn::album::IsMovieMakerInitialized());
    EXPECT_FALSE(nn::album::IsMovieMakerRunning());

    nn::vi::NativeWindowHandle hVideoNativeWindow;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::GetMovieMakerNativeWindow(&hVideoNativeWindow));

    auto pControlMemory     = aligned_alloc(4096, ControlMemorySize);
    auto pCommandPoolMemory = aligned_alloc(4096, CommandPoolMemorySize);
    auto pFramePoolMemory   = aligned_alloc(4096, FramePoolMemorySize);
    NN_ABORT_UNLESS_NOT_NULL(pControlMemory);
    NN_ABORT_UNLESS_NOT_NULL(pCommandPoolMemory);
    NN_ABORT_UNLESS_NOT_NULL(pFramePoolMemory);
    NN_UTIL_SCOPE_EXIT{
        free(pControlMemory);
        free(pCommandPoolMemory);
        free(pFramePoolMemory);
    };

    // init nvn
    NNT_GRC_SCOPED_DEVICE           (device);
    NNT_GRC_SCOPED_COMMANDMEMORYPOOL(commandPool, device, pCommandPoolMemory, CommandPoolMemorySize);
    NNT_GRC_SCOPED_QUEUE            (queue, device);
    NNT_GRC_SCOPED_COMMANDBUFFER    (commandBuffer, device);
    NNT_GRC_SCOPED_TEXTUREMEMORYPOOL(framePool, device, pFramePoolMemory, FramePoolMemorySize);
    NNT_GRC_SCOPED_FRAMETEXTURE     (frameTexs, 2, 1280, 720, device, framePool, 0, FramePoolMemorySize);
    NNT_GRC_SCOPED_WINDOW           (displayWindow, frameTexs, 2, hDisplayNativeWindow, device);
    NNT_GRC_SCOPED_WINDOW           (videoWindow, frameTexs, 2, hVideoNativeWindow, device);
    NNT_GRC_SCOPED_FRAMETEXTUREVIEW (texView);

#ifdef PRESENT_TO_RECORDER
    auto& window = videoWindow;
    auto& windowTextureList = videoWindowTextureList;
#else
    auto& window = displayWindow;
    auto& windowTextureList = displayWindowTextureList;
#endif

    static const int AudioSampleRate = 48000;
    static const int AudioChannelCount = 2;
    static const int AudioBufferLength = AudioChannelCount * AudioSampleRate / 30;
    nnt::grc::AudioSinWave sinWave(1000, 432, AudioChannelCount, AudioSampleRate);
    std::int16_t audioBuffer[AudioBufferLength] = {};

    auto frameFunction = [&](int fileNo, int frame) -> nn::Result
    {
        NNT_GRC_SCOPED_SYNC(acqSync, device);
        int texIdx = -1;
        NN_ABORT_UNLESS_EQUAL(nvnWindowAcquireTexture(&window, &acqSync, &texIdx), NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);
        NN_LOG("%d-%d (tex=%d)\n", fileNo, frame, texIdx);
        auto pTargetTexture = windowTextureList[texIdx];

        nvnCommandBufferAddCommandMemory(&commandBuffer, &commandPool, 0, sizeof(CommandPoolMemorySize));
        nvnCommandBufferAddControlMemory(&commandBuffer, pControlMemory, ControlMemorySize);
        nvnCommandBufferBeginRecording(&commandBuffer);

        // wait display
        nvnCommandBufferWaitSync(&commandBuffer, &acqSync);

        nnt::grc::scene::DrawRotationBoxScene(&commandBuffer, pTargetTexture, &texView, frame / 60.f);

        // フレーム数を描画
        {
            char buf[256];
            nn::util::SNPrintf(buf, sizeof(buf), "F%03d", frame + 1);
            nnt::grc::scene::DrawStringParameter param = {};
            param.color = nn::util::Color4f(1, 1, 1);
            param.size  = 256;
            param.posX  = 0;
            param.posY  = 0;
            nnt::grc::scene::DrawString(&commandBuffer, pTargetTexture, &texView, buf, param);
        }

        auto hCommand = nvnCommandBufferEndRecording(&commandBuffer);
        nvnQueueSubmitCommands(&queue, 1, &hCommand);
        nvnQueuePresentTexture(&queue, &window, texIdx);
        nvnQueueFinish(&queue);

        // オーディオ
        sinWave.Generate(audioBuffer, AudioBufferLength);
        NNT_EXPECT_RESULT_SUCCESS(nn::album::EncodeMovieMakerAudioSample(audioBuffer, sizeof(audioBuffer)));

#ifdef PRESENT_TO_RECORDER
        NN_RESULT_DO(nn::album::CheckMovieMakerError());
#endif
        NN_RESULT_SUCCESS;
    };

    auto fileFunction = [&](int fileNo) -> nn::Result
    {
        NN_LOG("%d:StartMovieMaker\n", fileNo);
        nn::album::MovieMakerMovieParameter param;
        param.SetVideoFrameRate(30);
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::album::StartMovieMaker(param));
        EXPECT_TRUE(nn::album::IsMovieMakerInitialized());
        EXPECT_TRUE(nn::album::IsMovieMakerRunning());
#ifdef EXIT_BY_ABORT
        NN_UTIL_SCOPE_EXIT{
            NN_LOG("%d:AbortMovieMaker\n", fileNo);
            nn::album::AbortMovieMaker();
        };
#else
        NN_UTIL_SCOPE_EXIT{
            NN_LOG("%d:FinishMovieMaker\n", fileNo);
            NNT_EXPECT_RESULT_SUCCESS(nn::album::FinishMovieMaker());
            EXPECT_TRUE(nn::album::IsMovieMakerInitialized());
            EXPECT_FALSE(nn::album::IsMovieMakerRunning());
        };
#endif

        NN_ABORT_UNLESS(nn::album::IsMovieMakerRunning());

        for(int frame = 0; frame < FrameCount; frame++)
        {
            auto frameResult = frameFunction(fileNo, frame);
            if(frameResult.IsFailure())
            {
                NN_LOG("%d-%d: error detected(%d-%d)\n", fileNo, frame, frameResult.GetModule(), frameResult.GetDescription());
                break;
            }
        }

        NN_RESULT_SUCCESS;
    };

    for(int fileNo = 0; fileNo < FileCount; fileNo++)
    {
        fileFunction(fileNo);
    }

}// NOLINT(impl/function_size)

