﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testGrc_MemoryManagement.h"

//#include "Config.h"

#ifdef NN_BUILD_TARGET_PLATFORM_OS_NN

#include <nn/mem/mem_StandardAllocator.h>
#include <new>
#include <cstdlib>
#include <cerrno>
#include <type_traits>
#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/os/os_TransferMemoryApi.h>

#include <nv/nv_MemoryManagement.h>
#include <nv/nv_ServiceName.h>

namespace
{
    static const size_t HeapSize = 256 * 1024 * 1024;
    static const size_t DonateSize = 8 * 1024 * 1024;

    std::aligned_storage<sizeof(nn::mem::StandardAllocator), NN_ALIGNOF(nn::mem::StandardAllocator)>::type g_AllocatorStorage;
    nn::mem::StandardAllocator* g_pAllocator;

    NN_ALIGNAS(4096) char g_DonateMemory[DonateSize];

    size_t g_TotalAllocatedSizeMax;

    void UpdateTotalAllocatedSizeMax() NN_NOEXCEPT
    {
        size_t s = GetTotalAllocatedSize();
        g_TotalAllocatedSizeMax = std::max(g_TotalAllocatedSizeMax, s);
    }
}


void InitializeAllocator() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(HeapSize));
    uintptr_t address = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, HeapSize));
    g_pAllocator = new(&g_AllocatorStorage) nn::mem::StandardAllocator();
    g_pAllocator->Initialize(reinterpret_cast<void*>(address), HeapSize);
    g_TotalAllocatedSizeMax = 0;
}


size_t GetApplicationHeapSize() NN_NOEXCEPT
{
    return HeapSize;
}

size_t GetTotalAllocatedSize() NN_NOEXCEPT
{
    return HeapSize - g_pAllocator->GetTotalFreeSize();
}

size_t GetTotalAllocatedSizeMax() NN_NOEXCEPT
{
    return g_TotalAllocatedSizeMax;
}

extern "C"
{
    void* malloc(size_t size)
    {
        void* p = g_pAllocator->Allocate(size);
        if ( p == NULL )
        {
            errno = ENOMEM;
        }

        UpdateTotalAllocatedSizeMax();
        return p;
    }

    void free(void* p)
    {
        if (p)
        {
            g_pAllocator->Free(p);
        }
    }

    void* calloc(size_t num, size_t size)
    {
        size_t sum = num * size;
        void*  p   = malloc(sum);

        if ( p != NULL )
        {
            memset(p, 0, sum);
        }
        else
        {
            errno = ENOMEM;
        }

        return p;
    }

    void* realloc(void* p, size_t newSize)
    {
        void* r = g_pAllocator->Reallocate(p, newSize);
        if ( r == NULL )
        {
            errno = ENOMEM;
        }

        UpdateTotalAllocatedSizeMax();
        return r;
    }

    void* aligned_alloc(size_t alignment, size_t size)
    {
        void* p = g_pAllocator->Allocate(size, alignment);
        if ( p == NULL )
        {
            errno = ENOMEM;
        }

        UpdateTotalAllocatedSizeMax();
        return p;
    }

    size_t malloc_usable_size(void* p)
    {
        if ( p == NULL )
        {
            errno = ENOMEM;
            return 0;
        }

        return g_pAllocator->GetSizeOf(p);
    }
}


#else

void InitializeAllocator() NN_NOEXCEPT
{
}
size_t GetApplicationHeapSize() NN_NOEXCEPT
{
    return 0;
}

size_t GetTotalAllocatedSize() NN_NOEXCEPT
{
    return 0;
}

size_t GetTotalAllocatedSizeMax() NN_NOEXCEPT
{
    return 0;
}

#endif

namespace {
    void* GraphicsAllocate(size_t size, size_t alignment, void*) NN_NOEXCEPT
    {
        return aligned_alloc(alignment, size);
    }

    void GraphicsFree(void* ptr, void*) NN_NOEXCEPT
    {
        free(ptr);
    }

    void* GraphicsRealloc(void* ptr, size_t size, void*) NN_NOEXCEPT
    {
        return realloc(ptr, size);
    }
}

void InitializeGraphics() NN_NOEXCEPT
{
    static int s_InitializeCount = 0;
    if(s_InitializeCount > 0)
    {
        return;
    }

    NN_LOG("InitializeGraphics\n");
    //nv::SetGraphicsServiceName("nvdrv:t");
    //nv::SetGraphicsServiceName("nvdrv:a");
    nv::SetGraphicsAllocator(GraphicsAllocate, GraphicsFree, GraphicsRealloc, nullptr);
    nv::InitializeGraphics(g_DonateMemory, sizeof(g_DonateMemory));

    s_InitializeCount++;
}

namespace {
    NN_ALIGNAS(4096) char g_RecorderMemory[96 * 1024 * 1024];
}

void* GetRecorderFirmwareMemory() NN_NOEXCEPT
{
    return g_RecorderMemory;
}

size_t GetRecorderFirmwareMemorySize() NN_NOEXCEPT
{
    return sizeof(g_RecorderMemory);
}

nn::sf::NativeHandle MakeTransferMemory(void* memory, size_t size) NN_NOEXCEPT
{
    nn::os::TransferMemoryType transfer;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateTransferMemory(&transfer, memory, size, nn::os::MemoryPermission_None));
    nn::os::NativeHandle h = nn::os::DetachTransferMemory(&transfer);
    nn::os::DestroyTransferMemory(&transfer);
    return nn::sf::NativeHandle(h, true);
}
